/*
 * Decompiled with CFR 0.152.
 */
package com.atlassian.oai.validator.interaction.request;

import com.atlassian.oai.validator.model.ApiOperation;
import com.atlassian.oai.validator.model.Request;
import com.atlassian.oai.validator.report.MessageResolver;
import com.atlassian.oai.validator.report.ValidationReport;
import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.oas.models.security.SecurityRequirement;
import io.swagger.v3.oas.models.security.SecurityScheme;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;
import javax.annotation.Nonnull;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SecurityValidator {
    private static final Logger log = LoggerFactory.getLogger(SecurityValidator.class);
    private static final String MISSING_SECURITY_PARAMETER_KEY = "validation.request.security.missing";
    private static final String INVALID_SECURITY_PARAMETER_KEY = "validation.request.security.invalid";
    private final MessageResolver messages;
    private final OpenAPI api;

    SecurityValidator(MessageResolver messages, OpenAPI api) {
        this.messages = messages;
        this.api = api;
    }

    @Nonnull
    public ValidationReport validateSecurity(Request request, ApiOperation apiOperation) {
        List securityRequired = apiOperation.getOperation().getSecurity();
        if (securityRequired == null || securityRequired.isEmpty()) {
            return ValidationReport.empty();
        }
        if (this.api.getComponents() == null || this.api.getComponents().getSecuritySchemes() == null) {
            log.warn("Operation '{} {}' defines a 'security' block but no 'securitySchemes' are defined", (Object)apiOperation.getMethod().name(), (Object)apiOperation.getApiPath().normalised());
            return ValidationReport.empty();
        }
        List<ValidationReport> reports = securityRequired.stream().map(requirement -> this.validateSecurityRequirement(request, (SecurityRequirement)requirement)).collect(Collectors.toList());
        if (this.atLeastOneRequirementFulfilled(reports)) {
            return ValidationReport.empty();
        }
        if (this.allSecurityRequirementsMissing(reports)) {
            return this.missingSecurityParameter(request);
        }
        return this.findMostFulfilledRequirement(reports).orElse(this.combineAllReports(reports));
    }

    @Nonnull
    private ValidationReport validateSecurityRequirement(Request request, SecurityRequirement requirement) {
        return requirement.keySet().stream().map(schemeName -> {
            SecurityScheme scheme = (SecurityScheme)this.api.getComponents().getSecuritySchemes().get(schemeName);
            if (scheme == null) {
                log.warn("Security scheme definition not found for {}", schemeName);
            }
            return scheme;
        }).filter(Objects::nonNull).map(scheme -> this.validateSecurityScheme(request, (SecurityScheme)scheme)).reduce(ValidationReport.empty(), ValidationReport::merge);
    }

    @Nonnull
    private ValidationReport validateSecurityScheme(Request request, SecurityScheme securityScheme) {
        switch (securityScheme.getType()) {
            case APIKEY: {
                switch (securityScheme.getIn()) {
                    case HEADER: {
                        return this.validateApiKeyAuthByHeader(request, securityScheme);
                    }
                    case QUERY: {
                        return this.validateApiKeyAuthByQueryParameter(request, securityScheme);
                    }
                    case COOKIE: {
                        return this.validateApiKeyAuthByCookie(request, securityScheme);
                    }
                }
                return ValidationReport.empty();
            }
            case HTTP: {
                return this.validateHttpAuthorization(request, securityScheme);
            }
        }
        log.info("Security scheme '{}' not currently supported", (Object)securityScheme.getType());
        return ValidationReport.empty();
    }

    @Nonnull
    private ValidationReport validateHttpAuthorization(Request request, SecurityScheme securityScheme) {
        if ("BASIC".equalsIgnoreCase(securityScheme.getScheme())) {
            return this.validateBasicAuthHeader(request);
        }
        if ("BEARER".equalsIgnoreCase(securityScheme.getScheme())) {
            return this.validateBearerAuthHeader(request);
        }
        return ValidationReport.empty();
    }

    private ValidationReport validateBasicAuthHeader(Request request) {
        return request.getHeaderValue("Authorization").map(header -> header.startsWith("Basic ") ? ValidationReport.empty() : this.invalidSecurityParameter(request)).orElse(this.missingSecurityParameter(request));
    }

    private ValidationReport validateBearerAuthHeader(Request request) {
        return request.getHeaderValue("Authorization").map(header -> header.startsWith("Bearer ") ? ValidationReport.empty() : this.invalidSecurityParameter(request)).orElse(this.missingSecurityParameter(request));
    }

    @Nonnull
    private ValidationReport validateApiKeyAuthByQueryParameter(Request request, SecurityScheme securityScheme) {
        Optional<String> authQueryParam = request.getQueryParameterValues(securityScheme.getName()).stream().findFirst();
        if (!authQueryParam.isPresent()) {
            return this.missingSecurityParameter(request);
        }
        return ValidationReport.empty();
    }

    @Nonnull
    private ValidationReport validateApiKeyAuthByHeader(Request request, SecurityScheme securityScheme) {
        Optional<String> headerValue = request.getHeaderValue(securityScheme.getName());
        if (!headerValue.isPresent() || headerValue.get().isEmpty()) {
            return this.missingSecurityParameter(request);
        }
        return ValidationReport.empty();
    }

    @Nonnull
    private ValidationReport validateApiKeyAuthByCookie(Request request, SecurityScheme securityScheme) {
        Optional<String> cookieHeader = request.getHeaderValue("Cookie");
        boolean cookieExists = cookieHeader.map(c -> c.split("; ")).map(cookies -> Arrays.stream(cookies).filter(cookie -> cookie.toLowerCase().startsWith(securityScheme.getName().toLowerCase())).anyMatch(cookie -> !StringUtils.isBlank((CharSequence)StringUtils.substringAfter((String)cookie, (String)"=")))).orElse(false);
        if (!cookieExists) {
            return this.missingSecurityParameter(request);
        }
        return ValidationReport.empty();
    }

    @Nonnull
    private ValidationReport missingSecurityParameter(Request request) {
        return ValidationReport.singleton(this.messages.get(MISSING_SECURITY_PARAMETER_KEY, new Object[]{request.getMethod(), request.getPath()}));
    }

    @Nonnull
    private ValidationReport invalidSecurityParameter(Request request) {
        return ValidationReport.singleton(this.messages.get(INVALID_SECURITY_PARAMETER_KEY, new Object[]{request.getMethod(), request.getPath()}));
    }

    @Nonnull
    private Optional<ValidationReport> findMostFulfilledRequirement(List<ValidationReport> reports) {
        return reports.stream().filter(r -> r.getMessages().stream().noneMatch(m -> MISSING_SECURITY_PARAMETER_KEY.equals(m.getKey()))).findFirst();
    }

    private boolean atLeastOneRequirementFulfilled(List<ValidationReport> reports) {
        return reports.stream().anyMatch(r -> !r.hasErrors());
    }

    private boolean allSecurityRequirementsMissing(List<ValidationReport> reports) {
        return reports.stream().allMatch(r -> r.getMessages().stream().anyMatch(m -> MISSING_SECURITY_PARAMETER_KEY.equals(m.getKey())));
    }

    @Nonnull
    private ValidationReport combineAllReports(List<ValidationReport> reports) {
        return reports.stream().reduce(ValidationReport.empty(), ValidationReport::merge);
    }
}

