package com.kms.katalon.core.logging.model;

import java.io.File;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.logging.model.TestStatus.TestStatusValue;
import com.kms.katalon.core.util.TestCloudPropertyUtil;
import com.kms.katalon.selenium.constant.SeleniumW3CCapabilityConstant;

public class TestSuiteLogRecord extends AbstractLogRecord {

    private static final String TEST_CASE_REMAINING_RETRY_COUNT = "remainingRetryCount";
    
    private static final String TOTAL_TEST_CASES = "total_test_cases";
    
    private static final String PASSED_TEST_CASES = "passed_test_cases";
    
    private static final String FAILED_TEST_CASES = "failed_test_cases";
    
    private static final String ERROR_TEST_CASES = "error_test_cases";
    
    private static final String INCOMPLETE_TEST_CASES = "incomplete_test_cases";
    
    private static final String SKIPPED_TEST_CASES = "skipped_test_cases";
    
    private static final EnumSet<TestStatusValue> ERROR_STATUSES = EnumSet.of(
            TestStatusValue.ERROR,
            TestStatusValue.FAILED,
            TestStatusValue.INCOMPLETE
        );

    private String devicePlatform;

    private String logFolder;

    private Map<String, String> runData;

    private String testSuiteCollectionId;

    private String testSuiteCollectionName;

    private String testSuiteCollectionPath;

    private Map<String, String> buildData;

    private Map<String, String> desiredCaps;

    private boolean isCompleted;

    private List<ILogRecord> testSuiteLogRecords;

    private List<ILogRecord> beforeTestSuiteLogRecords;

    private List<ILogRecord> afterTestSuiteLogRecords;
    
    private String executionProfile;

    public String getTestSuiteCollectionId() {
        return testSuiteCollectionId;
    }

    public void setTestSuiteCollectionId(String testSuiteCollectionId) {
        this.testSuiteCollectionId = testSuiteCollectionId;
    }

    public String getTestSuiteCollectionName() {
        return testSuiteCollectionName;
    }

    public void setTestSuiteCollectionName(String testSuiteCollectionName) {
        this.testSuiteCollectionName = testSuiteCollectionName;
    }

    public String getTestSuiteCollectionPath() {
        return testSuiteCollectionPath;
    }

    public void setTestSuiteCollectionPath(String testSuiteCollectionPath) {
        this.testSuiteCollectionPath = testSuiteCollectionPath;
    }

    public TestSuiteLogRecord(String name, String logFolder) {
        super(name);
        this.logFolder = logFolder;
        runData = new HashMap<String, String>();
        buildData = new HashMap<String, String>();
        setType(ILogRecord.LOG_TYPE_TEST_SUITE);
        this.testSuiteLogRecords = new ArrayList<>();
    }

    public void setStatus(boolean isCompleted) {
        this.isCompleted = isCompleted;
    }

    public boolean isCompleted() {
        return isCompleted;
    }

    public String getBrowser() {
        String browser = StringUtils.EMPTY;
        String browserName = StringUtils.EMPTY;
        if (getRunData().containsKey("browser")) {
            browserName = getRunData().get("browser");
        }

        if (StringUtils.isBlank(browserName)) {
            browserName = getValueFromDesiredCaps(SeleniumW3CCapabilityConstant.BROWSER_NAME_CAP);
            String browserVersion = getValueFromDesiredCaps(SeleniumW3CCapabilityConstant.BROWSER_VERSION_CAP);
            if (StringUtils.isNoneBlank(browserVersion)) {
                browserName = browserName + " " + browserVersion;
            }
        }

        if (StringUtils.isNoneBlank(browserName)) {
            browserName = StringUtils.capitalize(browserName);
            String devicePlatform = getDevicePlatform();
            if (StringUtils.isNoneBlank(devicePlatform)) {
                boolean isMobile = ((StringUtils.equals(devicePlatform.toLowerCase(), "android")
                        || StringUtils.equals(devicePlatform.toLowerCase(), "ios")));
                if (!isMobile) {
                    browserName = formatDevicePlatform(devicePlatform) + " - " + browserName;

                    String driverType = getRemoteWebDriverType();
                    if (StringUtils.equals(driverType, "TESTCLOUD_DRIVER")) {
                        String browserFullVersion = TestCloudPropertyUtil.getInstance()
                                .buildBrowserFullVersion(browserName, getRunData().get("browserVersionType"));

                        browserName = "TestCloud - " + browserFullVersion;
                    }
                }
            }

            browser = browserName;
        }

        return browser;
    }

    /**
     * @return the remote web driver type if running Mobile keyword on cloud services
     */
    public String getRemoteWebDriverType() {
        RunConfiguration.setExecutionSettingFile(this.getLogFolder() + "/execution.properties");
        return RunConfiguration.getDriverSystemProperty(RunConfiguration.REMOTE_DRIVER_PROPERTY, "browserType");
    }

    public String getLogFolder() {
        return logFolder;
    }

    public int getTotalTestCases() {
        return getTotalTestCasesWithTestStatusValue(null);
    }

    public int getTotalPassedTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.PASSED);
    }

    public int getTotalFailedTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.FAILED);
    }

    public int getTotalErrorTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.ERROR);
    }

    public int getTotalIncompleteTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.INCOMPLETE);
    }

    public int getTotalSkippedTestCases() {
        return getTotalTestCasesWithTestStatusValue(TestStatusValue.SKIPPED);
    }

    public TestStatusValue getSummaryStatus() {
        if (getTotalIncompleteTestCases() > 0) {
            return TestStatusValue.INCOMPLETE;
        }

        if (getTotalErrorTestCases() > 0) {
            return TestStatusValue.ERROR;
        }

        if (getTotalFailedTestCases() > 0) {
            return TestStatusValue.FAILED;
        }

        if (getTotalSkippedTestCases() == getTotalTestCases()) {
            return TestStatusValue.SKIPPED;
        }

        return TestStatusValue.PASSED;
    }

    public ILogRecord[] getFinalTestCases() {
        return filterFinalTestCasesResult(true);
    }

    public ILogRecord[] filterFinalTestCasesResult() {
        return filterFinalTestCasesResult(false);
    }

    public ILogRecord[] filterFinalTestCasesResult(boolean testCaseOnly) {
        ILogRecord[] childLogRecords = getChildRecords();
        List<ILogRecord> filterChildLogRecords = new ArrayList<>();
        for (int i = 0; i < childLogRecords.length; i++) {
            ILogRecord childLogRecord = childLogRecords[i];
            if (testCaseOnly && !(childLogRecord instanceof TestCaseLogRecord)) {
                continue;
            }
            if (isFailedOrErrorTestCase(childLogRecord)) {
                TestCaseLogRecord testCaseLog = (TestCaseLogRecord) childLogRecord;
                Map<String, String> testCaseProperties = testCaseLog.getProperties();
                if (testCaseProperties != null && testCaseProperties.containsKey(TEST_CASE_REMAINING_RETRY_COUNT)) {
                    int remainingRetryCount = Integer.parseInt(testCaseProperties.get(TEST_CASE_REMAINING_RETRY_COUNT));
                    if (remainingRetryCount > 0) {
                        if (i == childLogRecords.length - 1) {
                            filterChildLogRecords.add(childLogRecord);
                        }
                        continue;
                    }
                }
            }
            filterChildLogRecords.add(childLogRecord);
        }
        return filterChildLogRecords.toArray(new ILogRecord[filterChildLogRecords.size()]);
    }

    private boolean isFailedOrErrorTestCase(ILogRecord logRecord) {
        if (!(logRecord instanceof TestCaseLogRecord)) {
            return false;
        }
        switch (logRecord.getStatus().getStatusValue()) {
            case FAILED:
            case ERROR:
                return true;
            default:
                return false;
        }
    }

    private int getTotalTestCasesWithTestStatusValue(TestStatusValue testStatusValue) {
        ILogRecord[] childLogRecords = filterFinalTestCasesResult();
        int total = 0;
        for (ILogRecord childLogRecord : childLogRecords) {
            if (childLogRecord instanceof TestCaseLogRecord) {
                TestCaseLogRecord testCaseLog = (TestCaseLogRecord) childLogRecord;
                if (testStatusValue == null || testCaseLog.getStatus().statusValue == testStatusValue) {
                    total++;
                }
            }
        }
        return total;
    }

    public String getDeviceName() {
        String deviceName = null;
        if (getRunData().containsKey(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY)) {
            deviceName = getRunData().get(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY);
        }
        if (StringUtils.isBlank(deviceName)) {
            deviceName = getValueFromDesiredCaps("deviceName");
        }
        if (StringUtils.isBlank(deviceName)) {
            deviceName = getValueFromDesiredCaps("appium:deviceName");
        }
        deviceName = enrichDeviceNameWhenPrivateDevice(deviceName);
        return deviceName;
    }

    private String enrichDeviceNameWhenPrivateDevice(String deviceName) {
        String udid = getValueFromDesiredCaps("appium:udid");
        if (StringUtils.isBlank(udid)) {
            return deviceName;
        }
        return deviceName + " (UDID: " + udid.substring(0, 6) + ")";
    }

    // Device name:
    // TestCloud Mobile: TestCloud - OS N - <Name of the mobile device> with N is OS version.
    // eg: TestCloud - iOS 17 - Apple iPhone 12 Pro Max
    // Other mobile: OS - OS N - <Name of the mobile device>
    // eg: iOS 17 - Apple iPhone 12 Pro Max
    // If NOT mobile: device name is empty
    public String getMobileDeviceName() {
        String devicePlatform = getDevicePlatform();
        String mobileDeviceName = StringUtils.EMPTY;
        boolean isMobile = StringUtils.isNoneBlank(devicePlatform)
                && (StringUtils.equals(devicePlatform.toLowerCase(), "android")
                        || StringUtils.equals(devicePlatform.toLowerCase(), "ios"));
        if (isMobile) {
            String driverType = getRemoteWebDriverType();
            String deviceVersion = getDeviceVersion();
            String deviceName = getDeviceName();
            if (StringUtils.isNotBlank(deviceName)) {
                devicePlatform = formatDevicePlatform(devicePlatform);
                if (StringUtils.equals(driverType, "TESTCLOUD_DRIVER")) {
                    mobileDeviceName = MessageFormat.format("TestCloud - {0} {1} - {2}", devicePlatform, deviceVersion,
                            deviceName);
                } else {
                    mobileDeviceName = MessageFormat.format("{0} {1} - {2}", devicePlatform, deviceVersion, deviceName);
                }
            }
        }

        return mobileDeviceName;
    }

    public String getDeviceId() {
        return (getRunData().containsKey(StringConstants.XML_LOG_DEVICE_ID_PROPERTY))
                ? getRunData().get(StringConstants.XML_LOG_DEVICE_ID_PROPERTY) : "";
    }

    public String getDevicePlatform() {
        if (StringUtils.isBlank(devicePlatform)) {
            String platform = StringUtils.EMPTY;
            if (getRunData().containsKey(StringConstants.XML_LOG_DEVICE_OS_PROPERTY)) {
                platform = getRunData().get(StringConstants.XML_LOG_DEVICE_OS_PROPERTY);
            } else {
                platform = getValueFromDesiredCaps("platform");
                if (StringUtils.isBlank(platform)) {
                    platform = getValueFromDesiredCaps("platformName");
                    if (StringUtils.isBlank(platform)) {
                        platform = getRunData().get(StringConstants.XML_LOG_DEVICE_OS_PROPERTY);
                    }
                }
            }

            devicePlatform = platform;
        }

        return devicePlatform;
    }

    public String getDeviceVersion() {
        if (getRunData().containsKey(StringConstants.XML_LOG_DEVICE_OS_VERSION_PROPERTY)) {
            return getRunData().get(StringConstants.XML_LOG_DEVICE_OS_VERSION_PROPERTY);
        }

        // Gets device os version from desired capabilities
        String platformVersion = getValueFromDesiredCaps("platformVersion");
        if (StringUtils.isBlank(platformVersion)) {
            platformVersion = getValueFromDesiredCaps("appium:platformVersion");
            if (StringUtils.isBlank(platformVersion)) {
                if (getRunData().containsKey("desiredCapabilities")) {
                    String json = getRunData().get("desiredCapabilities");
                    JsonObject jsonObject = JsonParser.parseString(json).getAsJsonObject();
                    try {
                        String deviceVersion = jsonObject
                                .getAsJsonObject(TestCloudPropertyUtil.TEST_CLOUD_KATALON_OPTIONS)
                                .get("deviceVersion")
                                .getAsString();
                        if (StringUtils.isNoneBlank(deviceVersion)) {
                            platformVersion = deviceVersion;
                        }
                    } catch (NullPointerException e) {
                        return StringUtils.EMPTY;
                    }
                }
            }
        }

        return platformVersion;
    }

    @SuppressWarnings("unchecked")
    private String getValueFromDesiredCaps(String key) {
        if (desiredCaps != null && !desiredCaps.isEmpty()) {
            return desiredCaps.get(key);
        }
        if (getRunData().containsKey("desiredCapabilities")) {
            String json = getRunData().get("desiredCapabilities");
            try {
                desiredCaps = new ObjectMapper().readValue(json, HashMap.class);
                return desiredCaps.get(key);
            } catch (JsonProcessingException e) {
                return StringUtils.EMPTY;
            }
        }
        return StringUtils.EMPTY;
    }

    public void setDevicePlatform(String devicePlatform) {
        this.devicePlatform = devicePlatform;
    }

    public String getUserFullName() {
        return (getRunData().containsKey(RunConfiguration.USER_FULL_NAME_PROPERTY))
                ? getRunData().get(RunConfiguration.USER_FULL_NAME_PROPERTY) : "";
    }

    public String getProjectName() {
        return (getRunData().containsKey(RunConfiguration.PROJECT_NAME_PROPERTY))
                ? getRunData().get(RunConfiguration.PROJECT_NAME_PROPERTY) : "";
    }

    public String getOs() {
        return (getRunData().containsKey(RunConfiguration.HOST_OS)) ? getRunData().get(RunConfiguration.HOST_OS) : "";
    }

    public String getHostName() {
        return (getRunData().containsKey(RunConfiguration.HOST_NAME)) ? getRunData().get(RunConfiguration.HOST_NAME)
                : "";
    }

    public String getAppVersion() {
        return (getRunData().containsKey(RunConfiguration.APP_VERSION)) ? getRunData().get(RunConfiguration.APP_VERSION)
                : "";
    }

    public String getQtestBuildNumber() {
        return getBuildData().containsKey("qTestBuildNumber") ? getBuildData().get("qTestBuildNumber") : "";
    }

    public String getQtestBuildURL() {
        return getBuildData().containsKey("qTestBuildURL") ? getBuildData().get("qTestBuildURL") : "";
    }

    public String getAzurebBuildDefinitionId() {
        if (getBuildData().containsKey("adoDefinitionId")) {
            return getBuildData().get("adoDefinitionId");
        }
        if (getBuildData().containsKey("adoDefinitionID")) {
            return getBuildData().get("adoDefinitionID");
        }
        if (getBuildData().containsKey("adoBuildDefId")) {
            return getBuildData().get("adoBuildDefId");
        }
        if (getBuildData().containsKey("adoBuildDefID")) {
            return getBuildData().get("adoBuildDefID");
        }
        return StringUtils.EMPTY;
    }

    public String getAzurebReleaseDefinitionId() {
        if (getBuildData().containsKey("adoReleaseDefId")) {
            return getBuildData().get("adoReleaseDefId");
        }
        if (getBuildData().containsKey("adoReleaseDefID")) {
            return getBuildData().get("adoReleaseDefID");
        }
        return StringUtils.EMPTY;
    }

    public Map<String, String> getRunData() {
        return runData;
    }

    public void addRunData(Map<String, String> runData) {
        this.runData.putAll(runData);
    }

    public Map<String, String> getBuildData() {
        return buildData;
    }

    public void addBuildData(Map<String, String> buildData) {
        this.buildData.putAll(buildData);
    }

    public <T extends ILogRecord> int getChildIndex(T child) {
        return Arrays.asList(getChildRecords()).indexOf(child);
    }

    public List<String> getLogFiles() {
        List<String> logFiles = new ArrayList<String>();
        for (String childFile : new File(getLogFolder()).list()) {
            if (!FilenameUtils.getExtension(childFile).equals("log")) {
                continue;
            }
            logFiles.add(childFile);
        }
        return logFiles;
    }

    @Override
    public String getSystemOutMsg() {
        return getJUnitMessage();
    }

    @Override
    public String getSystemErrorMsg() {
        TestStatus status = getStatus();
        String stackTrace = status.getStackTrace();
        if (status.getStatusValue().isError()) {
            return getJUnitMessage() + stackTrace;
        }
        return stackTrace;
    }

    public TestCaseLogRecord getLastTestCaseLogRecord() {
        return getChildRecords().length > 0 ? (TestCaseLogRecord) getChildRecords()[getChildRecords().length - 1]
                : null;
    }

    public List<TestCaseLogRecord> getAllTestCaseLogRecords() {
        return Arrays.stream(filterFinalTestCasesResult())
                .filter(TestCaseLogRecord.class::isInstance)
                .map(TestCaseLogRecord.class::cast)
                .collect(Collectors.toList());
    }

    private String formatDevicePlatform(String devicePlatform) {
        if (StringUtils.isBlank(devicePlatform)) {
            return StringUtils.EMPTY;
        }

        if (StringUtils.equals(devicePlatform.toLowerCase(), "ios")) {
            return "iOS";
        }

        if (StringUtils.equals(devicePlatform.toLowerCase(), "macos")
                || StringUtils.equals(devicePlatform.toLowerCase(), "mac os x")
                || StringUtils.equals(devicePlatform.toLowerCase(), "mac")) {
            return "macOS";
        }

        return StringUtils.capitalize(devicePlatform.toLowerCase());
    }

    public List<ILogRecord> getTestSuiteListenerLogRecords() {
        if (testSuiteLogRecords == null) {
            return new ArrayList<>();
        }
        return testSuiteLogRecords;
    }

    public void addTestSuiteLogRecords(ILogRecord logRecord) {
        if (testSuiteLogRecords == null) {
            testSuiteLogRecords = new ArrayList<>();
        }
        this.testSuiteLogRecords.add(logRecord);
    }

    public long getFirstTestCaseStartTime() {
        ILogRecord[] childRecords = getChildRecords();
        if (childRecords.length == 0) {
            return 0;
        }
        return childRecords[0].getStartTime();
    }

    public List<ILogRecord> getBeforeTestSuiteLogRecords() {
        if (CollectionUtils.isEmpty(testSuiteLogRecords))
            return Collections.emptyList();
        if (!CollectionUtils.isEmpty(beforeTestSuiteLogRecords))
            return beforeTestSuiteLogRecords;
        long firstTestCaseStartTime = getFirstTestCaseStartTime();
        this.beforeTestSuiteLogRecords = testSuiteLogRecords.stream()
                .filter(logRecord -> logRecord.getStartTime() < firstTestCaseStartTime)
                .collect(Collectors.toList());
        return beforeTestSuiteLogRecords;
    }

    public List<ILogRecord> getAfterTestSuiteLogRecords() {
        if (CollectionUtils.isEmpty(testSuiteLogRecords))
            return Collections.emptyList();
        if (!CollectionUtils.isEmpty(afterTestSuiteLogRecords))
            return afterTestSuiteLogRecords;
        Set<ILogRecord> beforeLogRecordSet = new HashSet<>(getBeforeTestSuiteLogRecords());
        this.afterTestSuiteLogRecords = testSuiteLogRecords.stream()
                .filter(logRecord -> !beforeLogRecordSet.contains(logRecord))
                .collect(Collectors.toList());
        return afterTestSuiteLogRecords;
    }

    @Override
    public TestStatus getStatus() {
        TestStatus status = new TestStatus();
        Set<TestStatusValue> statuses = getAllTestCaseLogRecords().stream()
                .map(testCase -> testCase.getStatus().getStatusValue())
                .collect(Collectors.toSet());

        // Priority order: ERROR > FAILED > PASSED > SKIPPED
        if (statuses.contains(TestStatusValue.ERROR)) {
            status.setStatusValue(TestStatusValue.ERROR);
        } else if (statuses.contains(TestStatusValue.FAILED)) {
            status.setStatusValue(TestStatusValue.FAILED);
        } else if (statuses.contains(TestStatusValue.PASSED)) {
            status.setStatusValue(TestStatusValue.PASSED);
        } else if (statuses.contains(TestStatusValue.SKIPPED)) {
            status.setStatusValue(TestStatusValue.SKIPPED);
        } else {
            status.setStatusValue(TestStatusValue.INCOMPLETE);
        }
        return status;
    }
    
    public String getErrorMessage() {
        return getAllTestCaseLogRecords().stream()
            .filter(tc -> ERROR_STATUSES.contains(tc.getStatus().getStatusValue()))
            .map(TestCaseLogRecord::getMessage)
            .filter(StringUtils::isNotBlank)
            .findFirst()
            .orElse(StringUtils.EMPTY);
    }

    public String getExecutionProfile() {
        return executionProfile;
    }

    public void setExecutionProfile(String executionProfile) {
        this.executionProfile = executionProfile;
    }
    
    public Map<String, Map<String, Integer>> getOrigin() {
        Map<String, Map<String, Integer>> originMap = new HashMap<>();
        for (TestCaseLogRecord testCaseLog : getAllTestCaseLogRecords()) {
            String origin = testCaseLog.getOrigin();
            if (StringUtils.isBlank(origin)) {
                continue;
            }
            Map<String, Integer> countMap = originMap.getOrDefault(origin, new HashMap<>());
            countMap.put(TOTAL_TEST_CASES, countMap.getOrDefault(TOTAL_TEST_CASES, 0) + 1);
            TestStatusValue statusValue = testCaseLog.getStatus().getStatusValue();
            countMap.put(PASSED_TEST_CASES, countMap.getOrDefault(PASSED_TEST_CASES, 0) + (statusValue == TestStatusValue.PASSED ? 1 : 0));
            countMap.put(FAILED_TEST_CASES, countMap.getOrDefault(FAILED_TEST_CASES, 0) + (statusValue == TestStatusValue.FAILED ? 1 : 0));
            countMap.put(ERROR_TEST_CASES, countMap.getOrDefault(ERROR_TEST_CASES, 0) + (statusValue == TestStatusValue.ERROR ? 1 : 0));
            countMap.put(INCOMPLETE_TEST_CASES, countMap.getOrDefault(INCOMPLETE_TEST_CASES, 0) + (statusValue == TestStatusValue.INCOMPLETE ? 1 : 0));
            countMap.put(SKIPPED_TEST_CASES, countMap.getOrDefault(SKIPPED_TEST_CASES, 0) + (statusValue == TestStatusValue.SKIPPED ? 1 : 0));
            originMap.put(origin, countMap);
        }
        return originMap;
    }
}
