package com.kms.katalon.core.reporting.newreport.extractor;

import java.util.Optional;

import org.apache.commons.lang3.StringUtils;

/**
 * Extracts failure reasons from messages with an explicit "Reason:" section.
 * <p>
 * This format is commonly used in Katalon-specific exceptions where the reason
 * is clearly separated from the stacktrace.
 * </p>
 * 
 * <p><b>Example input:</b></p>
 * <pre>
 * Test case 'Login Test' FAILED.
 * Reason:
 * groovy.lang.MissingPropertyException: No such property: username for class: LoginPage
 *     at LoginPage.fillCredentials(LoginPage.groovy:25)
 * </pre>
 * 
 * <p><b>Example output:</b></p>
 * <pre>groovy.lang.MissingPropertyException: No such property: username for class: LoginPage</pre>
 */
public class ReasonSectionExtractor implements IFailureReasonExtractor {
    private static final String REASON_MARKER = "Reason:\n";

    @Override
    public Optional<String> extract(String message) {
        if (StringUtils.isBlank(message)) {
            return Optional.empty();
        }

        String reason = extractFromMarker(message, REASON_MARKER);
        if (reason != null) {
            return Optional.of(reason);
        }

        return Optional.empty();
    }

    private String extractFromMarker(String message, String marker) {
        if (!message.contains(marker)) {
            return null;
        }

        try {
            String[] parts = message.split(marker, 2);
            if (parts.length > 1) {
                String reasonBody = parts[1].trim();
                if (StringUtils.isNotBlank(reasonBody)) {
                    String firstLine = reasonBody.split("\n")[0].trim();
                    if (StringUtils.isNotBlank(firstLine)) {
                        return firstLine;
                    }
                }
            }
        } catch (Exception ignored) {
        }

        return null;
    }
}
