package com.kms.katalon.core.testdata.reader;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;
import java.text.MessageFormat;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.supercsv.cellprocessor.Optional;
import org.supercsv.cellprocessor.constraint.NotNull;
import org.supercsv.cellprocessor.ift.CellProcessor;
import org.supercsv.io.CsvListWriter;
import org.supercsv.io.ICsvListWriter;
import org.supercsv.prefs.CsvPreference;

import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.logging.model.ILogRecord;
import com.kms.katalon.core.logging.model.TestCaseLogRecord;
import com.kms.katalon.core.logging.model.TestStatus;
import com.kms.katalon.core.logging.model.TestSuiteLogRecord;
import com.kms.katalon.core.reporting.ReportUtil;
import com.kms.katalon.core.util.internal.DateUtil;
import com.kms.katalon.core.util.internal.JsonUtil;

public class CsvWriter {

    private static final CellProcessor[] SUMMARY_PROCESSORS = new CellProcessor[] { 
            new NotNull(), // Suite Name
            new NotNull(), // Test Name
            new Optional(), // Browser
            new Optional(), // Description
            new Optional(), // Tag
            new NotNull(), // Start time
            new Optional(), // End Time
            new Optional(), // Duration
            new NotNull() // Status
            };

    private static final CellProcessor[] DETAILS_PROCESSORS = new CellProcessor[] { 
            new NotNull(), // Suite/Test/Step Name
            new Optional(), // Browser
            new Optional(), // Description
            new Optional(), // Tag
            new NotNull(), // Start time
            new Optional(), // End Time
            new Optional(), // Duration
            new NotNull() // Status
            };

    public static final String[] SUMMARY_HEADER = new String[] {
            "Suite Name",
            "Test Name",
            "Browser",
            "Description",
            "Tag",
            "Start time",
            "End time",
            "Duration",
            "Status" };

    public static final String[] DETAILS_HEADER = new String[] {
            "Suite/Test/Step Name",
            "Browser",
            "Description",
            "Tag",
            "Start time",
            "End time",
            "Duration",
            "Status" };

    public static void writeCsvReport(TestSuiteLogRecord suiteLog, File file, List<ILogRecord> filteredTestCaseRecords)
            throws IOException {
        writeCsvReport(suiteLog, file, filteredTestCaseRecords, true);
    }

    public static void writeCsvReport(TestSuiteLogRecord suiteLog, File file, List<ILogRecord> filteredTestCaseRecords,
            boolean stepsIncluded) throws IOException {
        ICsvListWriter csvWriter = new CsvListWriter(new OutputStreamWriter(new FileOutputStream(file), StandardCharsets.UTF_8), CsvPreference.STANDARD_PREFERENCE);

        try {
            // Test Suite
            String browser = suiteLog.getBrowser();

            csvWriter.writeHeader(DETAILS_HEADER);

            writeRecord(csvWriter, suiteLog, browser);
            
            writeTestSuiteListenerLogRecords(csvWriter, suiteLog.getBeforeTestSuiteLogRecords(), browser);

            // Test cases
            for (ILogRecord testLog : filteredTestCaseRecords) {
                // Blank line
                writeEmptyLine(csvWriter);
                
                // Write test case line
                writeRecord(csvWriter, testLog, browser);

                if (testLog instanceof TestCaseLogRecord) {
                    String rawDataBinding = ReportUtil.getTestCaseLogRecordProperty((TestCaseLogRecord) testLog,
                            StringConstants.EXECUTION_BINDING_VARIABLES);
                    Map<String, Object> dataBinding = ReportUtil.parseDataBindingAsMap(rawDataBinding);
                    writeDataBinding(csvWriter, dataBinding);
                }

                // Test steps
                if (stepsIncluded) {
                    for (ILogRecord step : testLog.getChildRecords()) {
                        writeRecord(csvWriter, step, browser);
                    }
                }
            }
            writeTestSuiteListenerLogRecords(csvWriter, suiteLog.getAfterTestSuiteLogRecords(), browser);
        } finally {
            IOUtils.closeQuietly(csvWriter);
        }
    }
    
    private static void writeEmptyLine(ICsvListWriter csvWriter) throws IOException {
        csvWriter.write(Arrays.asList("", "", "", "", "", "", "", ""), DETAILS_PROCESSORS);
    }

    private static void writeTestSuiteListenerLogRecords(ICsvListWriter csvWriter, List<ILogRecord> logs, String browser) throws IOException {
        if (!CollectionUtils.isEmpty(logs)) {
            writeEmptyLine(csvWriter);
            for (ILogRecord log : logs) {
                writeRecord(csvWriter, log, browser);
            }
        }
    }

    private static void writeDataBinding(ICsvListWriter csvWriter, Map<String, Object> dataBinding) throws IOException {
        if (dataBinding == null || dataBinding.isEmpty()) {
            return;
        }
        writeSimpleLine(csvWriter, MessageFormat.format("DATA BINDING - {0}", JsonUtil.toJson(dataBinding, false)));
    }

    private static void writeSimpleLine(ICsvListWriter csvWriter, String content) throws IOException {
        List<Object> writtenObjects = Arrays.asList(new Object[] { content, "", "", "", "", "", "", "" });
        csvWriter.write(writtenObjects, DETAILS_PROCESSORS);
    }

    private static void writeRecord(ICsvListWriter csvWriter, ILogRecord logRecord, String browserName)
            throws IOException {
        String tag = "";
        String testCaseName = logRecord.getName();

        if (logRecord instanceof TestCaseLogRecord) {
            TestCaseLogRecord testCaseLogRecord = (TestCaseLogRecord) logRecord;
            tag = StringUtils.defaultString(testCaseLogRecord.getTag());
            String iteration = testCaseLogRecord.getIterationVariableValue();
            if (StringUtils.isNotEmpty(iteration)) {
                testCaseName += " | " + iteration;
            }
        }

        TestStatus status = logRecord.getStatus();
        if (logRecord instanceof TestSuiteLogRecord) {
            status = new TestStatus();
            status.setStatusValue(((TestSuiteLogRecord) logRecord).getSummaryStatus());
        }

        List<Object> writtenObjects = Arrays.asList(
            testCaseName,
            browserName,
            logRecord.getDescription(),
            tag,
            DateUtil.getDateTimeFormatted(logRecord.getStartTime()),
            DateUtil.getDateTimeFormatted(logRecord.getEndTime()),
            DateUtil.getElapsedTime(logRecord.getStartTime(), logRecord.getEndTime()),
            status != null ? status.getStatusValue().name() : ""
        );

        csvWriter.write(writtenObjects, DETAILS_PROCESSORS);
    }

    public static void writeArraysToCsv(String[] header, List<Object[]> datas, File csvFile) throws IOException {
        ICsvListWriter csvWriter = null;
        try {
            csvWriter = new CsvListWriter(new OutputStreamWriter(new FileOutputStream(csvFile), StandardCharsets.UTF_8), CsvPreference.STANDARD_PREFERENCE);
            csvWriter.writeHeader(header);
            for (Object[] arr : datas) {
                csvWriter.write(Arrays.asList(arr), SUMMARY_PROCESSORS);
            }
        } finally {
            IOUtils.closeQuietly(csvWriter);
        }
    }

    public static void writeArraysToCsv(String[] header, List<Object[]> datas, File csvFile,
            CellProcessor[] cellProcessor) throws IOException {
        ICsvListWriter csvWriter = null;
        try {
            csvWriter = new CsvListWriter(new OutputStreamWriter(new FileOutputStream(csvFile), StandardCharsets.UTF_8),
                    CsvPreference.STANDARD_PREFERENCE);
            csvWriter.writeHeader(header);
            for (Object[] arr : datas) {
                csvWriter.write(Arrays.asList(arr), cellProcessor);
            }
        } finally {
            IOUtils.closeQuietly(csvWriter);
        }
    }
}
