package com.kms.katalon.core.util.internal;

import java.util.Arrays;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;
import org.codehaus.groovy.runtime.StackTraceUtils;

import com.google.common.base.Throwables;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.main.ScriptEngine;
import com.kms.katalon.core.util.TestCloudPropertyUtil;

import groovy.lang.MissingPropertyException;

public class ExceptionsUtil {

    private static final Pattern CLASS_PATTERN = Pattern.compile("Script[0-9]{13}");

    private static final Pattern HOSTNAME_REGEX = Pattern.compile("host: '(.*?)'");

    private static final Pattern IP_REGEX = Pattern.compile("ip: '(.*?)'");

    private static final int REPEAT_TIMES = 10;

    public static String getMessageForThrowable(Throwable t) {
        if (t == null) {
            return "";
        }
        return handleExceptionMessage(getExceptionMessage(t));
    }

    private static String getExceptionMessage(Throwable throwable) {
        if (throwable instanceof MissingPropertyException) {
            return getExceptionMessage((MissingPropertyException) throwable);
        } else {
            return throwable.getClass().getName()
                    + (throwable.getMessage() != null ? (": " + throwable.getMessage()) : "");
        }
    }

    private static String getExceptionMessage(MissingPropertyException exception) {
        return "Variable '" + exception.getProperty() + "' is not defined for test case.";
    }

    public static String getStackTraceForThrowable(Throwable t) {
        t.setStackTrace(getRawStackTraceForThrowable(t));
        String stackTrace = Throwables.getStackTraceAsString(t);
        if (TestCloudPropertyUtil.getInstance().isRunFromTestCloud()) {
            stackTrace = TestCloudPropertyUtil.getInstance().hideSauceLabsDocumentLink(stackTrace);
        }
        return handleExceptionMessage(stackTrace);
    }

    public static StackTraceElement[] getRawStackTraceForThrowable(Throwable t) {
        t = StackTraceUtils.deepSanitize(t);
        return Arrays.stream(t.getStackTrace()).map(stackTraceElement -> {
            String declaringClass = stackTraceElement.getClassName();
            if (CLASS_PATTERN.matcher(declaringClass).matches()) {
                declaringClass = ScriptEngine.getTestCaseName(declaringClass);
                if (declaringClass != null) {
                    return new StackTraceElement(declaringClass, stackTraceElement.getMethodName(), declaringClass,
                            stackTraceElement.getLineNumber());
                }
            }
            return stackTraceElement;
        }).collect(Collectors.toList()).toArray(new StackTraceElement[] {});
    }

    public static String handleExceptionMessage(String message) {
        String customMessage = message;
        if (RunConfiguration.isHideHostname()) {
            Matcher hostname = HOSTNAME_REGEX.matcher(customMessage);
            Matcher ip = IP_REGEX.matcher(customMessage);
            if (hostname.find()) {
                customMessage = customMessage.replace(hostname.group(1), StringUtils.repeat("*", REPEAT_TIMES));
            }
            if (ip.find()) {
                customMessage = customMessage.replace(ip.group(1), StringUtils.repeat("*", REPEAT_TIMES));
            }
            return customMessage;
        }
        return customMessage;
    }

    public static Exception customException(Exception e) {
        String customMessage = handleExceptionMessage(e.getMessage());
        return new Exception(customMessage);
    }
}
