package com.kms.katalon.core.appium.driver;

import java.io.File;
import java.io.FileReader;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.ServerSocket;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.GeneralSecurityException;
import java.text.MessageFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.SystemUtils;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.Platform;
import org.openqa.selenium.net.UrlChecker;
import org.openqa.selenium.net.UrlChecker.TimeoutException;
import org.openqa.selenium.os.CommandLine;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebDriver;
import org.openqa.selenium.remote.UnreachableBrowserException;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.kms.katalon.core.appium.constants.AppiumStringConstants;
import com.kms.katalon.core.appium.constants.CoreAppiumMessageConstants;
import com.kms.katalon.core.appium.constants.XCUITestSettingConstants;
import com.kms.katalon.core.appium.exception.AppiumStartException;
import com.kms.katalon.core.appium.exception.IOSWebkitStartException;
import com.kms.katalon.core.appium.exception.MobileDriverInitializeException;
import com.kms.katalon.core.appium.util.AppiumDriverUtil;
import com.kms.katalon.core.appium.util.AppiumVersionUtil;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.driver.IDriverType;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.keyword.internal.KeywordExecutor;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.logging.LogLevel;
import com.kms.katalon.core.util.ConsoleCommandExecutor;
import com.kms.katalon.core.util.CpuUtil;
import com.kms.katalon.core.util.TestCloudPropertyUtil;
import com.kms.katalon.core.util.internal.JsonUtil;
import com.kms.katalon.core.util.internal.ProcessUtil;
import com.kms.katalon.selenium.exception.W3CCapabilityViolationException;
import com.kms.katalon.util.CryptoUtil;

import io.appium.java_client.AppiumDriver;
import io.appium.java_client.android.AndroidDriver;
import io.appium.java_client.ios.IOSDriver;
import io.appium.java_client.remote.AppiumCommandExecutor;
import io.appium.java_client.service.local.InvalidNodeJSInstance;
import io.appium.java_client.service.local.flags.GeneralServerFlag;

public class AppiumDriverManager {

    private static final KeywordLogger logger = KeywordLogger.getInstance(AppiumDriverManager.class);

    public static final String WDA_LOCAL_PORT = "wdaLocalPort";

    public static final String SYSTEM_PORT = "systemPort";

    public static final String REAL_DEVICE_LOGGER = "realDeviceLogger";

    public static final String UIAUTOMATOR2 = "uiautomator2";

    public static final String XCUI_TEST = "XCUITest";

    private static final String XCODE = "Xcode";

    public static final String NODE_PATH = "NODE_BINARY_PATH";

    private static final String PORT_ARGUMENT = "-p";

    private static final String NODE_EXECUTABLE = "node";

    public static final String EXECUTED_PLATFORM = AppiumStringConstants.CONF_EXECUTED_PLATFORM;

    public static final String EXECUTED_DEVICE_ID = AppiumStringConstants.CONF_EXECUTED_DEVICE_ID;

    public static final String EXECUTED_DEVICE_MANUFACTURER = AppiumStringConstants.CONF_EXECUTED_DEVICE_MANUFACTURER;

    public static final String EXECUTED_DEVICE_MODEL = AppiumStringConstants.CONF_EXECUTED_DEVICE_MODEL;

    public static final String EXECUTED_DEVICE_NAME = AppiumStringConstants.CONF_EXECUTED_DEVICE_NAME;

    public static final String EXECUTED_DEVICE_OS = AppiumStringConstants.CONF_EXECUTED_DEVICE_OS;

    public static final String EXECUTED_DEVICE_OS_VERSON = AppiumStringConstants.CONF_EXECUTED_DEVICE_OS_VERSON;

    private static String APPIUM_RELATIVE_PATH_FROM_APPIUM_FOLDER_OLD = "bin" + File.separator + "appium.js";

    private static String APPIUM_RELATIVE_PATH_FROM_APPIUM_FOLDER_NEW = "build" + File.separator + "lib"
            + File.separator + "main.js";

    private static String APPIUM_RELATIVE_PATH_FROM_APPIUM_GUI = "node_modules" + File.separator + "appium";

    private static final String APPIUM_TEMP_RELATIVE_PATH = System.getProperty("java.io.tmpdir") + File.separator
            + "Katalon" + File.separator + "Appium" + File.separator + "Temp";

    private static final String C_FLAG = "-c";

    private static final String DEFAULT_APPIUM_SERVER_ADDRESS = "127.0.0.1";

    private static final String APPIUM_SERVER_URL_PREFIX = "http://" + DEFAULT_APPIUM_SERVER_ADDRESS + ":";

    private static final String APPIUM_SERVER_URL_SUFFIX = "/wd/hub";

    private static final String IOS_WEBKIT_DEBUG_PROXY_EXECUTABLE = "ios_webkit_debug_proxy";

    private static final String IOS_WEBKIT_LOG_FILE_NAME = "appium-proxy-server.log";

    private static final String MSG_START_IOS_WEBKIT_SUCCESS = "ios_webkit_debug_proxy server started on port ";

    private static final String LOCALHOST_PREFIX = "http://localhost:";

    public static final String REMOTE_WEB_DRIVER_URL = "remoteWebDriverUrl";

    public static final String IS_REMOTE_WEB_DRIVER_URL_ENCRYPTED = "isEncrypted";

    private static final ThreadLocal<Process> localStorageWebProxyProcess = new ThreadLocal<Process>() {
        @Override
        protected Process initialValue() {
            return null;
        }
    };

    private static final ThreadLocal<Process> localStorageAppiumServer = new ThreadLocal<Process>() {
        @Override
        protected Process initialValue() {
            return null;
        }
    };

    private static final ThreadLocal<Integer> localStorageAppiumPort = new ThreadLocal<Integer>() {
        @Override
        protected Integer initialValue() {
            return 0;
        }
    };

    private static final ThreadLocal<Integer> localStorageWebProxyPort = new ThreadLocal<Integer>() {
        @Override
        protected Integer initialValue() {
            return 0;
        }
    };

    private static final ThreadLocal<AppiumDriver> localStorageAppiumDriver = new ThreadLocal<AppiumDriver>() {
        @Override
        protected AppiumDriver initialValue() {
            return null;
        }
    };

    private static void ensureWebProxyServerStarted(String deviceId)
            throws IOException, InterruptedException, IOSWebkitStartException {
        if (!isWebProxyServerStarted(1)) {
            startWebProxyServer(deviceId);
        }
    }

    /**
     * Start proxy server, this server is optional
     *
     * @param deviceId
     * @throws Exception
     */
    private static void startWebProxyServer(String deviceId)
            throws IOException, InterruptedException, IOSWebkitStartException {
        int freePort = getFreePort();
        String[] webProxyServerCmd = { "/bin/sh", "-c",
                String.format("%s -c %s:%d -d", IOS_WEBKIT_DEBUG_PROXY_EXECUTABLE, deviceId, freePort) };
        ProcessBuilder webProxyServerProcessBuilder = new ProcessBuilder(webProxyServerCmd);
        webProxyServerProcessBuilder
                .redirectOutput(new File(new File(RunConfiguration.getAppiumLogFilePath()).getParent() + File.separator
                        + IOS_WEBKIT_LOG_FILE_NAME));

        Process webProxyProcess = webProxyServerProcessBuilder.start();

        // Check again if proxy server started
        if (!isServerStarted(10, new URL(LOCALHOST_PREFIX + freePort))) {
            throw new IOSWebkitStartException();
        }
        localStorageWebProxyProcess.set(webProxyProcess);
        localStorageWebProxyPort.set(freePort);
        logger.logInfo(MSG_START_IOS_WEBKIT_SUCCESS + freePort);
    }

    public static boolean isAppiumServerStarted(int timeToWait) {
        if (localStorageAppiumServer.get() == null) {
            return false;
        }
        try {
            // Detect if the process still alive?
            localStorageAppiumServer.get().exitValue();
            return false;
        } catch (IllegalThreadStateException e) {
            // The process is still alive, continue to ping it's HTTP end-point
        }
        try {
            return isServerStarted(timeToWait, new URL("http://" + DEFAULT_APPIUM_SERVER_ADDRESS + ":"
                    + localStorageAppiumPort.get() + APPIUM_SERVER_URL_SUFFIX + "/status"));
        } catch (IOException mex) {
            return false;
        }
    }

    private static boolean isWebProxyServerStarted(int timeOut) {
        if (localStorageWebProxyProcess.get() == null) {
            return false;
        }
        try {
            localStorageWebProxyProcess.get().exitValue();
            return false;
        } catch (IllegalThreadStateException e) {
            // Process is running
        }
        try {
            return isServerStarted(timeOut, new URL(LOCALHOST_PREFIX + localStorageWebProxyPort.get()));
        } catch (MalformedURLException e) {
            return false;
        }
    }

    private static boolean isServerStarted(int timeToWait, URL url) {
        try {
            new UrlChecker().waitUntilAvailable(timeToWait, TimeUnit.SECONDS, url);
            return true;
        } catch (TimeoutException ex1) {}
        return false;
    }

    private static void ensureServicesStarted(IDriverType driverType, String deviceId)
            throws IOException, InterruptedException, AppiumStartException {
        if (isIOSDriverType(driverType)) {
            // Proxy server is optional
            try {
                ensureWebProxyServerStarted(deviceId);
            } catch (IOException | InterruptedException | IOSWebkitStartException e) {
                logger.logWarning(e.getMessage(), null, e);
            }

            AppiumDriverManager.pairDevice(deviceId);
        }
        startAppiumServerJS(RunConfiguration.getElementTimeoutForMobile());
    }

    private static boolean isAndroidDriverType(IDriverType driverType) {
        return driverType != null && StringUtils.equals(AppiumStringConstants.ANDROID, driverType.toString());
    }

    private static boolean isIOSDriverType(IDriverType driverType) {
        return driverType != null && StringUtils.equals(AppiumStringConstants.IOS, driverType.toString());
    }

    public static void startAppiumServerJS(int timeout, Map<String, String> environmentVariables)
            throws AppiumStartException, IOException {
        // Appium server started already?
        if (isAppiumServerStarted(1)) {
            return;
        }
        // If not, start it
        startAppiumServer(environmentVariables);
        if (isAppiumServerStarted(timeout)) {
            logger.logInfo(
                    MessageFormat.format(AppiumStringConstants.APPIUM_STARTED_ON_PORT, localStorageAppiumPort.get()));
            return;
        }
        throw new AppiumStartException(MessageFormat
                .format(CoreAppiumMessageConstants.ERR_MSG_CANNOT_START_APPIUM_SERVER_AFTER_X_SECONDS, timeout));
    }

    public static void pairDevice(String deviceId) {
        try {
            if (StringUtils.isEmpty(deviceId) || deviceId.contains("-")) {
                return;
            }

            List<String> pairedInfos = ConsoleCommandExecutor.execSync("idevicepair validate -u " + deviceId, false);
            String pairedString = StringUtils.join(pairedInfos, "\n");
            if (!StringUtils.containsIgnoreCase(pairedString, "SUCCESS")) {
                // Call 'idevicepair pair' to show 'TRUST' dialog on the device
                ConsoleCommandExecutor.execSync("idevicepair pair -u " + deviceId, false);
                throw new InterruptedException();
            }
        } catch (IOException | InterruptedException e) {
            logger.logWarning("Device " + deviceId + " is not paired! Please tap 'TRUST' on device screen to pair");
        }
    }

    private static void startAppiumServer(Map<String, String> environmentVariables)
            throws AppiumStartException, IOException {
        if (localStorageAppiumServer.get() != null && localStorageAppiumServer.get().isAlive()) {
            return;
        }

        String appium = findAppiumJS();
        String appiumTemp = createAppiumTempFile();
        String appiumHome = appium;
        if (!AppiumVersionUtil.getInstance().isAppiumV1()) {
            appiumHome = (new File(SystemUtils.getUserHome(), ".appium")).getAbsolutePath();
        }

        File node = findNodeInCurrentFileSystem();
        localStorageAppiumPort.set(getFreePort());
        List<String> cmdList = new ArrayList<String>();
        // Identify whether the Mac is using an Intel or M chip and
        // ensure that the process starts in the appropriate mode.
        if (System.getProperty("os.name").toLowerCase().contains("mac")) {
            cmdList.add("arch");
            cmdList.add("-" + CpuUtil.getCpuArch());
        }

        cmdList.add(node.getAbsolutePath());
        cmdList.add(appium);
        cmdList.add(GeneralServerFlag.TEMP_DIRECTORY.getArgument());
        cmdList.add(appiumTemp);
        cmdList.add(PORT_ARGUMENT);
        cmdList.add(String.valueOf(localStorageAppiumPort.get()));
        if (!AppiumVersionUtil.getInstance().isAppiumV1()) {
            cmdList.add("--base-path");
            cmdList.add(APPIUM_SERVER_URL_SUFFIX);

            File packageJsonFile = new File(appiumHome, "package.json");
            if (packageJsonFile.exists()) {
                try (FileReader reader = new FileReader(packageJsonFile)) {
                    JsonObject root = JsonParser.parseReader(reader).getAsJsonObject();
                    JsonObject devDependencies = root.has("devDependencies")
                            && root.get("devDependencies").isJsonObject() ? root.getAsJsonObject("devDependencies")
                                    : null;

                    if (devDependencies != null && devDependencies.has("@appium/images-plugin")) {
                        cmdList.add("--use-plugins");
                        cmdList.add("images");
                    }
                } catch (IOException | IllegalStateException e) {
                    logger.logError("Failed to parse package.json: " + e.getMessage());
                }
            }
        }

        cmdList.add(GeneralServerFlag.LOG_LEVEL.getArgument());
        cmdList.add(getAppiumLogLevel());
        ProcessBuilder pb = new ProcessBuilder(cmdList.toArray(new String[cmdList.size()]));
        pb.environment().putAll(environmentVariables);

        // The Appium server requires the node binary path to run
        String path = pb.environment().get("PATH");
        if (path == null) {
            path = "";
        }

        String fileSeparator = FileSystems.getDefault().getSeparator();

        try {
            Path localPath = Paths.get(fileSeparator, "usr", "local", "bin");
            if (Files.exists(localPath) && !path.toLowerCase().contains(localPath.toString().toLowerCase())) {
                path = String.join(File.pathSeparator, new String[] { localPath.toString(), path });
            }
        } catch (Exception ignored) {}

        try {
            Path homebrewPath = Paths.get(fileSeparator, "opt", "homebrew", "bin");
            if (Files.exists(homebrewPath) && !path.toLowerCase().contains(homebrewPath.toString().toLowerCase())) {
                path = String.join(File.pathSeparator, new String[] { homebrewPath.toString(), path });
            }
        } catch (Exception ignored) {}

        File nodeParentPath = node.getParentFile();
        if (nodeParentPath.exists() && !path.toLowerCase().contains(nodeParentPath.toString().toLowerCase())) {
            path = String.join(File.pathSeparator, new String[] { nodeParentPath.toString(), path });
        }

        logger.logInfo("PATH: " + path);
        pb.environment().put("PATH", path);

        if (StringUtils.isNotBlank(pb.environment().get("APPIUM_HOME"))) {
            appiumHome = pb.environment().get("APPIUM_HOME");
        }

        logger.logInfo("APPIUM_HOME: " + appiumHome);
        pb.environment().put("APPIUM_HOME", appiumHome);

        final String appiumOutLogFilePath = RunConfiguration.getAppiumLogFilePath();
        final String appiumErrorLogFilePath = Paths
                .get(Paths.get(appiumOutLogFilePath).getParent().toString(), "appium_error.log")
                .toString();
        logger.logInfo("appium outlog: " + appiumOutLogFilePath);
        logger.logInfo("appium errlog: " + appiumErrorLogFilePath);
        pb.redirectOutput(new File(appiumOutLogFilePath));
        pb.redirectError(new File(appiumErrorLogFilePath));
        localStorageAppiumServer.set(pb.start());
        new Thread(AppiumOutputStreamHandler.create(appiumOutLogFilePath, System.out)).start();
    }

    private static File findNodeInCurrentFileSystem() {
        String nodeJSExec = System.getProperty(NODE_PATH);
        if (StringUtils.isBlank(nodeJSExec)) {
            nodeJSExec = System.getenv(NODE_PATH);
        }
        if (StringUtils.isBlank(nodeJSExec)) {
            nodeJSExec = ConsoleCommandExecutor.safeWhere(NODE_EXECUTABLE);
        }
        if (!StringUtils.isBlank(nodeJSExec)) {
            File result = new File(nodeJSExec);
            if (result.exists()) {
                logger.logInfo(MessageFormat.format(CoreAppiumMessageConstants.NODEJS_LOCATION_INFO, nodeJSExec));
                return result;
            }
        }

        CommandLine commandLine;
        File getNodeJSExecutable = Scripts.GET_NODE_JS_EXECUTABLE.getScriptFile();
        try {
            if (Platform.getCurrent().is(Platform.WINDOWS)) {
                commandLine = new CommandLine(NODE_EXECUTABLE + ".exe", getNodeJSExecutable.getAbsolutePath());
            } else {
                commandLine = new CommandLine(NODE_EXECUTABLE, getNodeJSExecutable.getAbsolutePath());
            }
            commandLine.execute();
        } catch (Throwable t) {
            throw new InvalidNodeJSInstance("Node.js is not installed!", t);
        }

        String filePath = (commandLine.getStdOut()).trim();

        try {
            if (StringUtils.isBlank(filePath) || !new File(filePath).exists()) {
                String errorOutput = commandLine.getStdOut();
                String errorMessage = "Can't get a path to the default Node.js instance";
                throw new InvalidNodeJSInstance(errorMessage, new IOException(errorOutput));
            }
            logger.logInfo(MessageFormat.format(CoreAppiumMessageConstants.NODEJS_LOCATION_INFO, filePath));
            return new File(filePath);
        } finally {
            commandLine.destroy();
        }
    }

    private static String getAppiumLogLevel() {
        return RunConfiguration.getDriverSystemProperty(StringConstants.CONF_PROPERTY_MOBILE_DRIVER,
                StringConstants.CONF_APPIUM_LOG_LEVEL);
    }

    private static String createAppiumTempFile() {
        return APPIUM_TEMP_RELATIVE_PATH + System.currentTimeMillis();
    }

    private static String findAppiumJS() throws AppiumStartException {
        String appiumHome = RunConfiguration.getAppiumDirectory();
        if (StringUtils.isEmpty(appiumHome)) {
            throw new AppiumStartException(AppiumStringConstants.APPIUM_START_EXCEPTION_APPIUM_DIRECTORY_NOT_SET);
        }
        String appium = getAppiumJSPathFromNPMBuild(appiumHome);
        if (!new File(appium).exists()) {
            appium = getAppiumJSPathFromAppiumGUI(appiumHome);
        }
        if (!new File(appium).exists()) {
            throw new AppiumStartException(
                    AppiumStringConstants.APPIUM_START_EXCEPTION_APPIUM_DIRECTORY_INVALID_CANNOT_FIND_APPIUM_JS);
        }
        return appium;
    }

    private static String getAppiumJSPathFromAppiumGUI(String appiumHome) {
        String appiumFolderFromGUIAppium = appiumHome + File.separator + APPIUM_RELATIVE_PATH_FROM_APPIUM_GUI;
        return getAppiumJSPathFromNPMBuild(appiumFolderFromGUIAppium);
    }

    private static String getAppiumJSPathFromNPMBuild(String appiumHome) {
        String oldAppiumJSPath = appiumHome + File.separator + APPIUM_RELATIVE_PATH_FROM_APPIUM_FOLDER_OLD;
        if (!new File(oldAppiumJSPath).exists()) {
            return appiumHome + File.separator + APPIUM_RELATIVE_PATH_FROM_APPIUM_FOLDER_NEW;
        }
        return oldAppiumJSPath;
    }

    public static void startAppiumServerJS(int timeout) throws AppiumStartException, IOException {
        startAppiumServerJS(timeout, new HashMap<String, String>());
    }

    public static int nextFreePort(int from, int to) {
        int port = ThreadLocalRandom.current().nextInt(from, to);
        while (true) {
            if (isLocalPortFree(port)) {
                return port;
            } else {
                port = ThreadLocalRandom.current().nextInt(from, to);
            }
        }
    }

    private static boolean isLocalPortFree(int port) {
        try {
            new ServerSocket(port).close();
            return true;
        } catch (IOException e) {
            return false;
        }
    }

    public static synchronized int getFreePort() {
        try (ServerSocket s = new ServerSocket(0)) {
            return s.getLocalPort();
        } catch (IOException e) {
            // do nothing
        }
        return -1;
    }

    public static AppiumDriver startExisitingMobileDriver(IDriverType driverType, String sessionId,
            String remoteServerUrl) throws MalformedURLException, MobileDriverInitializeException {
        int time = 0;
        long currentMilis = System.currentTimeMillis();
        int timeOut = RunConfiguration.getElementTimeoutForMobile();
        while (time < timeOut) {
            try {
                AppiumDriver driver = null;
                if (isIOSDriverType(driverType)) {
                    driver = new ExistingIosDriver(new URL(remoteServerUrl), sessionId);
                } else if (isAndroidDriverType(driverType)) {
                    driver = new ExistingAndroidDriver(new URL(remoteServerUrl), sessionId);
                }

                if (driver == null) {
                    throw new MobileDriverInitializeException(MessageFormat
                            .format(AppiumStringConstants.CANNOT_START_MOBILE_DRIVER_INVALID_TYPE, driver));
                }

                localStorageAppiumDriver.set(driver);
                new AppiumRequestService(remoteServerUrl).logAppiumInfo();

                if (isIOSDriverType(driverType)) {
                    driver.setSetting(XCUITestSettingConstants.RESPECT_SYSTEM_ALERTS, true);
                }

                return driver;
            } catch (UnreachableBrowserException e) {
                long newMilis = System.currentTimeMillis();
                time += ((newMilis - currentMilis) / 1000);
                currentMilis = newMilis;
                continue;
            }
        }
        throw new MobileDriverInitializeException(
                MessageFormat.format(AppiumStringConstants.CANNOT_CONNECT_TO_APPIUM_AFTER_X, timeOut));
    }

    public static AppiumDriver createMobileDriver(IDriverType driverType, String deviceId,
            DesiredCapabilities capabilities) throws IOException, InterruptedException, AppiumStartException,
            MobileDriverInitializeException, MalformedURLException {
        ensureServicesStarted(driverType, deviceId);
        Process appiumService = localStorageAppiumServer.get();
        if (appiumService == null) {
            throw new MobileDriverInitializeException(AppiumStringConstants.APPIUM_NOT_STARTED);
        }
        URL appiumServerUrl = new URL(
                APPIUM_SERVER_URL_PREFIX + localStorageAppiumPort.get() + APPIUM_SERVER_URL_SUFFIX);
        return createMobileDriver(driverType, capabilities, appiumServerUrl);
    }

    @SuppressWarnings("rawtypes")
    public static AppiumDriver createMobileDriver(IDriverType driverType, MutableCapabilities capabilities,
            URL appiumServerUrl) throws MobileDriverInitializeException {
        Capabilities responseCapabilities = null;
        try {
            int time = 0;
            long currentMilis = System.currentTimeMillis();
            int timeOut = RunConfiguration.getElementTimeoutForMobile();
            while (time < timeOut) {
                try {
                    AppiumDriver driver = null;
                    MutableCapabilities filteredCapabilities = AppiumDriverUtil
                            .filterByAcceptedW3CCapabilityKeys(capabilities);
                    var proxyInfo = RunConfiguration.getProxyInformation();

                    try {
                        AppiumCommandExecutor appiumExecutor = AppiumDriverUtil
                                .getAppiumExecutorForRemoteDriver(appiumServerUrl, proxyInfo, capabilities);
                        if (isIOSDriverType(driverType)) {
                            driver = new IOSDriver(appiumExecutor, filteredCapabilities);
                        } else if (isAndroidDriverType(driverType)) {
                            driver = new AndroidDriver(appiumExecutor, filteredCapabilities);
                        }
                    } catch (URISyntaxException | IOException e) {
                        logger.logWarning("Unable to create AppiumCommandExecutor under current proxy settings.");
                        if (isIOSDriverType(driverType)) {
                            driver = new IOSDriver(appiumServerUrl, filteredCapabilities);
                        } else if (isAndroidDriverType(driverType)) {
                            driver = new AndroidDriver(appiumServerUrl, filteredCapabilities);
                        }
                    }

                    if (driver == null) {
                        throw new MobileDriverInitializeException(MessageFormat
                                .format(AppiumStringConstants.CANNOT_START_MOBILE_DRIVER_INVALID_TYPE, driver));
                    }

                    localStorageAppiumDriver.set(driver);
                    new AppiumRequestService(appiumServerUrl.toString()).logAppiumInfo();

                    int timeout = RunConfiguration.getElementTimeoutForMobile();
                    if (timeout >= 0) {
                        driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(timeout));
                    }

                    if (isIOSDriverType(driverType)) {
                        driver.setSetting(XCUITestSettingConstants.RESPECT_SYSTEM_ALERTS, true);
                    }

                    responseCapabilities = driver.getCapabilities();
                    return driver;
                } catch (UnreachableBrowserException e) {
                    long newMilis = System.currentTimeMillis();
                    time += ((newMilis - currentMilis) / 1000);
                    currentMilis = newMilis;
                    continue;
                }
            }
            throw new MobileDriverInitializeException(
                    MessageFormat.format(AppiumStringConstants.CANNOT_CONNECT_TO_APPIUM_AFTER_X, timeOut));
        } catch (W3CCapabilityViolationException e) {
            logger.logMessage(LogLevel.WARNING, e.getMessage(), e);
            throw new MobileDriverInitializeException(e);
        } finally {
            logMobileRunData(capabilities, Optional.ofNullable(responseCapabilities));
        }
    }

    /**
     * @return the remote web driver url if running Mobile keyword on cloud services
     */
    public static String getRemoteWebDriverServerUrl() {
        String remoteServerUrl = RunConfiguration.getDriverSystemProperty(RunConfiguration.REMOTE_DRIVER_PROPERTY,
                REMOTE_WEB_DRIVER_URL);
        if (RunConfiguration.getDriverSystemProperties(RunConfiguration.REMOTE_DRIVER_PROPERTY) != null) {
            boolean isEncrypted = RunConfiguration.getBooleanProperty(IS_REMOTE_WEB_DRIVER_URL_ENCRYPTED,
                    RunConfiguration.getDriverSystemProperties(RunConfiguration.REMOTE_DRIVER_PROPERTY));
            if (isEncrypted) {
                try {
                    remoteServerUrl = CryptoUtil.decode(CryptoUtil.getDefault(remoteServerUrl));
                } catch (GeneralSecurityException | IOException e) {}
            }
        }
        return remoteServerUrl;
    }

    private static void logMobileRunData(MutableCapabilities desiredCapabilities,
            Optional<Capabilities> responseCapabilitiesOpt) {
        if (logger != null) {
            String remoteUrl = getRemoteWebDriverServerUrl();
            Map<String, Object> desiredCapMapForDisplay = new HashMap<String, Object>(desiredCapabilities.toJson());
            if (StringUtils.isNotEmpty(remoteUrl)) {
                if (TestCloudPropertyUtil.getInstance().isRunFromTestCloud(desiredCapabilities)) {
                    String log = JsonUtil.toJson(TestCloudPropertyUtil.getInstance()
                            .hideTestCloudSensitiveDesiredCapabilities(desiredCapMapForDisplay));
                    log = TestCloudPropertyUtil.getInstance().hideSauceLabsDocumentLink(log);
                    logger.logRunData("desiredCapabilities", log);
                } else {
                    logger.logRunData("remoteDriverUrl", remoteUrl);
                    logger.logRunData("desiredCapabilities", JsonUtil.toJson(desiredCapMapForDisplay));
                }
                responseCapabilitiesOpt.ifPresent(responseCapabilities -> {
                    if (TestCloudPropertyUtil.getInstance().isRunFromTestCloud(responseCapabilities)) {
                        Map<String, Object> responseCapabilitiesMap = new HashMap<>(responseCapabilities.asMap());
                        Object katalonCaps = responseCapabilitiesMap.get(TestCloudPropertyUtil.KATALON_CAPS);
                        if (katalonCaps != null && katalonCaps instanceof Map<?, ?>) {
                            String katalonCapsJson = JsonUtil.toJson(katalonCaps);
                            logger.logRunData(TestCloudPropertyUtil.KATALON_CAPS, katalonCapsJson);
                            Map<?, ?> katalonCapsMap = (Map<?, ?>) katalonCaps;
                            if (katalonCapsMap.containsKey(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY)) {
                                String deviceName = (String) katalonCapsMap
                                        .get(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY);
                                logger.logRunData(StringConstants.XML_LOG_DEVICE_NAME_PROPERTY, deviceName);
                            }
                        }
                    }
                });
            } else {
                logger.logRunData(EXECUTED_DEVICE_ID, getDeviceId(StringConstants.CONF_PROPERTY_MOBILE_DRIVER));
                logger.logRunData(EXECUTED_DEVICE_NAME, getDeviceName(StringConstants.CONF_PROPERTY_MOBILE_DRIVER));
                logger.logRunData(EXECUTED_DEVICE_MODEL, getDeviceModel(StringConstants.CONF_PROPERTY_MOBILE_DRIVER));
                logger.logRunData(EXECUTED_DEVICE_MANUFACTURER,
                        getDeviceManufacturer(StringConstants.CONF_PROPERTY_MOBILE_DRIVER));
                logger.logRunData(EXECUTED_DEVICE_OS, getDeviceOS(StringConstants.CONF_PROPERTY_MOBILE_DRIVER));
                logger.logRunData(EXECUTED_DEVICE_OS_VERSON,
                        getDeviceOSVersion(StringConstants.CONF_PROPERTY_MOBILE_DRIVER));
            }
        }
    }

    public static void closeDriver() {
        AppiumDriver webDriver = localStorageAppiumDriver.get();
        if (null != webDriver && null != ((RemoteWebDriver) webDriver).getSessionId()) {
            webDriver.quit();
        }
        RunConfiguration.removeDriver(webDriver);
        localStorageAppiumDriver.set(null);
        quitServer();
    }

    public static void quitServer() {
        if (localStorageAppiumServer.get() != null && localStorageAppiumServer.get().isAlive()) {
            try {
                ProcessUtil.terminateProcess(localStorageAppiumServer.get());
            } catch (Exception e) {
                logger.logWarning("Error when trying to stop Appium Server: " + e.getMessage(), null, e);
            } finally {
                localStorageAppiumServer.set(null);
            }
        }
        if (localStorageWebProxyProcess.get() != null) {
            try {
                ProcessUtil.terminateProcess(localStorageWebProxyProcess.get());
            } catch (Exception e) {
                logger.logWarning("Error when trying to stop Web Proxy Server: " + e.getMessage(), null, e);
            } finally {
                localStorageWebProxyProcess.set(null);
            }
        }
    }

    public static AppiumDriver getDriver() throws StepFailedException {
        verifyWebDriverIsOpen();
        return localStorageAppiumDriver.get();
    }

    /**
     * Sets the current active Appium driver.
     *
     * @param driver the Appium driver to be set
     * @see AppiumDriver
     * @since 7.6.0
     */
    public static void setDriver(AppiumDriver driver) {
        localStorageAppiumDriver.set(driver);
    }

    public static Process getAppiumSeverProcess() {
        return localStorageAppiumServer.get();
    }

    public static Process getIosWebKitProcess() {
        return localStorageWebProxyProcess.get();
    }

    private static void verifyWebDriverIsOpen() throws StepFailedException {
        if (localStorageAppiumDriver.get() == null) {
            throw new StepFailedException("No application is started yet.");
        }
    }

    public static void cleanup() throws InterruptedException, IOException {
        String os = System.getProperty("os.name");
        if (os.toLowerCase().contains("win")) {
            killProcessOnWin("adb.exe");
            killProcessOnWin("node.exe");
        } else {
            killProcessOnMac("adb");
            killProcessOnMac(NODE_EXECUTABLE);
            killProcessOnMac("instruments");
            killProcessOnMac("deviceconsole");
            killProcessOnMac("ios_webkit_debug_proxy");
        }
    }

    private static void killProcessOnWin(String processName) throws InterruptedException, IOException {
        ProcessBuilder pb = new ProcessBuilder("taskkill", "/f", "/im", processName, "/t");
        pb.start().waitFor();
    }

    private static void killProcessOnMac(String processName) throws InterruptedException, IOException {
        ProcessBuilder pb = new ProcessBuilder("killall", processName);
        pb.start().waitFor();
    }

    public static String getDeviceId(String parentProperty) {
        return RunConfiguration.getDriverSystemProperty(parentProperty, EXECUTED_DEVICE_ID);
    }

    public static String getDeviceName(String parentProperty) {
        return RunConfiguration.getDriverSystemProperty(parentProperty, EXECUTED_DEVICE_NAME);
    }

    public static String getDeviceModel(String parentProperty) {
        return RunConfiguration.getDriverSystemProperty(parentProperty, EXECUTED_DEVICE_MODEL);
    }

    public static String getDeviceManufacturer(String parentProperty) {
        return RunConfiguration.getDriverSystemProperty(parentProperty, EXECUTED_DEVICE_MANUFACTURER);
    }

    public static String getDeviceOSVersion(String parentProperty) {
        return RunConfiguration.getDriverSystemProperty(parentProperty, EXECUTED_DEVICE_OS_VERSON);
    }

    public static String getDeviceOS(String parentProperty) {
        return RunConfiguration.getDriverSystemProperty(parentProperty, EXECUTED_DEVICE_OS);
    }

    public static int getXCodeVersion() throws ExecutionException {
        try {
            String xcodeVersion = ConsoleCommandExecutor.runConsoleCommandAndCollectFirstResult(
                    new String[] { "xcodebuild", "-version", "|", "grep", "\"" + XCODE + ".*\"" });
            String majorVersionString = xcodeVersion.substring(XCODE.length(), xcodeVersion.indexOf(".")).trim();
            return Integer.parseInt(majorVersionString);
        } catch (IOException | InterruptedException | NumberFormatException | StringIndexOutOfBoundsException e) {
            throw new ExecutionException("Unable to find xcode version", e);
        }
    }
}
