package com.kms.katalon.core.mobile.keyword.builtin

import java.text.MessageFormat
import java.time.Duration

import org.openqa.selenium.TimeoutException
import org.openqa.selenium.WebElement
import org.openqa.selenium.support.ui.FluentWait

import com.google.common.base.Function
import com.kms.katalon.core.annotation.internal.Action
import com.kms.katalon.core.configuration.RunConfiguration
import com.kms.katalon.core.exception.StepFailedException
import com.kms.katalon.core.helper.KeywordHelper
import com.kms.katalon.core.keyword.internal.SupportLevel
import com.kms.katalon.core.mobile.constants.StringConstants
import com.kms.katalon.core.mobile.helper.MobileCommonHelper
import com.kms.katalon.core.mobile.keyword.internal.MobileAbstractKeyword
import com.kms.katalon.core.mobile.keyword.internal.MobileKeywordMain
import com.kms.katalon.core.model.FailureHandling
import com.kms.katalon.core.testobject.TestObject

import groovy.transform.CompileStatic

@Action(value = "waitForElementNotPresent")
public class WaitForElementNotPresentKeyword extends MobileAbstractKeyword {

    @CompileStatic
    @Override
    public SupportLevel getSupportLevel(Object ...params) {
        return super.getSupportLevel(params)
    }

    @CompileStatic
    @Override
    public Object execute(Object ...params) {
        TestObject to = getTestObject(params[0])
        int timeout = (int) params[1]
        FailureHandling flowControl = (FailureHandling)(params.length > 2 && params[2] instanceof FailureHandling ? params[2] : RunConfiguration.getDefaultFailureHandling())
        return waitForElementNotPresent(to, timeout, flowControl);
    }

    @CompileStatic
    public boolean waitForElementNotPresent(TestObject to, int timeout, FailureHandling flowControl) throws StepFailedException {
        return MobileKeywordMain.runKeyword({
            KeywordHelper.checkTestObjectParameter(to)
            timeout = MobileCommonHelper.checkTimeout(timeout)
            
            boolean elementNotFound = false
            try {
                // In theory, it should take 1.5 seconds each time it finds the element
                // (1s to search and 500ms to sleep). But in reality, finding the element can take longer than 1s
                // because the core implementation of `findElement` method (`findElementsByLocatorStrategy` method in MobileCommonHelper class)
                // only checks the timeout after finding
                elementNotFound = new FluentWait<TestObject>(to)
                    .pollingEvery(Duration.ofMillis(500))
                    .withTimeout(Duration.ofSeconds(timeout))
                    .until(new Function<TestObject, Boolean>() {
                        @Override
                        public Boolean apply(TestObject testObj) {
                            try {
                                // Set timeout to 1 second to ensure that the `findElementsByLocatorStrategy` method in MobileCommonHelper class
                                // runs at least once with the minimum timeout.
                                // Note: Do not set the timeout to 0, as the `checkTimeout` method in the `KeywordHelper` class
                                // will reset it to the default value (30 seconds) if the timeout is less than or equal to zero.
                                WebElement element = findElement(testObj, 1)
                                if (element == null) {
                                    return true
                                }
                            } catch (NoSuchElementException e) {
                                return true
                            }
                            
                            return false
                        }
                    })
            } catch (TimeoutException e) {
                // timeOut, do nothing
            }
            
            if (elementNotFound) {
                logger.logPassed(MessageFormat.format(StringConstants.KW_LOG_PASSED_ELEMENT_NOT_PRESENTED, to.getObjectId()))
            } else {
                logger.logWarning(MessageFormat.format(StringConstants.KW_LOG_PASSED_ELEMENT_PRESENTED, to.getObjectId()))
            }
            
            return elementNotFound
        }, flowControl, RunConfiguration.getTakeScreenshotOption(), to != null ? MessageFormat.format(StringConstants.KW_MSG_FAILED_TO_WAIT_FOR_ELEMENT_X_NOT_PRESENT, to.getObjectId()) : StringConstants.KW_MSG_FAILED_TO_WAIT_FOR_ELEMENT_NOT_PRESENT)
    }
}
