package com.kms.katalon.core.webservice.helper;

import java.io.File;
import java.io.IOException;
import java.io.StringReader;
import java.io.StringWriter;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import javax.wsdl.Definition;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLReader;
import javax.xml.XMLConstants;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Source;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.transform.stream.StreamSource;
import javax.xml.validation.Schema;
import javax.xml.validation.SchemaFactory;
import javax.xml.validation.Validator;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import com.atlassian.oai.validator.report.ValidationReport;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonMappingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.kms.katalon.core.testobject.HttpBodyContent;
import com.kms.katalon.core.testobject.RequestObject;
import com.kms.katalon.core.testobject.ResponseObject;
import com.kms.katalon.core.testobject.ValidationResult;
import com.kms.katalon.core.testobject.ValidationStatus;
import com.kms.katalon.core.testobject.ValidationStep;
import com.kms.katalon.core.testobject.ValidationTarget;
import com.kms.katalon.core.testobject.ValidationType;
import com.kms.katalon.core.testobject.impl.HttpGraphQLBodyContent;
import com.kms.katalon.core.testobject.impl.HttpTextBodyContent;
import com.kms.katalon.core.util.internal.GraphQLUtil;
import com.kms.katalon.core.util.internal.JsonUtil;
import com.kms.katalon.core.util.internal.XMLUtil;
import com.kms.katalon.core.webservice.common.ResourceResolver;
import com.kms.katalon.core.webservice.constants.CoreWebserviceMessageConstants;
import com.kms.katalon.core.webservice.helper.WebServiceCommonHelper.LocalFileProvider;
import com.kms.katalon.core.webservice.helper.WebServiceCommonHelper.RemoteFileProvider;
import com.kms.katalon.core.webservice.util.OpenAPIUtil;
import com.kms.katalon.core.webservice.util.WebServiceCommonUtil;
import com.kms.katalon.util.DocumentBuilderProvider;
import com.kms.katalon.util.TransformerFactoryProvider;
import com.networknt.schema.JsonMetaSchema;
import com.networknt.schema.JsonSchema;
import com.networknt.schema.JsonSchemaFactory;
import com.networknt.schema.SchemaValidatorsConfig;
import com.networknt.schema.SpecVersion.VersionFlag;
import com.networknt.schema.SpecVersionDetector;
import com.networknt.schema.ValidationMessage;

import graphql.parser.Parser;
import graphql.schema.GraphQLSchema;
import graphql.schema.idl.SchemaParser;
import graphql.schema.idl.TypeDefinitionRegistry;
import graphql.schema.idl.UnExecutableSchemaGenerator;
import graphql.validation.ValidationError;

public class WebServiceValidationHelper {

    private WebServiceValidationHelper() {
        // Hide the default constructor
    }

    public static List<ValidationResult> getProblems(ValidationStep step) {
        return step.results.stream().filter(resultI -> isFailedResult(resultI)).collect(Collectors.toList());
    }

    public static boolean isFailedResult(ValidationResult result) {
        List<ValidationStatus> statusList = Arrays.asList(ValidationStatus.ERROR, ValidationStatus.FAIL);
        return statusList.contains(result.status);
    }

    public static boolean isFailedStep(ValidationStep step) {
        if (step.results == null) {
            return false;
        }
        return testAnyStatus(step, ValidationStatus.ERROR, ValidationStatus.FAIL);
    }

    public static boolean isSkippedStep(ValidationStep step) {
        if (step.results == null) {
            return false;
        }
        return testAllStatus(step, ValidationStatus.SKIP);
    }

    public static boolean isPassedStep(ValidationStep step) {
        if (step.results == null) {
            return false;
        }
        return testAllStatus(step, ValidationStatus.PASS);
    }

    public static boolean testAllStatus(ValidationStep step, ValidationStatus... status) {
        List<ValidationStatus> statusList = Arrays.asList(status);
        return step.results.stream().allMatch(resultI -> statusList.contains(resultI.status));
    }

    public static boolean testAnyStatus(ValidationStep step, ValidationStatus... status) {
        List<ValidationStatus> statusList = Arrays.asList(status);
        return step.results.stream().anyMatch(resultI -> statusList.contains(resultI.status));
    }

    public static void executeValidationSteps(RequestObject requestObject, ResponseObject responseObject,
            String projectLocation) throws Exception {
        executeValidationSteps(requestObject, responseObject, null, null, projectLocation);
    }

    public static void executeValidationSteps(RequestObject request, ResponseObject response,
            LocalFileProvider localFileProvider, RemoteFileProvider remoteFileProvider, String projectLocation)
            throws Exception {
        List<ValidationStep> steps = request.getValidationSteps();
        if (steps == null) {
            return;
        }
        for (ValidationStep stepI : steps) {
            try {
                stepI.results = new ArrayList<ValidationResult>();
                if (!stepI.activate) {
                    stepI.results.add(new ValidationResult("", ValidationStatus.SKIP));
                    continue;
                }
                if (stepI.target == null) {
                    stepI.target = ValidationTarget.RESPONSE;
                }
                boolean isInvalidStep = StringUtils.isBlank(stepI.data) || stepI.target == null;
                if (isInvalidStep) {
                    stepI.results.add(new ValidationResult(
                            CoreWebserviceMessageConstants.KW_LOG_VALIDATION_LOCATION_OR_VALUE_IS_BLANK,
                            ValidationStatus.ERROR));
                    continue;
                }
                boolean isSOAP = response.isSOAPResponse();
                List<String> validationTargets = getValidationTarget(request, response, stepI.target);
                String schemaContent = WebServiceCommonHelper.smartGetData(stepI.data.trim(), stepI.dataType,
                        localFileProvider, remoteFileProvider);

                ValidationType validationType = stepI.type;
                if (validationType == null || validationType == ValidationType.AUTO_DETECT) {
                    if (JsonUtil.isValidJsonSchema(schemaContent)) {
                        validationType = ValidationType.JSON_SCHEMA;
                    } else if (XMLUtil.isValidXml(schemaContent)) {
                        validationType = ValidationType.XML_SCHEMA;
                    } else if (GraphQLUtil.isValidGraphQLSchema(schemaContent)) {
                        validationType = ValidationType.GRAPHQL_SCHEMA;
                    } else if (OpenAPIUtil.isValidOpenAPISpecification(stepI.data.trim(), stepI.dataType, projectLocation)) {
                        validationType = ValidationType.OPEN_API_SPEC;
                    }
                }
                if (validationType == null || validationType == ValidationType.AUTO_DETECT) {
                    stepI.results.add(
                            new ValidationResult(CoreWebserviceMessageConstants.KW_LOG_VALIDATION_CANNOT_DETECT_CHEMA,
                                    ValidationStatus.ERROR));
                }
                if ((validationTargets == null || validationTargets.isEmpty())
                        && validationType != ValidationType.OPEN_API_SPEC) {
                    // In case the validation type is OPEN_API_SPEC, validationTargets is null
                    stepI.results.add(
                            new ValidationResult(CoreWebserviceMessageConstants.KW_LOG_VALIDATION_CANNOT_FIND_TARGET,
                                    ValidationStatus.ERROR));
                }
                // validate for request swagger only
                if (validationType == ValidationType.OPEN_API_SPEC && stepI.target == ValidationTarget.REQUEST) {
                    String specSrc = OpenAPIUtil.getFullSpecLocation(stepI.dataType, stepI.data.trim(),
                            projectLocation);
                    ValidationReport report = OpenAPIUtil.validateRequest(request, specSrc);

                    if (report.hasErrors()) {
                        List<ValidationResult> results = report.getMessages().stream().map((err) -> {
                            return new ValidationResult(err.getMessage(), ValidationStatus.FAIL);
                        }).collect(Collectors.toList());
                        stepI.results.addAll(results);
                    } else {
                        stepI.results.add(new ValidationResult(ValidationStatus.PASS));
                    }
                } else {
                    for (String validationTargetI : validationTargets) {
                        if (validationType == ValidationType.JSON_SCHEMA) {
                            Set<ValidationMessage> errors = validateJsonSchema(validationTargetI, schemaContent);

                            if (errors == null || errors.isEmpty()) {
                                stepI.results.add(new ValidationResult(ValidationStatus.PASS));
                            } else {
                                List<ValidationResult> results = errors.stream().<ValidationResult> map((errorI) -> {
                                    return new ValidationResult(errorI.getMessage(), ValidationStatus.FAIL);
                                }).collect(Collectors.toList());
                                stepI.results.addAll(results);
                            }
                        } else if (validationType == ValidationType.XML_SCHEMA) {
                            ResourceResolver resourceResolver = new ResourceResolver(localFileProvider,
                                    remoteFileProvider);
                            String errorString = null;
                            if (!isSOAP && isWSDLSchema(schemaContent)) {
                                errorString = CoreWebserviceMessageConstants.KW_LOG_VALIDATION_CANNOT_VALIDATE_XML_AGAINST_WSDL;
                            } else if (isWSDLSchema(schemaContent)) {
                                errorString = validateWSDLSchema(validationTargetI, schemaContent);
                            } else {
                                errorString = validateXmlSchema(validationTargetI, schemaContent, resourceResolver);
                            }

                            if (StringUtils.isBlank(errorString)) {
                                stepI.results.add(new ValidationResult(ValidationStatus.PASS));
                            } else {
                                errorString = WebServiceCommonUtil.getXmlValidationErrorString(errorString);
                                errorString = StringUtils.trim(errorString);
                                List<String> errorList = Arrays.asList(errorString.split("\\. "));
                                List<ValidationResult> xmlResults = errorList.stream().<ValidationResult> map((str) -> {
                                    return new ValidationResult(str, ValidationStatus.FAIL);
                                }).collect(Collectors.toList());
                                stepI.results.addAll(xmlResults);
                            }
                        } else if (validationType == ValidationType.GRAPHQL_SCHEMA) {
                            if (stepI.target == ValidationTarget.REQUEST) {
                                List<ValidationError> errors = WebServiceValidationHelper
                                        .validateGraphQLBodyWithSchema(validationTargetI, schemaContent);
                                if (errors == null) {
                                    stepI.results.add(new ValidationResult(
                                            CoreWebserviceMessageConstants.KW_LOG_VALIDATION_INVALID_GRAPHQL_SCHEMA,
                                            ValidationStatus.FAIL));
                                } else if (errors.isEmpty()) {
                                    stepI.results.add(new ValidationResult(ValidationStatus.PASS));
                                } else {
                                    List<ValidationResult> graphQLResults = new ArrayList<ValidationResult>();
                                    for (ValidationError error : errors) {
                                        graphQLResults
                                                .add(new ValidationResult(error.getMessage(), ValidationStatus.FAIL));
                                    }
                                    stepI.results.addAll(graphQLResults);
                                }
                            }
                        } else if (validationType == ValidationType.OPEN_API_SPEC) {
                            if (stepI.target == ValidationTarget.RESPONSE) {
                                String specSrc = OpenAPIUtil.getFullSpecLocation(stepI.dataType, stepI.data.trim(),
                                        projectLocation);
                                ValidationReport report = OpenAPIUtil.validateResponse(response, request, specSrc);

                                if (report.hasErrors()) {
                                    List<ValidationResult> results = report.getMessages().stream().map((err) -> {
                                        return new ValidationResult(err.getMessage(), ValidationStatus.FAIL);
                                    }).collect(Collectors.toList());
                                    stepI.results.addAll(results);
                                } else {
                                    stepI.results.add(new ValidationResult(ValidationStatus.PASS));
                                }
                            }

                        } else {
                            stepI.results.add(new ValidationResult(
                                    CoreWebserviceMessageConstants.KW_LOG_VALIDATION_VALIDATE_GRAPHQL_SCHEMA_FAILED,
                                    ValidationStatus.FAIL));
                        }

                    }
                }
            } catch (Exception error) {
                stepI.results.add(new ValidationResult(error.getMessage(), ValidationStatus.ERROR));
            }
        }
    }

    public static List<String> getValidationTarget(RequestObject request, ResponseObject response,
            ValidationTarget targetType) throws Exception {
        if (targetType == ValidationTarget.RESPONSE) {
            if (response.isSOAPResponse()) {
                return Arrays.asList(response.extractSOAPBodyAsContent());
            }
            return Arrays.asList(response.getResponseText());
        } else {
            if (WebServiceCommonUtil.isSOAPRequest(request)) {
                return Arrays.asList(StringUtils.defaultString(request.getSoapBody()));
            }
            HttpBodyContent requestBody = request.getBodyContent();
            if (requestBody == null) {
                return Collections.emptyList();
            }
            if (requestBody instanceof HttpGraphQLBodyContent) {
                return Arrays
                        .asList(StringUtils.defaultString(((HttpGraphQLBodyContent) requestBody).getDisplayText()));
            } else if (requestBody instanceof HttpTextBodyContent) {
                return Arrays.asList(StringUtils.defaultString(((HttpTextBodyContent) requestBody).getText()));
            }
            return Collections.emptyList();
        }
    }

    public static Set<ValidationMessage> validateJsonSchema(String jsonObject, String rawJsonSchema)
            throws JsonMappingException, JsonProcessingException {
        SchemaValidatorsConfig config = new SchemaValidatorsConfig();
        config.setTypeLoose(false);

        ObjectMapper mapper = new ObjectMapper();
        JsonNode schemaNode = mapper.readTree(rawJsonSchema);
        JsonSchemaFactory factory = JsonSchemaFactory.getInstance(VersionFlag.V201909);
        JsonSchema schema = factory.getSchema(schemaNode, config);
        schema.initializeValidators();

        JsonNode responseNode = mapper.readTree(jsonObject);

        Set<ValidationMessage> errors = schema.validate(responseNode);
        return errors;
    }

    public static String validateXmlSchema(String responseText, String schemaContent) {
        return validateXmlSchema(responseText, schemaContent,
                new ResourceResolver(WebServiceCommonHelper.DEFAULT_LOCAL_FILE_PROVIDER,
                        WebServiceCommonHelper.DEFAULT_REMOTE_FILE_PROVIDER));
    }

    public static String validateXmlSchema(String responseText, String schemaContent,
            ResourceResolver resourceResolver) {
        try {
            SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
            factory.setResourceResolver(resourceResolver);
            Source schemaSource = new StreamSource(new StringReader(schemaContent));
            Schema schema = factory.newSchema(schemaSource);
            Validator validator = schema.newValidator();
            Source source = new StreamSource(new StringReader(responseText));
            validator.validate(source);

            return StringUtils.EMPTY;
        } catch (SAXException e) {
            return e.getMessage();
        } catch (IOException e) {
            return e.getMessage();
        }
    }

    @SuppressWarnings("unchecked")
    public static String validateWSDLSchema(String responseText, String schemaContent) {
        File f = null;
        try {
            WSDLFactory wsdlFactory = WSDLFactory.newInstance();
            WSDLReader wsdlReader = wsdlFactory.newWSDLReader();
            f = File.createTempFile("temp", ".wsdl");
            FileUtils.writeStringToFile(f, schemaContent, StandardCharsets.UTF_8);
            Definition wsdlDefinition = wsdlReader.readWSDL(f.getPath());
            DocumentBuilder docBuilder = DocumentBuilderProvider.newBuilderInstance();
            Document doc = docBuilder.newDocument();

            if (wsdlDefinition.getTypes() != null) {
                for (Object o : wsdlDefinition.getTypes().getExtensibilityElements()) {
                    if (o instanceof javax.wsdl.extensions.schema.Schema) {
                        Element ele = ((javax.wsdl.extensions.schema.Schema) o).getElement();
                        Node newNode = doc.importNode(ele, true);
                        doc.appendChild(newNode);
                    }
                }
            }
            Map<String, String> ns = wsdlDefinition.getNamespaces();
            if (ns != null) {
                for (Entry<String, String> e : ns.entrySet()) {
                    if (e.getKey() != null && !e.getKey().isEmpty()) {
                        doc.getDocumentElement()
                                .setAttributeNS(XMLConstants.XMLNS_ATTRIBUTE_NS_URI, "xmlns:" + e.getKey(),
                                        e.getValue());
                    }
                }
            }
            StringWriter writer = new StringWriter();
            TransformerFactory tf = TransformerFactoryProvider.newInstance();
            Transformer t = tf.newTransformer();
            t.setOutputProperty(OutputKeys.INDENT, "yes");
            t.transform(new DOMSource(doc), new StreamResult(writer));

            SchemaFactory factory = SchemaFactory.newInstance(XMLConstants.W3C_XML_SCHEMA_NS_URI);
            Schema schema = factory.newSchema(new StreamSource(new StringReader(writer.toString())));

            Validator validator = schema.newValidator();
            Source source = new StreamSource(new StringReader(responseText));
            validator.validate(source);
            return StringUtils.EMPTY;
        } catch (Exception e) {
            return e.getMessage();
        } finally {
            if (f != null) {
                FileUtils.deleteQuietly(f);
            }
        }
    }

    public static boolean isWSDLSchema(String schema) {
        Pattern wsdlPattern = Pattern.compile(":definitions>");
        Matcher wsdlMatcher = wsdlPattern.matcher(schema);
        return wsdlMatcher.find();
    }

    public static List<graphql.validation.ValidationError> validateGraphQLBodyWithSchema(String GraphQLQuery,
            String schema) {
        graphql.validation.Validator validator = new graphql.validation.Validator();
        TypeDefinitionRegistry definitionRegistry = new SchemaParser().parse(schema);
        GraphQLSchema qlSchema = UnExecutableSchemaGenerator.makeUnExecutableSchema(definitionRegistry);
        graphql.language.Document doc = Parser.parse(GraphQLQuery);
        List<graphql.validation.ValidationError> errors = validator.validateDocument(qlSchema, doc, Locale.getDefault());
        return errors;
    }
}
