package com.kms.katalon.core.testobject.impl;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;

import org.apache.commons.lang.StringUtils;

import com.kms.katalon.core.testobject.HttpBodyContent;

/**
 * Represents the body content of a HTTP message (request/response) that contains binary data.
 * 
 * @since 10.3.0
 */
public class HttpBinaryBodyContent implements HttpBodyContent {

    private static final String DF_CONTENT_TYPE = "application/octet-stream";

    private final byte[] binaryData;
    private final String contentType;
    private final String contentEncoding;

    /**
     * Creates an instance with the given binary data and default content type.
     * 
     * @param binaryData The raw binary content.
     */
    public HttpBinaryBodyContent(byte[] binaryData) {
        this(binaryData, DF_CONTENT_TYPE);
    }

    /**
     * Creates an instance with the given binary data and content type.
     * 
     * @param binaryData The raw binary content.
     * @param contentType The content type to be used. If null or empty, the default content type is used.
     */
    public HttpBinaryBodyContent(byte[] binaryData, String contentType) {
        this(binaryData, contentType, null);
    }

    /**
     * Creates an instance with the given binary data, content type, and content encoding.
     * 
     * @param binaryData The raw binary content.
     * @param contentType The content type to be used. If null or empty, the default content type is used.
     * @param contentEncoding The content encoding to be used.
     */
    public HttpBinaryBodyContent(byte[] binaryData, String contentType, String contentEncoding) {
        this.binaryData = binaryData != null ? binaryData : new byte[0];
        this.contentType = contentType;
        this.contentEncoding = contentEncoding;
    }

    @Override
    public String getContentType() {
        if (StringUtils.isEmpty(contentType)) {
            return DF_CONTENT_TYPE;
        }
        return contentType;
    }

    @Override
    public long getContentLength() {
        return binaryData.length;
    }

    @Override
    public String getContentEncoding() {
        return contentEncoding;
    }

    @Override
    public InputStream getInputStream() throws IOException, UnsupportedOperationException {
        return new ByteArrayInputStream(binaryData);
    }

    @Override
    public void writeTo(OutputStream outstream) throws IOException {
        outstream.write(binaryData);
        outstream.flush();
    }

    /**
     * Gets the binary data.
     * 
     * @return The binary data.
     */
    public byte[] getBinaryData() {
        return binaryData;
    }
}
