package com.kms.katalon.core.mobile.keyword.builtin

import java.nio.file.InvalidPathException
import java.text.MessageFormat
import java.time.LocalDateTime

import org.apache.commons.lang3.StringUtils

import com.google.common.base.Preconditions
import com.google.gson.Gson
import com.google.gson.GsonBuilder
import com.kms.katalon.core.annotation.internal.Action
import com.kms.katalon.core.configuration.RunConfiguration
import com.kms.katalon.core.exception.StepFailedException
import com.kms.katalon.core.json.adapters.LocalDateTimeAdapter
import com.kms.katalon.core.keyword.internal.SupportLevel
import com.kms.katalon.core.mobile.constants.CoreMobileMessageConstants
import com.kms.katalon.core.mobile.constants.StringConstants
import com.kms.katalon.core.mobile.keyword.internal.MobileAbstractKeyword
import com.kms.katalon.core.mobile.keyword.internal.MobileDriverFactory
import com.kms.katalon.core.mobile.keyword.internal.MobileKeywordMain
import com.kms.katalon.core.model.FailureHandling
import com.kms.katalon.core.model.TakeScreenshotOption
import com.kms.katalon.core.util.TestCloudPropertyUtil
import com.kms.katalon.core.util.internal.PathUtil
import com.kms.katalon.testcloud.core.constants.TestCloudStringConstants
import com.kms.katalon.testcloud.core.model.TestCloudApplication
import com.kms.katalon.util.NameUtil

import groovy.transform.CompileStatic


@Action(value = "startApplication")
public class StartApplicationKeyword extends MobileAbstractKeyword {

    @CompileStatic
    @Override
    public SupportLevel getSupportLevel(Object ...params) {
        return super.getSupportLevel(params)
    }

    @CompileStatic
    @Override
    public Object execute(Object ...params) {
        String appFile = (String) params[0]
        boolean uninstallAfterCloseApp = (boolean) params[1]
        FailureHandling flowControl = (FailureHandling)(params.length > 2 && params[2] instanceof FailureHandling ? params[2] : RunConfiguration.getDefaultFailureHandling())
        startApplication(appFile,uninstallAfterCloseApp,flowControl)
    }

    @CompileStatic
    public void startApplication(String appFile, boolean uninstallAfterCloseApp, FailureHandling flowControl) throws StepFailedException {
        String appFileDisplayName = appFile
        TestCloudPropertyUtil tcInstance = TestCloudPropertyUtil.getInstance();
        if (tcInstance.isRunFromTestCloud() && tcInstance.containsSensitiveName(appFileDisplayName)) {
            appFileDisplayName = NameUtil.censor(appFileDisplayName, -1)
        }

        MobileKeywordMain.runKeyword({
            Preconditions.checkArgument(StringUtils.isNotEmpty(appFile), CoreMobileMessageConstants.KW_MSG_APP_FILE_MISSING);
            String applicationFileAbs = getAbsolutePath(appFile)
            String applicationFileAbsDisplayLog = applicationFileAbs;
            if (tcInstance.isRunFromTestCloud() && tcInstance.containsSensitiveName(applicationFileAbs)) {
                applicationFileAbsDisplayLog = NameUtil.censor(applicationFileAbs, -1);
            }

            Map<String, Object> remoteWebDriverPreferences = MobileDriverFactory.getRemoteWebDriverPreferences();
            if (Objects.nonNull(remoteWebDriverPreferences)) {
                String applicationJson = (String) remoteWebDriverPreferences.get(TestCloudStringConstants.TESTCLOUD_OVERRIDE_MOBILE_APPLICATION);
                if (StringUtils.isNoneBlank(applicationJson)) {
                    Gson gson = new GsonBuilder().registerTypeAdapter(LocalDateTime.class, new LocalDateTimeAdapter()).create();
                    TestCloudApplication testCloudApplication = gson.fromJson(applicationJson, TestCloudApplication.class);
                    applicationFileAbs = testCloudApplication.getId();
                    String oldApplicationFileAbsDisplayLog = applicationFileAbsDisplayLog;
                    applicationFileAbsDisplayLog = applicationFileAbs;

                    if (tcInstance.isRunFromTestCloud() && tcInstance.containsSensitiveName(applicationFileAbs)) {
                        applicationFileAbsDisplayLog = NameUtil.censor(applicationFileAbs, -1);
                    }

                    logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_INFO_OVERRIDING_APP,
                            StringUtils.defaultString(oldApplicationFileAbsDisplayLog),
                            StringUtils.defaultString(applicationFileAbsDisplayLog)));
                    logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_INFO_OVERRIDE_APPLICATION, applicationJson));
                }
            }

            logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_INFO_STARTING_APP_AT, StringUtils.defaultString(applicationFileAbsDisplayLog)))
            MobileDriverFactory.startMobileDriver(applicationFileAbs, uninstallAfterCloseApp)
            logger.logPassed(MessageFormat.format(StringConstants.KW_LOG_PASSED_START_APP_AT,  StringUtils.defaultString(applicationFileAbsDisplayLog)))
        }, flowControl, TakeScreenshotOption.NONE, MessageFormat.format(StringConstants.KW_MSG_UNABLE_TO_START_APP_AT, StringUtils.defaultString(appFileDisplayName)))
    }

    private String getAbsolutePath(String filePath) {
        try {
            String absFilePath = PathUtil.relativeToAbsolutePath(filePath, RunConfiguration.getProjectDir())
            File absFile = new File(absFilePath)
            return absFile.isFile() && absFile.exists() ? absFilePath : filePath
        } catch (InvalidPathException e) {
            return filePath
        }
    }
}
