package com.kms.katalon.core.windows.driver;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.text.MessageFormat;
import java.text.ParseException;
import java.time.Duration;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.NoSuchWindowException;
import org.openqa.selenium.SessionNotCreatedException;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.http.HttpClient.Factory;

import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.network.ProxyInformation;
import com.kms.katalon.core.selenium.remote.http.ConfiguredHttpClientFactory;
import com.kms.katalon.core.util.internal.JsonUtil;
import com.kms.katalon.core.windows.keyword.helper.WindowsActionHelper;
import com.kms.katalon.core.windows.keyword.helper.WindowsShortcut;
import com.kms.katalon.core.windows.model.StringMatchingStrategy;

import io.appium.java_client.MobileCommand;
import io.appium.java_client.remote.AppiumCommandExecutor;
import io.appium.java_client.windows.WindowsDriver;

public class WindowsDriverFactory {

    private static final int DEFAULT_CONNECT_TIMEOUT_IN_SECONDS = 120;

    private static final int DEFAULT_READ_TIMEOUT_IN_SECONDS = 10800;

    private static final int DEFAULT_WRITE_TIMEOUT_IN_SECONDS = 0;

    private static KeywordLogger logger = KeywordLogger.getInstance(WindowsDriverFactory.class);

    public static final String DESIRED_CAPABILITIES_PROPERTY = "desiredCapabilities";

    /**
     * @deprecated use `FLAUI_DRIVER_PROPERTY` instead
     * Still keep this property for backward compatibility with KS version 9
     */
    @Deprecated
    public static final String WIN_APP_DRIVER_PROPERTY = "winAppDriverUrl";

    public static final String FLAUI_DRIVER_PROPERTY = "desktopDriverUrl";

    private static WindowsSession windowsSession;

    public static WindowsDriver getWindowsDriver() {
        return windowsSession.getRunningDriver();
    }

    public static WindowsSession getWindowsSession() {
        return windowsSession;
    }

    public static WindowsDriver startApplication(String appFile, String appTitle)
            throws IOException, URISyntaxException, InterruptedException {
        return startApplication(appFile, appTitle, null);
    }

    @SuppressWarnings("unchecked")
    public static WindowsDriver startApplication(String appFile, String appTitle, StringMatchingStrategy strategy)
            throws IOException, URISyntaxException, InterruptedException {
        ProxyInformation proxyInfo = RunConfiguration.getProxyInformation();
        // ProxyConfig proxyConfig = proxyInfo == null ? null : proxyInfo.toNewProxyConfigModel();

        Map<String, Object> userConfigProperties = RunConfiguration.getDriverPreferencesProperties("Windows");
        if (userConfigProperties == null) {
            userConfigProperties = new HashMap<String, Object>();
        }

        String driverUrlString = (String) userConfigProperties.get(FLAUI_DRIVER_PROPERTY);
        URL driverUrl = new URL(driverUrlString);

        logger.logInfo(
                String.format("Starting application %s on machine at address %s", appFile, driverUrl.toString()));
        logger.logRunData(FLAUI_DRIVER_PROPERTY, driverUrl.toString());

        var capsBuilder = new WindowsDesiredCapsBuilder();
        Object desiredCapabilitiesAsObject = userConfigProperties.getOrDefault(DESIRED_CAPABILITIES_PROPERTY, null);

        if (desiredCapabilitiesAsObject instanceof Map) {
            capsBuilder.merge((Map) desiredCapabilitiesAsObject);
        }

        DesiredCapabilities desiredCapabilities = capsBuilder.build();

        logger.logRunData(DESIRED_CAPABILITIES_PROPERTY, JsonUtil.toJson(desiredCapabilities.toJson(), false));

        WindowsDriver windowsDriver = startApplication(driverUrl, appFile, desiredCapabilities, proxyInfo, appTitle,
                strategy).getRunningDriver();

        windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(RunConfiguration.getElementTimeoutForWindows()));

        return windowsDriver;

    }

    public static WindowsSession startApplication(URL remoteAddressURL, String appFile,
            DesiredCapabilities initCapabilities, ProxyInformation proxyInfo, String appTitle,
            StringMatchingStrategy strategy) throws IOException, URISyntaxException, InterruptedException {
        var capsBuilder = new WindowsDesiredCapsBuilder();

        // To handle "window shortcut" appFile
        var finalAppFile = appFile;

        try {
            if (WindowsShortcut.isShortcutFilePath(appFile)) {
                try {
                    WindowsShortcut shortcut = WindowsShortcut.from(appFile);
                    finalAppFile = shortcut.getRealFilename();
                    capsBuilder.withAppArguments(shortcut.getCommandLineArguments())
                            .withAppWorkingDir(shortcut.getWorkingDirectory());
                } catch (IOException | ParseException error) {
                    logger.logWarning(MessageFormat.format("Failed to read the shortcut file \"{0}\": {1}", appFile,
                            error.getMessage()));
                }
            }

            try {
                capsBuilder.withAppWorkingDir(new File(finalAppFile).getParentFile().getCanonicalPath());
            } catch (Throwable error) {
                // Ignore
            }

            DesiredCapabilities desiredCapabilities = capsBuilder.withApp(finalAppFile).merge(initCapabilities).build();

            windowsSession = new WindowsSession(remoteAddressURL, finalAppFile, initCapabilities, proxyInfo, appTitle,
                    strategy);
            windowsSession.setApplicationDriver(newWindowsDriver(remoteAddressURL, desiredCapabilities, proxyInfo));

            return windowsSession;
        } catch (WebDriverException e) {
            // Typical exception from FlaUI that can be retriable
            // "Could not start a new session. Response code 500. Message: Could not find process with id: 2012"

            if (StringUtils.isEmpty(appTitle)) {
                throw e;
            }
            if (!(e instanceof NoSuchWindowException) && !(e instanceof SessionNotCreatedException)) {
                throw e;
            }
            if (e.getMessage() != null && e.getMessage().contains("The system cannot find the file specified")) {
                // appFile is not correct
                throw e;
            }
            if ("Root".equals(appFile)) {
                throw e;
            }

            // Wait a bit and give it a 2nd attempt to start application by matching app title.
            // If still fails, re-throw the original exception
            try {
                Thread.sleep(1000); // This sleep is required. Otherwise it will fail to find that window/process again
                WindowsDriver foundDriver = switchToWindowTitle(appTitle, strategy);
                if (foundDriver != null) {
                    return windowsSession;
                }
            } catch (Exception secondTryEx) {
                logger.logWarning(
                        MessageFormat.format("Failed to re-attempt to switch to window with title = \"{0}\": {1}",
                                appTitle, secondTryEx.getMessage()));
            }

            throw e;
        }
    }

    public static WindowsDriver switchToWindowTitle(String title, StringMatchingStrategy strategy)
            throws IOException, URISyntaxException, InterruptedException {
        return new WindowsActionHelper(windowsSession).switchToWindowTitle(title, strategy);
    }

    public static WindowsDriver newWindowsDriver(URL remoteAddressURL, DesiredCapabilities desiredCapabilities,
            ProxyInformation proxyInfo) throws IOException, URISyntaxException {
        if (remoteAddressURL != null) {
            return new WindowsDriver(getAppiumExecutorForRemoteDriver(remoteAddressURL, proxyInfo, desiredCapabilities),
                    desiredCapabilities);
        } else {
            return new WindowsDriver(desiredCapabilities);
        }
    }

    public static AppiumCommandExecutor getAppiumExecutorForRemoteDriver(URL remoteWebServerUrl,
            ProxyInformation proxyInfo, MutableCapabilities capabilities) throws IOException, URISyntaxException {
        Factory clientFactory = ConfiguredHttpClientFactory.of(proxyInfo, remoteWebServerUrl, capabilities);
        AppiumCommandExecutor executor = new AppiumCommandExecutor(MobileCommand.commandRepository, remoteWebServerUrl,
                clientFactory);
        return executor;
    }
}
