package com.kms.katalon.core.webui.driver;

import java.io.IOException;

import org.openqa.selenium.Capabilities;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.remote.AbstractDriverOptions;

import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.network.ProxyOption;
import com.kms.katalon.core.webui.driver.bidi.BiDiDriverUtil;
import com.kms.katalon.core.webui.driver.smart.SmartDriverUtil;
import com.kms.katalon.core.webui.util.WebDriverProxyUtil;
import com.kms.katalon.selenium.constant.SeleniumW3CCapabilityConstant;
import com.kms.katalon.selenium.validator.W3CCapabilitiesValidator;

public abstract class AbstractDriverBuilder extends BaseDriverBuilder {

    @Override
    public WebDriver build() throws Exception {
        var capabilities = prepareCapabilities(this.capabilities);
        W3CCapabilitiesValidator.validate(capabilities);

        var useBiDi = canUseBiDi() && BiDiDriverUtil.getBiDiCapability(capabilities);
        var useExtension = !useBiDi;
        var options = createOptions(capabilities);

        if (useExtension) {
            if (withSmartLocator) {
                options = addSmartLocatorExtension(options);
            }

            if (withSmartWait) {
                options = addSmartWaitExtension(options);
            }

            if (withFlutterAppTesting) {
                options = addFlutterAppTestingExtension(options);
            }

            if (withCanvasTextExtraction) {
                options = addCanvasTextExtractionExtension(options);
            }

            if (withClosedShadowDOMTesting) {
                options = addClosedShadowDOMTestingExtension(options);
            }
        }

        options = applyProxy(options);

        var driver = createDriver(options);

        if (useBiDi) {
            injectSmartExtensions(driver);
        }

        // Use smart driver for action delay, and optionally smart wait
        driver = SmartDriverUtil.createSmartDriver(driver);

        return driver;
    }

    protected boolean canUseBiDi() {
        return false;
    }

    protected Capabilities prepareCapabilities(Capabilities capabilities) {
        if (canUseBiDi()) {
            capabilities = enableDefaultBiDi(capabilities);
        }

        capabilities = ignoreUnhandledPrompt(capabilities);

        return capabilities;
    }

    protected abstract AbstractDriverOptions<?> createOptions(Capabilities capabilities);

    protected AbstractDriverOptions<?> addSmartWaitExtension(AbstractDriverOptions<?> options) throws IOException {
        return options;
    }

    protected AbstractDriverOptions<?> addSmartLocatorExtension(AbstractDriverOptions<?> options) throws IOException {
        return options;
    }

    protected AbstractDriverOptions<?> addFlutterAppTestingExtension(AbstractDriverOptions<?> options)
            throws IOException {
        return options;
    }

    protected AbstractDriverOptions<?> addCanvasTextExtractionExtension(AbstractDriverOptions<?> options)
            throws IOException {
        return options;
    }

    protected AbstractDriverOptions<?> addClosedShadowDOMTestingExtension(AbstractDriverOptions<?> options)
            throws IOException {
        return options;
    }

    protected AbstractDriverOptions<?> applyProxy(AbstractDriverOptions<?> options) throws IOException {
        var proxyInfo = RunConfiguration.getProxyInformation();
        if (proxyInfo != null && !ProxyOption.NO_PROXY.equals(ProxyOption.valueOf(proxyInfo.getProxyOption()))
                && proxyInfo.isApplyToDesiredCapabilities()) {
            options.setProxy(WebDriverProxyUtil.getSeleniumProxy(proxyInfo));
        }

        return options;
    }

    protected abstract WebDriver createDriver(AbstractDriverOptions<?> options) throws Exception;

    protected Capabilities enableDefaultBiDi(Capabilities capabilities) {
        var hasBiDi = BiDiDriverUtil.hasBiDiCapability(capabilities);

        if (!hasBiDi) {
            var newCaps = new MutableCapabilities(capabilities);
            newCaps.setCapability(SeleniumW3CCapabilityConstant.WEB_SOCKET_URL_CAP, true);
            return newCaps;
        }

        return capabilities;
    }
    
    /**
     * Change the default value of unhandledPromptBehavior capability
     *
     * @param capabilities capabilities to update
     * @return capabilities with ignored unhandled prompt behavior by default
     * @see <a href="https://katalon.atlassian.net/browse/STUDIO-7838">STUDIO-7838</a>
     */
    protected abstract Capabilities ignoreUnhandledPrompt(Capabilities capabilities);
}
