package com.kms.katalon.core.webui.driver.bidi;

import java.io.File;
import java.io.IOException;

import org.apache.commons.io.FileUtils;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.bidi.HasBiDi;
import org.openqa.selenium.bidi.module.Script;

import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.webui.driver.DriverFactory;
import com.kms.katalon.core.webui.util.FileUtil;
import com.kms.katalon.selenium.constant.SeleniumW3CCapabilityConstant;
import com.kms.katalon.util.UrlUtil;

public class BiDiDriverUtil {
    protected static final KeywordLogger logger = KeywordLogger.getInstance(DriverFactory.class);

    private static final String[] SMART_LOCATOR_SCRIPTS = new String[] { "smart-locator-bundle.js",
            "smart-locator.js" };

    private static final String SMART_WAIT_SCRIPT = "smart-wait.js";

    private static final String FLUTTER_APP_TESTING_SCRIPT = "flutter-based-app-testing.js";

    private static final String CANVAS_TEXT_EXTRACTION_SCRIPT = "canvas-text-extraction.js";

    private static final String CLOSED_SHADOW_DOM_SCRIPT = "closed-shadow-dom-testing.js";

    private static File getScriptFolder() throws IOException {
        return new File(FileUtil.getExtensionsDirectory(), "scripts");
    }

    public static boolean hasBiDiCapability(Capabilities capabilities) {
        return capabilities.getCapability(SeleniumW3CCapabilityConstant.WEB_SOCKET_URL_CAP) != null;
    }

    public static boolean getBiDiCapability(Capabilities capabilities) {
        Object webSocketUrlCap = capabilities.getCapability(SeleniumW3CCapabilityConstant.WEB_SOCKET_URL_CAP);
        Boolean value = null;
        
        if (webSocketUrlCap instanceof String webSocketUrlCapAsString) {
            value = Boolean.parseBoolean(webSocketUrlCapAsString) ||
                    (Boolean) UrlUtil.isWebUrl(webSocketUrlCapAsString);
        } else {
            value = (Boolean) webSocketUrlCap;
        }
        
        return value != null && value;
    }

    public static boolean injectSmartWaitScript(WebDriver driver) {
        return safeInjectScript(driver, SMART_WAIT_SCRIPT);
    }

    public static boolean injectSmartLocatorScript(WebDriver driver) {
        boolean allScriptsSuccessful = true;
        for (var script : SMART_LOCATOR_SCRIPTS) {
            boolean isInjected = safeInjectScript(driver, script);
            if (!isInjected) {
                allScriptsSuccessful = false;
            }
        }
        return allScriptsSuccessful;
    }

    public static boolean injectFlutterAppTestingScript(WebDriver driver) {
        return safeInjectScript(driver, FLUTTER_APP_TESTING_SCRIPT);
    }

    public static boolean injectCanvasTextExtractionScript(WebDriver driver) {
        return safeInjectScript(driver, CANVAS_TEXT_EXTRACTION_SCRIPT);
    }

    public static boolean injectClosedShadowDOMTestingScript(WebDriver driver) {
        return safeInjectScript(driver, CLOSED_SHADOW_DOM_SCRIPT);
    }

    protected static boolean safeInjectScript(WebDriver driver, String scriptName) {
        if (driver instanceof HasBiDi hasBiDiWebDriver && hasBiDiWebDriver.maybeGetBiDi().isPresent()) {
            try {
                injectPreloadScript(driver, scriptName);
                return true;
            } catch (IOException error) {
                logger.logWarning("Cannot inject preload script: " + scriptName, null, error);
                return false;
            }
        } else {
            logger.logWarning("Your browser version or type does not support "
                    + "the BiDirectional WebDriver Protocol which is required for Smart Locator functionality. "
                    + "Please update your browser or switch to a compatible one.");
            return false;
        }
    }

    protected static void injectPreloadScript(WebDriver driver, String scriptName) throws IOException {
        File scriptFile = new File(getScriptFolder(), scriptName);
        String scriptContent = FileUtils.readFileToString(scriptFile, "UTF-8");
        try (Script script = new Script(driver)) {
            script.addPreloadScript("() => {" + scriptContent + "}");
        }
    }
    
    public static String getScriptContent(String scriptName) throws IOException {
        File scriptFile = new File(getScriptFolder(), scriptName);
        return FileUtils.readFileToString(scriptFile, "UTF-8");
    }
}
