package com.kms.katalon.core.webui.driver.remote;

import java.net.URL;
import java.text.MessageFormat;
import java.util.Map;

import org.openqa.selenium.Capabilities;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.Platform;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.remote.CapabilityType;

import com.kms.katalon.core.appium.constants.XCUITestSettingConstants;
import com.kms.katalon.core.appium.util.AppiumDriverUtil;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.util.TestCloudPropertyUtil;
import com.kms.katalon.core.webui.constants.StringConstants;
import com.kms.katalon.core.webui.driver.BaseDriverBuilder;
import com.kms.katalon.core.webui.driver.DriverFactory;
import com.kms.katalon.core.webui.driver.smart.SmartDriverUtil;
import com.kms.katalon.core.webui.util.URLUtils;
import com.kms.katalon.core.webui.util.WebDriverProxyUtil;
import com.kms.katalon.core.webui.util.WebDriverUtil;
import com.kms.katalon.selenium.driver.CRemoteWebDriver;
import com.kms.katalon.selenium.validator.W3CCapabilitiesValidator;

import io.appium.java_client.AppiumDriver;
import io.appium.java_client.android.AndroidDriver;
import io.appium.java_client.ios.IOSDriver;

public class GenericRemoteDriverBuilder extends BaseDriverBuilder {

    public static final String DRIVER_TYPE_APPIUM = "Appium";

    public static final String DRIVER_TYPE_SELENIUM = "Selenium";

    private static final String APPIUM_CAPABILITY_PLATFORM_NAME = "platformName";

    private static final String APPIUM_CAPABILITY_PLATFORM_NAME_ANDROID = "android";

    private static final String APPIUM_CAPABILITY_PLATFORM_NAME_IOS = "ios";

    @Override
    public WebDriver build() throws Exception {
        var desiredCapabilities = new MutableCapabilities().merge(capabilities);

        var remoteWebServerUrl = DriverFactory.getRemoteWebDriverServerUrl();
        var remoteWebServerType = DriverFactory.getRemoteWebDriverServerType();
        if (remoteWebServerType == null) {
            remoteWebServerType = DRIVER_TYPE_SELENIUM;
        }

        var proxyInfo = RunConfiguration.getProxyInformation();
        if (proxyInfo != null && proxyInfo.isApplyToDesiredCapabilities() && !isEdgeBrowser(desiredCapabilities)) {
            desiredCapabilities.setCapability(CapabilityType.PROXY, WebDriverProxyUtil.getSeleniumProxy(proxyInfo));
        }

        if (!TestCloudPropertyUtil.getInstance().isRunFromTestCloud(desiredCapabilities)) {
            logger.logInfo(MessageFormat.format(StringConstants.XML_LOG_CONNECTING_TO_REMOTE_WEB_SERVER_X_WITH_TYPE_Y,
                    URLUtils.removeCrendentialsFromURL(remoteWebServerUrl), remoteWebServerType));
        }

        WebDriver driver;

        if (!remoteWebServerType.equals(DRIVER_TYPE_APPIUM)) {
            W3CCapabilitiesValidator.validate(desiredCapabilities);
            var seleniumExecutor = WebDriverUtil.getHttpCommandExecutorForRemoteDriver(new URL(remoteWebServerUrl),
                    desiredCapabilities);
            driver = new CRemoteWebDriver(seleniumExecutor, desiredCapabilities,
                    driverConfigProvider.getActionDelayInMilisecond());
        } else {
            driver = createAppiumDriver(desiredCapabilities, remoteWebServerUrl);
        }

        injectSmartExtensions(driver);
        // // Use smart driver for action delay, and optionally smart wait
        return SmartDriverUtil.createSmartDriver(driver);
    }

    private WebDriver createAppiumDriver(MutableCapabilities desiredCapabilities, String remoteWebServerUrl)
            throws Exception {
        var platform = getPlatformName(desiredCapabilities);
        if (platform instanceof Platform platformEnum) {
            switch (platformEnum) {
                case ANDROID -> platform = APPIUM_CAPABILITY_PLATFORM_NAME_ANDROID;
                case IOS -> platform = APPIUM_CAPABILITY_PLATFORM_NAME_IOS;
                default -> {
                    // do nothing with other platform
                }
            }
        }

        if (!(platform instanceof String platformName)) {
            throw new StepFailedException(
                    MessageFormat.format(StringConstants.DRI_MISSING_PROPERTY_X_FOR_APPIUM_REMOTE_WEB_DRIVER,
                            APPIUM_CAPABILITY_PLATFORM_NAME));
        }

        var proxyInfo = RunConfiguration.getProxyInformation();
        var appiumExecutor = AppiumDriverUtil.getAppiumExecutorForRemoteDriver(new URL(remoteWebServerUrl), proxyInfo,
                desiredCapabilities);
        var finalCaps = AppiumDriverUtil.filterByAcceptedW3CCapabilityKeys(desiredCapabilities);

        if (APPIUM_CAPABILITY_PLATFORM_NAME_ANDROID.equalsIgnoreCase(platformName)) {
            return new AndroidDriver(appiumExecutor, finalCaps);
        } else if (APPIUM_CAPABILITY_PLATFORM_NAME_IOS.equalsIgnoreCase(platformName)) {
            AppiumDriver driver = new IOSDriver(appiumExecutor, finalCaps);
            driver.setSetting(XCUITestSettingConstants.RESPECT_SYSTEM_ALERTS, true);
            return driver;
        }

        throw new StepFailedException(MessageFormat
                .format(StringConstants.DRI_PLATFORM_NAME_X_IS_NOT_SUPPORTED_FOR_APPIUM_REMOTE_WEB_DRIVER, platform));
    }

    private static boolean isEdgeBrowser(Capabilities capabilities) {
        var names = capabilities.getCapabilityNames();
        if (names.contains("browserName") && capabilities.getCapability("browserName") instanceof String) {
            return ((String) capabilities.getCapability("browserName")).toLowerCase().contains("edge");
        }

        if (names.contains(StringConstants.BROWSER_CAPABILITY_NAME)
                && capabilities.getCapability(StringConstants.BROWSER_CAPABILITY_NAME) instanceof String) {
            return ((String) capabilities.getCapability(StringConstants.BROWSER_CAPABILITY_NAME)).toLowerCase()
                    .contains("edge");
        }

        return false;
    }

    private static Object getPlatformName(Capabilities capabilities) {
        Object platformName = capabilities.getCapability(APPIUM_CAPABILITY_PLATFORM_NAME);

        if (null == platformName) {
            var entrySet = capabilities.asMap().entrySet();
            for (var entry : entrySet) {
                if (entry.getValue() instanceof Map
                        && ((Map<?, ?>) entry.getValue()).containsKey(APPIUM_CAPABILITY_PLATFORM_NAME)) {
                    return ((Map<?, ?>) entry.getValue()).get(APPIUM_CAPABILITY_PLATFORM_NAME);
                }
            }
        }

        return platformName;
    }
}
