package com.kms.katalon.core.webui.model;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.atomic.AtomicInteger;

import org.apache.commons.lang3.exception.ExceptionUtils;
import org.openqa.selenium.WebDriverException;

import com.kms.katalon.core.util.TimeUtil;
import com.kms.katalon.core.webui.common.controller.AbortController;
import com.kms.katalon.core.webui.exception.WebElementNotFoundException;
import com.kms.katalon.core.webui.exception.WebElementNotInteractableException;
import com.kms.katalon.util.ExecutorUtils;

public class SeleniumActionRetryController {
    public static final long DEFAULT_RETRY_DELAY = 0L; // 1 second

    public static class RetryContext {
        public int retryCount;

        public long startTime;

        public RetryContext(int retryCount, long startTime) {
            this.retryCount = retryCount;
            this.startTime = startTime;
        }

        public long getElapsedTime() {
            return System.currentTimeMillis() - startTime;
        }
    }

    public static interface RetryableAction<ReturnType> {
        ReturnType execute(RetryContext retryContext) throws Exception;
    }

    public static interface ShouldRetryCondition {
        boolean get(RetryContext retryContext);
    }

    public <ReturnType> ReturnType retry(RetryableAction<ReturnType> action, int maxRetries) throws Exception {
        return retry(action, maxRetries, DEFAULT_RETRY_DELAY);
    }

    public <ReturnType> ReturnType retry(RetryableAction<ReturnType> action, int maxRetries, long delayBetweenRetries)
            throws Exception {
        return performAction(action, delayBetweenRetries, (retryContext) -> retryContext.retryCount < maxRetries);
    }

    public <ReturnType> ReturnType retry(RetryableAction<ReturnType> action, long timeout) throws Exception {
        return retry(action, timeout, DEFAULT_RETRY_DELAY);
    }

    public <ReturnType> ReturnType retry(RetryableAction<ReturnType> action, long timeout, long delayBetweenRetries)
            throws Exception {
        AbortController abortController = new AbortController();
        TimeUtil.setTimeout(() -> {
            abortController.abort("Action timed out");
        }, timeout);

        Exception[] lastException = new Exception[1];
        ExecutorService actionExecutor = Executors.newSingleThreadExecutor();
        abortController.getSignal().addListener((signal) -> {
            actionExecutor.shutdownNow();
        });
        var actionFuture = actionExecutor.submit(() -> {
            return performAction(action, delayBetweenRetries, (currentRetryCount) -> true, lastException);
        });
        try {
            return ExecutorUtils.await(actionFuture, actionExecutor);
        } catch (InterruptedException | ExecutionException e) {
            if (lastException[0] != null) {
                throw lastException[0];
            }
        }
        return null;
    }

    public <ReturnType> ReturnType performAction(RetryableAction<ReturnType> action, long delayBetweenRetries,
            ShouldRetryCondition shouldRetry) throws Exception {
        return performAction(action, delayBetweenRetries, shouldRetry, new Exception[1]);
    }

    @SuppressWarnings("deprecation")
    public <ReturnType> ReturnType performAction(RetryableAction<ReturnType> action, long delayBetweenRetries,
            ShouldRetryCondition shouldRetry, Exception[] lastException) throws Exception {
        AtomicInteger retryCount = new AtomicInteger(0);
        long startTime = System.currentTimeMillis();
        RetryContext retryContext = null;
        do {
            int currentRetryCount = retryCount.get();
            retryContext = new RetryContext(currentRetryCount, startTime);
            try {
                return action.execute(retryContext);
            } catch (InterruptedException error) {
                throw error;
            } catch (WebDriverException | WebElementNotFoundException | WebElementNotInteractableException error) {
                lastException[0] = error;
                if (!shouldRetry.get(retryContext)) {
                    throw error;
                }
            } catch (Exception error) {
                var rootCause = ExceptionUtils.getRootCause(error);
                if (rootCause instanceof InterruptedException) {
                    throw (InterruptedException) rootCause;
                }
                lastException[0] = error;
                throw error;
            }
            Thread.sleep(delayBetweenRetries);
            retryContext = new RetryContext(retryCount.getAndIncrement(), startTime);
        } while (shouldRetry.get(retryContext));
        if (lastException[0] != null) {
            throw lastException[0];
        }
        return null;
    }
}
