package com.kms.katalon.core.windows.keyword.helper;

import java.util.regex.Pattern;

import org.apache.commons.lang3.StringUtils;

import com.kms.katalon.core.util.ObjectUtil;
import com.kms.katalon.core.util.Strings;
import com.kms.katalon.core.windows.model.StringMatchingStrategy;
import com.sun.jna.Native;
import com.sun.jna.Pointer;
import com.sun.jna.WString;
import com.sun.jna.win32.StdCallLibrary;

public class WindowFinder {
    public static class Window {
        public String title;

        public String handle;
    }

    static final User32 user32 = User32.INSTANCE;

    public Window findWindowWithTitle(String title, StringMatchingStrategy strategy, int timeout)
            throws InterruptedException {
        if (strategy == null) {
            return findWindowWithTitleContains(title, timeout);
        }
        switch (strategy) {
            case EXACT:
                return findWindowWithExactTitle(title, timeout);
            case CONTAINS:
                return findWindowWithTitleContains(title, timeout);
            case REGEXP:
                return findWindowWithTitlePattern(title, timeout);
            default:
                return findWindowWithTitleContains(title, timeout);
        }
    }

    public Window findWindowWithExactTitle(String title, int timeout) throws InterruptedException {
        return findWindow(title, false, timeout);
    }

    public Window findWindowWithTitleContains(String aPartOfTitle, int timeout) throws InterruptedException {
        return findWindowWithTitlePattern(Strings.getContainsPattern(aPartOfTitle), timeout);
    }

    public Window findWindowWithTitlePattern(String pattern, int timeout) throws InterruptedException {
        return findWindow(pattern, true, timeout);
    }

    public Window findWindow(String title, boolean useRegExp, int timeout) throws InterruptedException {
        long start = System.currentTimeMillis();
        final Window[] windows = { null };
        while (true) {
            if (useRegExp) {
                final Pattern pattern = Pattern.compile(title);
                user32.EnumWindows(new User32.WNDENUMPROC() {
                    @Override
                    public boolean callback(Pointer handle, Pointer arg) {
                        if (!user32.IsWindowVisible(handle)) {
                            return true;
                        }

                        char[] windowText = new char[1024];
                        user32.GetWindowTextW(handle, windowText, 1024);
                        String wText = Native.toString(windowText).trim();
                        if (StringUtils.isEmpty(wText)) {
                            return true;
                        }

                        if (pattern.matcher(wText).matches()) {
                            Window foundWindow = new Window();
                            foundWindow.title = wText;
                            foundWindow.handle = Long.toHexString(ObjectUtil.getField(handle, "peer"));
                            windows[0] = foundWindow;
                            return false;
                        }

                        return true;
                    }
                }, null);
                if (windows[0] != null) {
                    return windows[0];
                }
            } else {
                Pointer handle = user32.FindWindowW(null, new WString(title));
                if (handle != null) {
                    Window foundWindow = new Window();
                    foundWindow.handle = Long.toHexString(ObjectUtil.getField(handle, "peer"));
                    foundWindow.title = title;
                    return foundWindow;
                }
            }
            if (System.currentTimeMillis() - start <= timeout) {
                Thread.sleep(500);
            } else {
                break;
            }
        }
        return null;
    }

    static interface User32 extends StdCallLibrary {
        User32 INSTANCE = (User32) Native.load("user32", User32.class);

        interface WNDENUMPROC extends StdCallCallback {
            boolean callback(Pointer hWnd, Pointer arg);
        }

        boolean EnumWindows(WNDENUMPROC lpEnumFunc, Pointer userData);

        int GetWindowTextW(Pointer hWnd, char[] lpWString, int nMaxCount);

        Pointer GetWindow(Pointer hWnd, int uCmd);

        Pointer FindWindowW(WString className, WString windowName);

        Boolean IsWindowVisible(Pointer hWnd);
    }
}
