package com.kms.katalon.core.windows.keyword.helper;

import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.text.MessageFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.imageio.ImageIO;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.By;
import org.openqa.selenium.Dimension;
import org.openqa.selenium.JavascriptExecutor;
import org.openqa.selenium.Keys;
import org.openqa.selenium.NoSuchElementException;
import org.openqa.selenium.NotFoundException;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.Point;
import org.openqa.selenium.Rectangle;
import org.openqa.selenium.TimeoutException;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.interactions.Actions;
import org.openqa.selenium.remote.DesiredCapabilities;
import org.openqa.selenium.remote.RemoteWebElement;
import org.openqa.selenium.support.ui.FluentWait;

import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.enums.windows.LocatorStrategy;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.testobject.WindowsTestObject;
import com.kms.katalon.core.util.internal.PathUtil;
import com.kms.katalon.core.windows.constants.WindowsDriverConstants;
import com.kms.katalon.core.windows.driver.WindowsDesiredCapsBuilder;
import com.kms.katalon.core.windows.driver.WindowsDriverFactory;
import com.kms.katalon.core.windows.driver.WindowsSession;
import com.kms.katalon.core.windows.driver.WindowsSession.TargetAppType;
import com.kms.katalon.core.windows.keyword.exception.DriverNotStartedException;
import com.kms.katalon.core.windows.keyword.exception.SessionNotStartedException;
import com.kms.katalon.core.windows.model.StringMatchingStrategy;

import io.appium.java_client.AppiumBy;
import io.appium.java_client.windows.WindowsDriver;

public class WindowsActionHelper {

    private static KeywordLogger logger = KeywordLogger.getInstance(WindowsActionHelper.class);

    private static final String FLAUI_MSG_APP_CLOSED_OR_LOST_CONNECTION = "An event was unable to invoke any of the subscribers";

    public static final List<Keys> MODIFIER_KEYS = Collections
            .unmodifiableList(Arrays.asList(Keys.SHIFT, Keys.CONTROL, Keys.ALT, Keys.META));

    private WindowsSession windowsSession;

    public WindowsActionHelper(WindowsSession windowsSession) {
        this.windowsSession = windowsSession;
    }

    public static int checkTimeout(int timeout) throws IllegalArgumentException {
        if (timeout < 0) {
            throw new IllegalArgumentException(
                    String.format("Timeout '%s' is invalid. Cannot be a negative number", timeout));
        } else if (timeout == 0) {
            int defaultPageLoadTimeout = getDefaultTimeout();
            logger.logWarning(MessageFormat.format(StringConstants.COMM_LOG_WARNING_INVALID_TIMEOUT, timeout,
                    defaultPageLoadTimeout));
            return defaultPageLoadTimeout;
        }
        return timeout;
    }

    public static WindowsActionHelper create(WindowsSession windowsSession) {
        return new WindowsActionHelper(windowsSession);
    }

    public WebElement findElement(WindowsTestObject testObject) {
        try {
            return this.findElement(testObject, getDefaultTimeout());
        } catch (NoSuchElementException exception) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }
    }

    public WebElement findElement(WindowsTestObject testObject, int timeout) {
        try {
            return this.findElement(testObject, timeout, true);
        } catch (NoSuchElementException exception) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }
    }

    public WebElement findElement(WindowsTestObject testObject, int timeout, boolean continueWhenNotFound)
            throws IllegalArgumentException, DriverNotStartedException, NoSuchElementException {
        if (windowsSession == null) {
            throw new SessionNotStartedException("Windows Session has not started yet!");
        }

        WindowsDriver windowsDriver = windowsSession.getRunningDriver();
        if (windowsDriver == null) {
            throw new DriverNotStartedException("WindowsDriver has not started yet!");
        }

        By elementLocator = buildElementLocator(testObject);
        if (elementLocator == null) {
            throw new IllegalArgumentException("No selected locator found");
        }

        timeout = WindowsActionHelper.checkTimeout(timeout);

        try {
            windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));

            FluentWait<WindowsDriver> wait = new FluentWait<WindowsDriver>(windowsDriver)
                    .withTimeout(Duration.ofSeconds(timeout))
                    .pollingEvery(Duration.ofMillis(WindowsDriverConstants.DEFAULT_FLUENT_WAIT_POLLING_TIME_OUT));
            if (continueWhenNotFound) {
                wait.ignoring(NoSuchElementException.class);
            }

            WebElement webElement = wait.until(new Function<WindowsDriver, WebElement>() {
                @Override
                public WebElement apply(WindowsDriver driver) {
                    return runWithReattachToAppOnFailure(() -> {
                        // Must re-fetch the current driver in case it has been re-attached
                        WindowsDriver currentDriver = windowsSession.getRunningDriver();
                        return currentDriver.findElement(elementLocator);
                    });
                }
            });

            return webElement;
        } finally {
            windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(getDefaultTimeout()));
        }
    }

    public List<WebElement> findElements(WindowsTestObject testObject) {
        try {
            return this.findElements(testObject, getDefaultTimeout());
        } catch (NoSuchElementException exception) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }
    }

    public List<WebElement> findElements(WindowsTestObject testObject, int timeout) {
        try {
            return this.findElements(testObject, timeout, true);
        } catch (NoSuchElementException exception) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }
    }

    public List<WebElement> findElements(WindowsTestObject testObject, int timeout, boolean continueWhenNotFound)
            throws IllegalArgumentException, DriverNotStartedException, NoSuchElementException {
        if (windowsSession == null) {
            throw new SessionNotStartedException("Windows Session has not started yet!");
        }

        WindowsDriver windowsDriver = windowsSession.getRunningDriver();
        if (windowsDriver == null) {
            throw new DriverNotStartedException("WindowsDriver has not started yet!");
        }

        By selectedLocator = buildElementLocator(testObject);
        if (selectedLocator == null) {
            throw new IllegalArgumentException("No selected locator found");
        }

        try {
            windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));

            FluentWait<WindowsDriver> wait = new FluentWait<WindowsDriver>(windowsDriver)
                    .withTimeout(Duration.ofSeconds(timeout))
                    .pollingEvery(Duration.ofMillis(WindowsDriverConstants.DEFAULT_FLUENT_WAIT_POLLING_TIME_OUT));
            if (continueWhenNotFound) {
                wait.ignoring(NoSuchElementException.class);
            }

            List<WebElement> webElementList = wait.until(new Function<WindowsDriver, List<WebElement>>() {
                @Override
                public List<WebElement> apply(WindowsDriver arg0) {
                    return runWithReattachToAppOnFailure(() -> {
                        // Must re-fetch the current driver in case it has been re-attached
                        WindowsDriver currentDriver = windowsSession.getRunningDriver();
                        return currentDriver.findElements(selectedLocator);
                    });
                }
            });
            return webElementList;
        } finally {
            windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(getDefaultTimeout()));
        }
    }

    private By buildElementLocator(WindowsTestObject testObject) {
        if (testObject == null) {
            throw new IllegalArgumentException("Test object cannot be null");
        }

        LocatorStrategy strategy = testObject.getLocatorStrategy();
        String locatorValue = testObject.getLocator();
        if (StringUtils.isEmpty(locatorValue)) {
            throw new IllegalArgumentException(String.format("Test object %s does not have locator for strategy: %s. ",
                    testObject.getObjectId(), locatorValue));
        }

        By finalLocator = null;

        switch (strategy) {
            case ACCESSIBILITY_ID:
                finalLocator = AppiumBy.accessibilityId(locatorValue);
                break;
            case CLASS_NAME:
                finalLocator = AppiumBy.className(locatorValue);
                break;
            case NAME:
                finalLocator = AppiumBy.name(locatorValue);
                break;
            case TAG_NAME:
                finalLocator = AppiumBy.tagName(locatorValue);
                break;
            case XPATH:
                finalLocator = AppiumBy.xpath(locatorValue);
                break;
            default:
                break;
        }
        return finalLocator;
    }

    private static int getDefaultTimeout() {
        try {
            return RunConfiguration.getElementTimeoutForWindows();
        } catch (Exception exception1) {
            return 0;
        }
    }

    private WebElement findElementFromElement(WebElement originalElement, By locator, int timeoutInSeconds,
            boolean continueWhenNotFound)
            throws IllegalArgumentException, DriverNotStartedException, NoSuchElementException {
        if (windowsSession == null) {
            throw new SessionNotStartedException("Windows Session has not started yet!");
        }

        WindowsDriver windowsDriver = windowsSession.getRunningDriver();
        if (windowsDriver == null) {
            throw new IllegalArgumentException("WindowsDriver has not started yet!");
        }

        if (locator == null) {
            throw new IllegalArgumentException("No selected locator found");
        }

        int finalTimeoutInSeconds = WindowsActionHelper.checkTimeout(timeoutInSeconds);

        try {
            windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));

            FluentWait<WindowsDriver> wait = new FluentWait<WindowsDriver>(windowsDriver)
                    .withTimeout(Duration.ofSeconds(finalTimeoutInSeconds))
                    .pollingEvery(Duration.ofMillis(WindowsDriverConstants.DEFAULT_FLUENT_WAIT_POLLING_TIME_OUT));
            if (continueWhenNotFound) {
                wait.ignoring(NoSuchElementException.class);
            }

            WebElement webElement = wait.until(new Function<WindowsDriver, WebElement>() {
                @Override
                public WebElement apply(WindowsDriver driver) {
                    return originalElement.findElement(locator);
                }
            });

            return webElement;
        } finally {
            windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(getDefaultTimeout()));
        }
    }

    public Map<String, String> getAdditionalAttributes(WindowsTestObject testObject) {
        WebElement webElement = findElement(testObject, 0);
        if (webElement instanceof RemoteWebElement remoteElement) {
            WindowsDriver windowsDriver = windowsSession.getRunningDriver();
            Object result = ((JavascriptExecutor) windowsDriver).executeScript("windows: getAttributes",
                    remoteElement.getId(), true  // detailed
            );
            if (result instanceof Map<?, ?> map) {
                Map<String, String> attrs = new HashMap<>();
                for (Map.Entry<?, ?> entry : map.entrySet()) {
                    Object key = entry.getKey();
                    Object value = entry.getValue();
                    if (key instanceof String && value instanceof String) {
                        attrs.put((String) key, (String) value);
                    }
                }
                return attrs;
            }
        }
        return Map.of();
    }

    public String getText(WindowsTestObject testObject) {
        WebElement webElement = findElement(testObject);

        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }
        logger.logDebug("Getting text of test object: " + testObject.getObjectId());
        return webElement.getText();
    }

    public void clearText(WindowsTestObject testObject) {
        WebElement webElement = findElement(testObject);
        clearText(testObject, webElement);
    }

    public void clearText(WindowsTestObject testObject, WebElement webElement) {
        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }
        logger.logDebug("Clearing text of test object: " + testObject.getObjectId());
        logger.logDebug("Text before clearing: " + webElement.getText());
        webElement.clear();
    }

    public void click(WindowsTestObject testObject) {
        WebElement webElement = findElement(testObject);

        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        logger.logInfo("Clicking on element: " + testObject.getObjectId());
        webElement.click();
    }

    public void clickOffset(WindowsTestObject testObject, int offsetX, int offsetY) {
        WebElement webElement = findElement(testObject);

        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        logger.logInfo("Clicking on element: " + testObject.getObjectId() + " at offset " + offsetX + " ," + offsetY);
        Actions builder = new Actions(windowsSession.getRunningDriver());
        builder.moveToElement(webElement, offsetX, offsetY).click().build().perform();
    }

    public void rightClickOffset(WindowsTestObject testObject, int offsetX, int offsetY) {
        WebElement webElement = findElement(testObject);

        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        logger.logInfo(
                "Right clicking on element: " + testObject.getObjectId() + " at offset " + offsetX + " ," + offsetY);

        WindowsDriver windowsDriver = windowsSession.getRunningDriver();
        Actions action = new Actions(windowsDriver);
        action.moveToElement(webElement, offsetX, offsetY).contextClick().perform();
    }

    public void rightClick(WindowsTestObject testObject) {
        WebElement webElement = findElement(testObject);

        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        logger.logInfo("Right clicking on element: " + testObject.getObjectId());

        WindowsDriver windowsDriver = windowsSession.getRunningDriver();
        Actions action = new Actions(windowsDriver);
        action.moveToElement(webElement).contextClick().perform();
    }

    public void doubleClick(WindowsTestObject testObject) {
        WebElement webElement = findElement(testObject);

        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        logger.logInfo("Double clicking on element: " + testObject.getObjectId());

        WindowsDriver windowsDriver = windowsSession.getRunningDriver();

        Actions action = new Actions(windowsDriver);
        action.moveToElement(webElement);
        action.doubleClick();
        action.perform();
    }

    public void hover(WindowsTestObject testObject) {
        WebElement webElement = findElement(testObject);

        if (webElement == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        if (!webElement.isDisplayed()) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not visible to hover");
        }

        logger.logInfo("Hovering over element: " + testObject.getObjectId());

        WindowsDriver windowsDriver = windowsSession.getRunningDriver();

        Actions action = new Actions(windowsDriver);
        action.moveToElement(webElement);
        action.perform();
    }

    public void sendKeys(WindowsTestObject testObject, CharSequence... keys) {
        WebElement windowElement = findElement(testObject);

        if (!isModifierKeyIncluded(keys)) {
            windowElement.sendKeys(keys);
            return;
        }

        // If shift/control/meta/alt included, should use action chain instead
        WindowsDriver windowsDriver = windowsSession.getRunningDriver();
        Actions actions = buildActionsForSendKeys(windowsDriver, windowElement, keys);
        actions.perform();
    }

    /**
     * Return null if the input character is non-modifier
     * 
     * @param c
     * @return
     */
    private Keys extractModifierKey(char c) {
        for (Keys modifierKey : MODIFIER_KEYS) {
            if (c == modifierKey.charAt(0)) {
                return modifierKey;
            }
        }

        return null;
    }

    private boolean isModifierKeyIncluded(CharSequence... keys) {
        for (CharSequence key : keys) {
            for (int i = 0; i < key.length(); i++) {
                if (extractModifierKey(key.charAt(i)) != null) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Note that inputText can contain special character such as shift, alt, control, and meta (the windows key)
     * We need to build an action chain with proper keyUp keyDown for FlaUI to act properly
     * 
     * @param windowsDriver
     * @param element
     * @param keys
     * @return
     */
    private Actions buildActionsForSendKeys(WindowsDriver windowsDriver, WebElement element, CharSequence... keys) {
        Actions actions = new Actions(windowsDriver);

        actions.click(element);

        StringBuilder normalText = new StringBuilder();
        List<Keys> modifierKeysPressed = new ArrayList<Keys>();
        boolean gotNullKey = false;

        for (CharSequence key : keys) {
            for (int i = 0; i < key.length(); i++) {
                char c = key.charAt(i);

                if (c == Keys.NULL.charAt(0)) {
                    gotNullKey = true;
                    break; // inner loop
                }

                Keys modifierKey = extractModifierKey(c);
                if (modifierKey != null) {
                    if (!StringUtils.isEmpty(normalText)) {
                        actions.sendKeys(normalText.toString());
                        normalText.setLength(0);
                    }
                    actions.keyDown(modifierKey);
                    modifierKeysPressed.add(modifierKey);
                    continue;
                }

                normalText.append(c);
            }

            if (gotNullKey) {
                break; // outer loop
            }
        }

        // Send remaining text
        if (!StringUtils.isEmpty(normalText)) {
            actions.sendKeys(normalText.toString());
        }

        // Keyup
        for (Keys modifierKey : modifierKeysPressed) {
            actions.keyUp(modifierKey);
        }

        return actions;
    }

    public void setText(WindowsTestObject testObject, String text) {
        WebElement windowElement = findElement(testObject);
        setText(testObject, windowElement, text);
    }

    public void setText(WindowsTestObject testObject, WebElement windowElement, String text) {
        logger.logDebug("Setting text on test object: " + testObject.getObjectId());
        windowElement.sendKeys(text);
    }

    public Point getPosition(WindowsTestObject testObject) {
        WebElement windowElement = findElement(testObject);
        return windowElement.getLocation();
    }

    public Rectangle getRect(WindowsTestObject testObject) {
        WebElement windowElement = findElement(testObject);
        Point position = windowElement.getLocation();
        Dimension size = windowElement.getSize();
        return new Rectangle(position.x, position.y, size.height, size.width);
    }

    /**
     * Steps:
     * - Check if item visible (aka combobox is in "expanded" state). If not, alt+down to expand combobox
     * - Then click the item
     * 
     * @param testObject
     * @param labelText
     * @param timeoutInSeconds
     */
    public void selectComboBoxItemByLabel(WindowsTestObject testObject, String labelText, int timeoutInSeconds) {
        WebElement combobox = findElement(testObject);
        if (combobox == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        if (combobox instanceof RemoteWebElement remoteCombobox) {
            WindowsDriver windowsDriver = windowsSession.getRunningDriver();
            ((JavascriptExecutor) windowsDriver).executeScript("windows: selectFromComboBox", remoteCombobox.getId(),
                    labelText, timeoutInSeconds);
        }
    }

    public void closeApp() {
        runWithReattachToAppOnFailure(() -> {
            // Must re-fetch the current driver in case it has been re-attached
            windowsSession.getRunningDriver().close();
            return null;
        });
    }

    public void takeScreenshot(String screenshotLocation) throws IOException {
        byte[] screenshotBytes = runWithReattachToAppOnFailure(() -> {
            // Must re-fetch the current driver in case it has been re-attached
            WindowsDriver currentDriver = windowsSession.getRunningDriver();
            return currentDriver.getScreenshotAs(OutputType.BYTES);
        });

        writeBytesToPngFile(screenshotBytes, screenshotLocation);
    }

    public void takeElementScreenshot(String screenshotLocation, WindowsTestObject testObject) throws IOException {
        WebElement element = findElement(testObject);
        if (element == null) {
            throw new StepFailedException("Element: " + testObject.getObjectId() + " not found");
        }

        byte[] screenshotBytes = element.getScreenshotAs(OutputType.BYTES);
        writeBytesToPngFile(screenshotBytes, screenshotLocation);
    }

    private void writeBytesToPngFile(byte[] screenshotBytes, String filePath) throws IOException {
        File screenshotFile = new File(filePath);
        PathUtil.ensureDirectory(screenshotFile, true);

        try (ByteArrayInputStream inputStream = new ByteArrayInputStream(screenshotBytes)) {
            BufferedImage image = ImageIO.read(inputStream);
            ImageIO.write(image, "png", screenshotFile);
        }
    }

    public void switchToDesktop() throws IOException, URISyntaxException {
        windowsSession.getOrCreateDesktopDriver();
        windowsSession.setTargetAppType(TargetAppType.ROOT_APP);
    }

    public void switchToApplication() {
        windowsSession.setTargetAppType(TargetAppType.SPECIFIC_APP);
    }

    public WindowsDriver switchToWindowTitle(String windowName, StringMatchingStrategy strategy)
            throws IOException, URISyntaxException, InterruptedException {
        String appTopLevelWindowTitleMatch;

        switch (strategy) {
            case EXACT:
            case REGEXP:
                appTopLevelWindowTitleMatch = windowName;
                break;
            default:
                appTopLevelWindowTitleMatch = ".*" + windowName + ".*";
        }

        DesiredCapabilities retryDesiredCapabilities = new WindowsDesiredCapsBuilder()
                .merge(windowsSession.getInitCapabilities())
                .withAppTopLevelWindowTitleMatch(appTopLevelWindowTitleMatch)
                .build();

        WindowsDriver windowsDriver = WindowsDriverFactory.newWindowsDriver(windowsSession.getRemoteAddressURL(),
                retryDesiredCapabilities, windowsSession.getProxyInfo());

        windowsSession.setApplicationDriver(windowsDriver);
        windowsSession.setTargetAppType(TargetAppType.SPECIFIC_APP);
        bringAppToForegroundIfPossible();
        return windowsDriver;
    }

    public WindowsDriver switchToWindow(WindowsTestObject windowsObject)
            throws IOException, URISyntaxException, IllegalAccessException {
        windowsSession.getOrCreateDesktopDriver();

        WebElement webElement = findElement(windowsObject);
        if (webElement == null) {
            throw new NoSuchElementException("No such window matches with the given windowsObject");
        }

        String appTopLevelWindow = webElement.getDomAttribute("NativeWindowHandle");

        if (StringUtils.isNotEmpty(appTopLevelWindow)) {
            DesiredCapabilities retryDesiredCapabilities = new WindowsDesiredCapsBuilder()
                    .merge(windowsSession.getInitCapabilities())
                    .withAppTopLevelWindow(Integer.toHexString(Integer.parseInt(appTopLevelWindow)))
                    .build();

            WindowsDriver windowsDriver = WindowsDriverFactory.newWindowsDriver(windowsSession.getRemoteAddressURL(),
                    retryDesiredCapabilities, windowsSession.getProxyInfo());

            windowsDriver.manage().timeouts().implicitlyWait(Duration.ofSeconds(getDefaultTimeout()));

            windowsSession.setApplicationDriver(windowsDriver);
            windowsSession.setTargetAppType(TargetAppType.SPECIFIC_APP);
            return windowsDriver;
        }
        throw new NotFoundException("The found window does not have NativeWindowHandle property");
    }

    public void bringAppToForegroundIfPossible() {
        if (!TargetAppType.SPECIFIC_APP.equals(windowsSession.getTargetAppType())) {
            return; // No specific app to bring to foreground
        }

        var handle = windowsSession.getRunningDriver().getWindowHandle();
        windowsSession.getRunningDriver().switchTo().window(handle);
    }

    public void reattachToApplicationIfPossible() throws IOException, URISyntaxException, InterruptedException {
        if (windowsSession == null) {
            return;
        }

        if (TargetAppType.ROOT_APP.equals(windowsSession.getTargetAppType())) {
            WindowsActionHelper.create(windowsSession).switchToDesktop();
            return;
        }

        if (StringUtils.isBlank(windowsSession.getAppTitle())) {
            return;
        }

        this.switchToWindowTitle(windowsSession.getAppTitle(), windowsSession.getAppTitleMatchingStrategy());
    }

    private <T> T runWithReattachToAppOnFailure(Supplier<T> action) {
        try {
            return action.get();
        } catch (Exception ex) {
            if (ex.getMessage() != null && ex.getMessage().contains(FLAUI_MSG_APP_CLOSED_OR_LOST_CONNECTION)) {
                try {
                    this.reattachToApplicationIfPossible();
                } catch (Exception ex2) {
                    logger.logWarning(MessageFormat.format("Failed to re-attach to the application or desktop: {0}",
                            ex2.getMessage()));
                }

                // Retry the action after re-attaching no matter success or failure
                return action.get();
            }

            throw ex;
        }
    }
}
