package com.kms.katalon.core.webservice.common;

import java.io.File;
import java.nio.file.Path;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicLong;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.testobject.RequestObject;
import com.kms.katalon.core.testobject.ResponseObject;
import com.kms.katalon.core.util.HarFileWriter;
import com.kms.katalon.core.util.KatalonHarEntry;
import com.kms.katalon.core.util.RequestInformation;
import com.kms.katalon.core.util.internal.NamingUtil;

import net.lightbody.bmp.core.har.Har;
import net.lightbody.bmp.core.har.HarEntry;
import net.lightbody.bmp.core.har.HarLog;

public class HarLogger {
    private static String HAR_FILE_REPORT_SUB_FOLDER = "requests";
    
    private static final KeywordLogger logger = KeywordLogger.getInstance(HarLogger.class);
    
    private static final AtomicLong requestNumber = new AtomicLong(0);
    
    private HarConverter harConverter;

    public void initHarFile() {
        harConverter = new HarConverter();
        harConverter.initHarFile();
    }
    
    @SuppressWarnings("java:S5852") // Suppress slow regex rule (example rule id; check yours in Sonar)
    private static String formatCurlCommand(String curlCommand) {
        if (curlCommand == null || curlCommand.isEmpty()) {
            return curlCommand;
        }
        String newline = System.lineSeparator();
        String continuation = " \\";  // space + backslash for line continuation
 
        // Step 1: Insert newlines for major flags
        String formatted = curlCommand
                .replace(" --location ", continuation + newline + "  --location ")
                .replace(" --request ", continuation + newline + "  --request ")
                .replace(" --header ", continuation + newline + "  --header ")
                .replace(" --form ", continuation + newline + "  --form ")
                .replace(" --data ", continuation + newline + "  --data ")
                .replace(" --data-urlencode ", continuation + newline + "  --data-urlencode ")
                .replace(" --data-binary ", continuation + newline + "  --data-binary ");
        
        // Safe, non-backtracking regex for moving URL to new line
        // Uses possessive quantifiers (++), preventing ReDoS
        formatted = formatted.replaceAll("(?m)\\s++'(https?://[^'\\s]++)'\\s*+$", " \\\\\n  '$1'");
 
        return formatted.trim();
    }

    public File logHarFile(RequestObject request, ResponseObject response, String logFolder) {
        try {
            if (harConverter == null) {
                initHarFile();
            }
            
            Har har = harConverter.endHar(request, response);

            String requestName = MessageFormat.format("{0}_{1}", request.getName(),
                    String.valueOf(requestNumber.getAndIncrement()));

            RequestInformation requestInformation = new RequestInformation();
            requestInformation.setName(requestName);

            String harId = UUID.randomUUID().toString();
            requestInformation.setHarId(harId);
            requestInformation.setTestObjectId(request.getObjectId());

            String directoryPath = logFolder;
            File directory = Path.of(directoryPath, HAR_FILE_REPORT_SUB_FOLDER, Thread.currentThread().getName())
                    .toFile();
            if (!directory.exists()) {
                directory.mkdirs();
            }

            File file = NamingUtil.truncateAndSanitizePath(new File(directory, requestInformation.getName() + ".har"));
            file.createNewFile();

            Map<String, String> attributes = new HashMap<>();
            attributes.put("harId", harId);
            logger.logInfo("HAR: " + file.getAbsolutePath(), attributes);

            HarLog harLog = har.getLog();
            List<HarEntry> originalEntries = harLog.getEntries();
            List<KatalonHarEntry> newEntries = originalEntries.stream().map(entry -> {
                KatalonHarEntry katalonEntry = new KatalonHarEntry(entry, logger.getSecureValuesPattern());
                katalonEntry.set_katalonRequestInformation(requestInformation);
                
                // Generate and log curl command
                String curlCommand = CurlCommandGenerator.generateCurlCommand(entry);
                if (!curlCommand.isEmpty()) {
                    String formattedCurl = formatCurlCommand(curlCommand);
                    Map<String, String> curlAttributes = new HashMap<>();
                    curlAttributes.put("curlCommand", curlCommand);
                    curlAttributes.put("harId", harId);
                    logger.logInfo("cURL Command:\n" + formattedCurl, curlAttributes);
                    requestInformation.setCurlCommand(curlCommand);
                }
                
                return katalonEntry;
            }).collect(Collectors.toList());
            originalEntries.clear();
            originalEntries.addAll(newEntries);

            HarFileWriter.write(har, file);
            return file;
        } catch (Exception error) {
            logger.logError("Cannot create HAR file", null, error);
        }

        return null;
    }
}
