package com.kms.katalon.core.webservice.util;

import java.io.File;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Enumeration;
import java.util.List;
import java.util.Properties;

import org.apache.commons.lang3.StringUtils;

import com.atlassian.oai.validator.OpenApiInteractionValidator;
import com.atlassian.oai.validator.model.Request;
import com.atlassian.oai.validator.model.Request.Method;
import com.atlassian.oai.validator.model.SimpleRequest;
import com.atlassian.oai.validator.model.SimpleResponse;
import com.atlassian.oai.validator.report.ValidationReport;
import com.kms.katalon.core.testobject.RequestObject;
import com.kms.katalon.core.testobject.ResponseObject;
import com.kms.katalon.core.testobject.TestObjectProperty;
import com.kms.katalon.core.testobject.ValidationDataType;

import io.swagger.v3.oas.models.OpenAPI;
import io.swagger.v3.parser.OpenAPIV3Parser;
import io.swagger.v3.parser.converter.SwaggerConverter;
import io.swagger.v3.parser.core.models.SwaggerParseResult;

public class OpenAPIUtil {
    public static boolean isValidOpenAPISpecification(String specSrc, ValidationDataType dataType,
            String projectLocation) {
        specSrc = getFullSpecLocation(dataType, specSrc, projectLocation);
        try {
            return isOpenAPIv2(specSrc) || isOpenAPIv3(specSrc);
        } catch (Exception e) {
            return false;
        }
    }

    public static String getFullSpecLocation(ValidationDataType type, String specSrc, String projectLocation) {
        if (type == ValidationDataType.FILE || ((type == null || type == ValidationDataType.AUTO)
                && !isUrl(specSrc))) {
            File file = new File(specSrc);
            if (!file.isAbsolute()) {
                specSrc = new File(projectLocation, specSrc).getAbsolutePath();
            }
        }
        return specSrc;
    }
    
    private static boolean isOpenAPIv3(String specSrc) {
        OpenAPIV3Parser openAPIV3Parser = new OpenAPIV3Parser();
        SwaggerParseResult swaggerParseResult = isUrlOrFile(specSrc) ? 
                openAPIV3Parser.readLocation(specSrc, null, null) : 
                openAPIV3Parser.readContents(specSrc, null, null);
        OpenAPI openAPI = swaggerParseResult.getOpenAPI();
        List<String> messages = swaggerParseResult.getMessages();
        return openAPI != null && (messages == null || messages.isEmpty());
    }
    
    private static boolean isOpenAPIv2(String specSrc) {
        SwaggerConverter converter = new SwaggerConverter();
        SwaggerParseResult swaggerParseResult = isUrlOrFile(specSrc) ? 
                converter.readLocation(specSrc, null, null) : 
                converter.readContents(specSrc, null, null);
        OpenAPI openAPI = swaggerParseResult.getOpenAPI();
        List<String> messages = swaggerParseResult.getMessages();
        return openAPI != null && (messages == null || messages.isEmpty());
    }
    
    private static OpenApiInteractionValidator getValidator(String specSrc) throws Exception {
        OpenAPIV3Parser openAPIV3Parser = new OpenAPIV3Parser();
        SwaggerParseResult swaggerParseResult = isUrlOrFile(specSrc) ? 
                openAPIV3Parser.readLocation(specSrc, null, null) : 
                openAPIV3Parser.readContents(specSrc, null, null);
        OpenAPI openAPI = swaggerParseResult.getOpenAPI();
        List<String> messages = swaggerParseResult.getMessages();
        if (openAPI != null && (messages == null || messages.isEmpty())) {
            return OpenApiInteractionValidator.createFor(openAPI).build();
        }

        SwaggerConverter swaggerConverter = new SwaggerConverter();
        swaggerParseResult = isUrlOrFile(specSrc) ? 
                swaggerConverter.readLocation(specSrc, null, null) : 
                swaggerConverter.readContents(specSrc, null, null);
        openAPI = swaggerParseResult.getOpenAPI();
        messages = swaggerParseResult.getMessages();
        if (openAPI != null && (messages == null || messages.isEmpty())) {
            return OpenApiInteractionValidator.createFor(openAPI).build();
        } else {
            if (messages == null || messages.isEmpty()) {
                throw new Exception("Invalid OpenAPI specification");
            } else {
                throw new Exception("Invalid OpenAPI specification: " + messages.get(0));
            }
        }
    }
    
    private static boolean isUrlOrFile(String specSrc) {
        return isUrl(specSrc) ||  isFile(specSrc);
    }
    
    public static boolean isUrl(String specSrc) {
        try {
            new URL(specSrc);
            return true;
        } catch (Exception e) {
            // If an Exception is thrown, the specSrc is not a valid URL
            return false;
        }
    }
    
    private static boolean isFile(String specSrc) {
        try {
            return Files.exists(Paths.get(specSrc));
        } catch (Exception e) {
            // If an Exception is thrown, the specSrc is not a valid file path
            return false;
        }
    }

    public static ValidationReport validateRequest(RequestObject request, String specSrc) 
            throws Exception {
        OpenApiInteractionValidator validator = getValidator(specSrc);
        return validator.validateRequest(getRequestBuilder(request));
    }

    public static ValidationReport validateResponse(ResponseObject response, RequestObject request, String specSrc) 
            throws Exception {
        Request.Method method = OpenAPIUtil.parseToRequestMethod(request.getHttpMethod());
        URI uri = new URI(request.getUrl());
        OpenApiInteractionValidator validator = getValidator(specSrc);
        return validator.validateResponse(uri.getPath(), method, getResponseBuilder(response));
    }

    private static Request.Method parseToRequestMethod(String method) {
        if (StringUtils.isBlank(method) || StringUtils.startsWithIgnoreCase(method, "SOAP")) {
            return null;
        }
        return Request.Method.valueOf(method);
    }

    private static Properties extractQueries(String queryString) throws MalformedURLException {
        Properties props = new Properties();
        if (StringUtils.isNotBlank(queryString)) {
            String[] elements = queryString.split("&");
            if (elements.length != 0) {
                for (String str : elements) {
                    String[] a = str.split("=");
                    if (a.length != 1) {
                        props.setProperty(a[0], a[1]);   
                    } else {
                        props.setProperty(a[0], "");
                    }
                }
            }
        }
        return props;
    }

    private static SimpleRequest getRequestBuilder(RequestObject request) throws MalformedURLException {
        URL url = new URL(request.getRestUrl());
        String path = url.getPath();

        com.atlassian.oai.validator.model.SimpleRequest.Builder builder = createRequestBuilder(request, path);
        buildRequestQueryParams(url, builder);
        buildRequestHeaders(builder, request.getHttpHeaderProperties());
        buildRequestBody(builder, request);

        return builder.build();
    }

    private static void buildRequestBody(com.atlassian.oai.validator.model.SimpleRequest.Builder requestBuilder,
            RequestObject request) {
        if (request.getBodyContent() != null) {
            requestBuilder.withBody(request.getHttpBody());
        }
    }

    private static com.atlassian.oai.validator.model.SimpleRequest.Builder createRequestBuilder(RequestObject request,
            String path) {
        String requestMethod = request.getRestRequestMethod();
        if (StringUtils.isNotBlank(request.getSoapRequestMethod())) {
            requestMethod = request.getSoapRequestMethod();
        }
        Method method = Enum.valueOf(Method.class, requestMethod);
        return new com.atlassian.oai.validator.model.SimpleRequest.Builder(method, path);

    }

    private static void buildRequestHeaders(com.atlassian.oai.validator.model.SimpleRequest.Builder requestBuilder,
            List<TestObjectProperty> headers) {
        for (TestObjectProperty prop : headers) {
            if (prop.isActive()) {
                requestBuilder.withHeader(prop.getName(), prop.getValue());
            }
        }
    }

    private static void buildRequestQueryParams(URL url,
            com.atlassian.oai.validator.model.SimpleRequest.Builder requestBuilder) throws MalformedURLException {
        Properties props = extractQueries(url.getQuery());
        Enumeration<Object> enuKeys = props.keys();
        while (enuKeys.hasMoreElements()) {
            String key = (String) enuKeys.nextElement();
            String value = props.getProperty(key);
            requestBuilder.withQueryParam(key, value);
        }
    }

    private static SimpleResponse getResponseBuilder(ResponseObject response) throws IOException {
        com.atlassian.oai.validator.model.SimpleResponse.Builder builder = SimpleResponse.Builder
                .status(response.getStatusCode())
                .withBody(response.getResponseText());
        return builder.build();
    }
}
