package com.kms.katalon.core.mobile.helper;

import java.awt.Color;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.text.MessageFormat;
import java.util.List;
import java.util.Objects;

import javax.imageio.ImageIO;

import org.apache.commons.lang.StringUtils;
import org.openqa.selenium.OutputType;
import org.openqa.selenium.Rectangle;
import org.openqa.selenium.TakesScreenshot;
import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.WebElement;

import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.helper.screenshot.ScreenCaptor;
import com.kms.katalon.core.helper.screenshot.ScreenCaptureException;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.mobile.constants.StringConstants;
import com.kms.katalon.core.mobile.driver.AppiumDriverSession;
import com.kms.katalon.core.mobile.driver.AppiumSessionCollector;
import com.kms.katalon.core.mobile.exception.MobileException;
import com.kms.katalon.core.mobile.keyword.internal.MobileDriverFactory;
import com.kms.katalon.core.testobject.TestObject;

import io.appium.java_client.AppiumDriver;
import io.appium.java_client.remote.SupportsContextSwitching;

public class MobileScreenCaptor extends ScreenCaptor {

    private static final KeywordLogger _logger = KeywordLogger.getInstance(MobileScreenCaptor.class);

    private static final Color DEFAULT_HIDDEN_COLOR = Color.RED;

    /**
     * Takes screenshot by using
     * {@link TakesScreenshot#getScreenshotAs(OutputType)}.
     * </p>
     * Using try with multi-catch to prevent error when generating groovy document.
     */
    @Override
    protected void take(File newFile) throws ScreenCaptureException {
        String failureScreenshotFormat = System.getProperty("katalon.execution.failureScreenshotFormat");
        if (StringUtils.isNotBlank(failureScreenshotFormat) && failureScreenshotFormat.toLowerCase().equals("raw")) {
            takeRawScreenshot(newFile);
            return;
        }

        takeScreenshot(newFile);
    }

    protected void takeScreenshot(File fileName) throws ScreenCaptureException {
        try {
            AppiumDriver driver = getAnyAppiumDriver();
            takeScreenshot(driver, fileName);
        } catch (WebDriverException | StepFailedException e) {
            throw new ScreenCaptureException(e);
        }
    }

    protected void takeRawScreenshot(File fileName) throws ScreenCaptureException {
        AppiumDriver driver = getAnyAppiumDriver();
        takeScreenshot(driver, fileName);
    }

    protected AppiumDriver getAnyAppiumDriver() {
        AppiumDriver driver = null;
        try {
            driver = MobileDriverFactory.getDriver();
        } catch (StepFailedException e) {
            // Native app not running, so get from driver store
            for (Object driverObject : RunConfiguration.getStoredDrivers()) {
                if (driverObject instanceof AppiumDriver) {
                    driver = (AppiumDriver) driverObject;
                }
            }
        }

        if (driver == null) {
            throw new StepFailedException(StringConstants.KW_MSG_UNABLE_FIND_DRIVER);
        }

        return driver;
    }

    protected boolean internalSwitchToNativeContext(AppiumDriver driver) {
        return internalSwitchToContext(driver, "NATIVE");
    }

    protected boolean internalSwitchToContext(AppiumDriver driver, String contextName) {
        try {
            var contextSwitch = (SupportsContextSwitching) driver;

            for (String context : contextSwitch.getContextHandles()) {
                if (context.contains(contextName)) {
                    contextSwitch.context(context);
                    return true;
                }
            }
        } catch (WebDriverException e) {
            // Appium will raise WebDriverException error when driver.getContextHandles() is
            // called but ios-webkit-debug-proxy is not started.
            // Catch it here and ignore
        }

        return false;
    }

    public static void takeScreenshot(AppiumDriver driver, File fileName) throws ScreenCaptureException {
        try {
            BufferedImage bufferedImage = takeScreenshot(driver);
            ImageIO.write(bufferedImage, "png", fileName);
        } catch (Exception e) {
            throw new ScreenCaptureException(e);
        }
    }

    public static BufferedImage takeViewportScreenshot(AppiumDriver driver, List<TestObject> ignoredElements,
            Color hidingColor) throws MobileException {
        AppiumDriverSession session = AppiumSessionCollector.getSession(driver);
        int statusBarHeight = (int) session.getProperties().get(MobileCommonHelper.PROPERTY_NAME_STATUS_BAR_HEIGHT);
        float scaleFactor = (float) session.getProperties().get(MobileCommonHelper.PROPERTY_NAME_SCALE_FACTOR);
        Color paintColor = hidingColor != null ? hidingColor : DEFAULT_HIDDEN_COLOR;
        BufferedImage screenshot = takeScreenshot(driver);
        screenshot = removeStatusBar(screenshot, statusBarHeight);
        screenshot = hideElements(screenshot, driver, ignoredElements, paintColor, scaleFactor, statusBarHeight);
        return screenshot;
    }

    public static BufferedImage takeElementScreenshot(AppiumDriver driver, WebElement element,
            List<TestObject> ignoredElements, Color hidingColor) throws Exception {
        AppiumDriverSession session = AppiumSessionCollector.getSession(driver);

        int statusBarHeight = 0;
        float scaleFactor = 0;
        Object statusBarHeightObj = session.getProperties().get(MobileCommonHelper.PROPERTY_NAME_STATUS_BAR_HEIGHT);
        if (statusBarHeightObj == null) {
            statusBarHeight = MobileCommonHelper.getStatusBarHeight(driver);
            session.getProperties().put(MobileCommonHelper.PROPERTY_NAME_STATUS_BAR_HEIGHT, statusBarHeight);
        } else {
            statusBarHeight = ((Integer) statusBarHeightObj).intValue();
        }

        Object scaleFactorObj = session.getProperties().get(MobileCommonHelper.PROPERTY_NAME_SCALE_FACTOR);
        if (scaleFactorObj == null) {
            scaleFactor = MobileCommonHelper.getScaleFactor(driver);
            session.getProperties().put(MobileCommonHelper.PROPERTY_NAME_SCALE_FACTOR, scaleFactor);
        } else {
            scaleFactor = ((Float) scaleFactorObj).floatValue();
        }

        BufferedImage screenshot = takeViewportScreenshot(driver, ignoredElements, hidingColor);
        Rectangle rect = getBoundedRect(element, scaleFactor, statusBarHeight, screenshot.getWidth(),
                screenshot.getHeight());
        return screenshot.getSubimage(rect.x, rect.y, rect.width, rect.height);
    }

    public static BufferedImage takeAreaScreenshot(AppiumDriver driver, Rectangle rect,
            List<TestObject> ignoredElements, Color hidingColor) throws MobileException {
        try {
            BufferedImage screenshot = takeViewportScreenshot(driver, ignoredElements, hidingColor);
            return screenshot.getSubimage(rect.x, rect.y, rect.width, rect.height);
        } catch (Exception e) {
            throw new MobileException(StringConstants.KW_MSG_UNABLE_TO_TAKE_SCREENSHOT, e);
        }
    }

    private static BufferedImage takeScreenshot(AppiumDriver driver) throws MobileException {
        try {
            byte[] rawImage = driver.getScreenshotAs(OutputType.BYTES);
            return ImageIO.read(new ByteArrayInputStream(rawImage));
        } catch (Exception e) {
            throw new MobileException(StringConstants.KW_MSG_UNABLE_TO_TAKE_SCREENSHOT, e);
        }
    }

    /**
     * Hide elements by drawing overlap color layer.
     * Get the actual rectangle that bounds the element. Returned rectangle is
     *
     * @param screenshot the screenshot that need to hide elements
     * @param driver AppiumDriver used to detect hidden elements
     * @param ignoredElements hidden elements
     * @param hideColor color used to draw overlap layer.
     * @return BufferedImage that all elements is hidden.
     */
    private static BufferedImage hideElements(BufferedImage screenshot, AppiumDriver driver,
            List<TestObject> ignoredElements, Color hideColor, float scaleFactor, int statusBarHeight) {
        Objects.requireNonNull(driver);
        Objects.requireNonNull(screenshot);
        if (ignoredElements == null || ignoredElements.isEmpty()) {
            _logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_SCREENSHOT_HIDDEN_OBJECTS_COUNT, "0"));
            return screenshot;
        }

        Graphics2D g = screenshot.createGraphics();
        if (hideColor == null) {
            hideColor = DEFAULT_HIDDEN_COLOR;
        }

        g.setColor(hideColor);
        int counter = 0;
        for (TestObject to : ignoredElements) {
            try {
                WebElement element = MobileElementCommonHelper.findElementWithCheck(to, 0);
                Rectangle boundedRect = getBoundedRect(element, scaleFactor, statusBarHeight, screenshot.getWidth(),
                        screenshot.getHeight());
                g.fillRect(boundedRect.x, boundedRect.y, boundedRect.width, boundedRect.height);
                ++counter;
            } catch (Exception e) {
                _logger.logInfo(
                        MessageFormat.format(StringConstants.KW_LOG_SCREENSHOT_FAILED_HIDE_OBJECT, to.getObjectId()));
            }
        }

        _logger.logInfo(
                MessageFormat.format(StringConstants.KW_LOG_SCREENSHOT_HIDDEN_OBJECTS_COUNT, String.valueOf(counter)));
        return screenshot;
    }

    /**
     * Get the actual rectangle that bounds the element. Returned rectangle is
     * affected by device pixel ratio.
     *
     * @param driver AppiumDriver that the element is linked to.
     * @param element The element that you want to get the rectangle.
     * @return a rectangle in actual size that bounds the elements.
     */

    public static Rectangle getBoundedRect(WebElement element, float scaleFactor, int statusBarHeight, int width,
            int height) {
        Objects.requireNonNull(element);
        Rectangle rect = element.getRect();
        int targetRectWidth = (int) (rect.width * scaleFactor);
        int targetRectHeight = (int) (rect.height * scaleFactor);
        int targetRectX = (int) (rect.x * scaleFactor);
        int targetRectY = (int) (rect.y * scaleFactor) - statusBarHeight;

        if (targetRectX < 0) {
            targetRectWidth += targetRectX; // Reduce width by the negative offset
            targetRectX = 0; // Adjust X to 0
        }

        if (targetRectY < 0) {
            targetRectHeight += targetRectY; // Reduce height by the negative offset
            targetRectY = 0; // Adjust Y to 0
        }

        if (targetRectX + targetRectWidth > width) {
            targetRectWidth = width - targetRectX;
        }

        if (targetRectY + targetRectHeight > height) {
            targetRectHeight = height - targetRectY;
        }

        return new Rectangle(targetRectX, targetRectY, targetRectHeight, targetRectWidth);
    }

    public static BufferedImage removeStatusBar(BufferedImage screenshot, int statusBarHeight) {
        if (statusBarHeight <= 0) {
            _logger.logWarning(StringConstants.KW_LOG_SCREENSHOT_STATUSBAR_EXIST);
            return screenshot;
        }

        int noStatusBarHeight = screenshot.getHeight() - statusBarHeight;
        return screenshot.getSubimage(0, statusBarHeight + 1, screenshot.getWidth(), noStatusBarHeight - 1);
    }
}
