package com.kms.katalon.core.testobject.authorization;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.GeneralSecurityException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;

import com.kms.katalon.util.CryptoUtil;

public class OAuth1Authorization extends BasicRequestAuthorization {

    private static final long serialVersionUID = 1L;

    private static final String CONSUMER_KEY_KEY = "consumer_key";

    private static final String CONSUMER_SECRET_KEY = "consumer_secret";

    private static final String SIGNATURE_METHOD_KEY = "signature_method";

    private static final String TOKEN_KEY = "token";

    private static final String TOKEN_SECRET_KEY = "token_secret";

    private static final String REALM_KEY = "realm";

    public static final String AUTHORIZATION_TYPE = "OAuth 1.0";

    public OAuth1Authorization(RequestAuthorization source)
            throws UnsupportedEncodingException, GeneralSecurityException {
        this(source.getAuthorizationInfo());
    }

    public OAuth1Authorization(Map<String, String> props) {
        this.init(props);
    }

    public OAuth1Authorization(String consumerKey, String consumerSecret, OAuth1SignatureMethod signatureMethod, String token,
            String tokenSecret, String realm) throws UnsupportedEncodingException, GeneralSecurityException {

        var authorizationInfo = new HashMap<String, String>();

        if (StringUtils.isNotBlank(consumerKey)) {
            authorizationInfo.put(CONSUMER_KEY_KEY, consumerKey);
        }

        if (StringUtils.isNotBlank(consumerSecret)) {
            authorizationInfo.put(CONSUMER_SECRET_KEY, encrypt(consumerSecret));
        }

        if (Objects.nonNull(signatureMethod)) {
            authorizationInfo.put(SIGNATURE_METHOD_KEY, signatureMethod.name());
        }

        if (StringUtils.isNotBlank(token)) {
            authorizationInfo.put(TOKEN_KEY, token);
        }

        if (StringUtils.isNotBlank(tokenSecret)) {
            authorizationInfo.put(TOKEN_SECRET_KEY, encrypt(tokenSecret));
        }

        if (StringUtils.isNotBlank(realm)) {
            authorizationInfo.put(REALM_KEY, realm);
        }

        init(authorizationInfo);
    }

    public static OAuth1Authorization adapt(Map<String, String> properties) {
        if (Objects.isNull(properties)) {
            throw new IllegalArgumentException("The Map<String, String> parameter is required but null");
        }

        return new OAuth1Authorization(properties);
    }
    
    public static OAuth1Authorization adapt(RequestAuthorization requestAuthorization)
            throws UnsupportedEncodingException, GeneralSecurityException {
        if (!is(requestAuthorization)) {
            throw new IllegalArgumentException(
                    "The RequestAuthorization parameter isn't the type of OAuth1Authorization");
        }

        return new OAuth1Authorization(requestAuthorization);
    }

    public String getConsumerKey() {
        return getWithEmpty(CONSUMER_KEY_KEY);
    }

    public String getConsumerSecret() throws GeneralSecurityException, IOException {
        String encryptedValue = getWithEmpty(CONSUMER_SECRET_KEY);
        if (StringUtils.isNotBlank(encryptedValue)) {
            CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(encryptedValue);
            return CryptoUtil.decode(cryptoInfo);
        }
        return StringUtils.EMPTY;
    }

    public String getSignatureMethod() {
        return getWithEmpty(SIGNATURE_METHOD_KEY);
    }

    public String getToken() {
        return getWithEmpty(TOKEN_KEY);
    }

    public String getTokenSecret() throws GeneralSecurityException, IOException {
        String encryptedValue = getWithEmpty(TOKEN_SECRET_KEY);
        if (StringUtils.isNotBlank(encryptedValue)) {
            CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(encryptedValue);
            return CryptoUtil.decode(cryptoInfo);
        }
        return StringUtils.EMPTY;
    }

    public String getRealm() {
        return getWithEmpty(REALM_KEY);
    }

    public void validate() {
    }

    public static boolean is(RequestAuthorization requestAuthorization) {
        if (Objects.nonNull(requestAuthorization)) {
            var authType = requestAuthorization.getAuthorizationType();
            if (StringUtils.isNotBlank(authType))
                return authType.toLowerCase().equals(AUTHORIZATION_TYPE.toLowerCase());
        }
        return false;
    }

    public static String encrypt(String vl) throws UnsupportedEncodingException, GeneralSecurityException {
        if (StringUtils.isBlank(vl)) {
            return StringUtils.EMPTY;
        }
        CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(vl);
        return CryptoUtil.encode(cryptoInfo);
    }

    private void init(Map<String, String> info) {
        this.setAuthorizationType(AUTHORIZATION_TYPE);
        this.setAuthorizationInfo(info);
    }

    private String getWithEmpty(String key) {
        var dict = getAuthorizationInfo();
        if (!dict.containsKey(key))
            return StringUtils.EMPTY;

        var value = dict.get(key);
        return StringUtils.isBlank(value) ? StringUtils.EMPTY : value;
    }

}
