package com.kms.katalon.core.util.internal;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStreamReader;
import java.lang.ProcessBuilder.Redirect;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import org.apache.commons.lang3.StringUtils;

import com.kms.katalon.core.logging.KeywordLogger;

public class ProcessUtil {
    private static final KeywordLogger logger = KeywordLogger.getInstance(ProcessUtil.class);

    /**
     * Terminate a process.
     * From Java 9, it has improved its process API.
     * New interfaces ProcessHandle and ProcessHandle.Info are added.
     * These interfaces help to get process's information, handle and control processes.
     * https://www.javatpoint.com/java-9-process-api-improvement
     * 
     * @param p the process to terminate
     */
    public static void terminateProcess(Process p) {
        ProcessHandle processHandle = p.toHandle();
        // The ProcessHandle cannot be used to destroy the current process (main process)
        if (processHandle.compareTo(ProcessHandle.current()) != 0) {
            // Try to destroy the process normally
            if (processHandle.supportsNormalTermination()) {
                processHandle.destroy();

                try {
                    // Waiting for a process to complete.
                    // Call get() with timeout instead join() to avoid blocking
                    processHandle.onExit().get(2000, TimeUnit.MILLISECONDS);
                    return;
                } catch (InterruptedException | ExecutionException | TimeoutException ex) {
                    logger.logInfo(MessageFormat.format("The process with PID {0} wasn't terminated, error: {1}",
                            processHandle.pid(), ex.getMessage()));
                }
            }

            // Force to destroy
            if (processHandle.isAlive()) {
                processHandle.destroyForcibly();
            }
        }
    }

    public static void killProcessOnWindows(String processName) throws InterruptedException, IOException {
        killProcessOnWindows(processName, null, null);
    }
    
    public static void killProcessOnWindows(String processName, File logFile, File errorLogFile) throws InterruptedException, IOException {
        ProcessBuilder pb = new ProcessBuilder("taskkill", "/f", "/im", processName, "/t");
        if (logFile != null) {
            FileWriter writer = new FileWriter(logFile, true);
            writer.write(String.format("\r\nTerminating process \"%s\"\r\n", processName));
            writer.close();
            pb.redirectOutput(Redirect.appendTo(logFile));
        }
        if (errorLogFile != null) {
            pb.redirectError(Redirect.appendTo(errorLogFile));
        }
        pb.start().waitFor();
    }

    public static void killProcessOnUnix(String processName) throws InterruptedException, IOException {
        killProcessOnUnix(processName, null, null);
    }

    public static void killProcessOnUnix(String processName, File logFile, File errorLogFile) throws InterruptedException, IOException {
        ProcessBuilder pb = new ProcessBuilder("killall", processName);
        if (logFile != null) {
            FileWriter writer = new FileWriter(logFile, true);
            writer.write(String.format("\r\nTerminating process \"%s\"\r\n", processName));
            writer.close();
            pb.redirectOutput(Redirect.appendTo(logFile));
        }
        if (errorLogFile != null) {
            pb.redirectError(Redirect.appendTo(errorLogFile));
        }
        pb.start().waitFor();
    }

    public static List<String> readSync(Process process) throws InterruptedException, IOException {
        return read(process, true);
    }

    public static List<String> read(Process process) throws InterruptedException, IOException {
        return read(process, false);
    }

    public static List<String> read(Process process, boolean waitForProcessEnd) throws InterruptedException, IOException {
        List<String> output = new ArrayList<String>();
        BufferedReader reader = new BufferedReader(new InputStreamReader(process.getInputStream()));
        try {
            String line;
            while ((line = reader.readLine()) != null) {
                output.add(line);
            }
            if (waitForProcessEnd) {
                process.waitFor();
            }
        } catch (IOException error) {
            // Just skip
        } finally {
            reader.close();
        }
        return output;
    }

    public static boolean includes(List<String> result, String signal) {
        return result.stream().anyMatch(line -> StringUtils.contains(line, signal));
    }

    public static boolean isKRE() {
        return org.eclipse.core.runtime.Platform.getProduct() != null
                && org.eclipse.core.runtime.Platform.getProduct().getId().equals("com.kms.katalon.console.product");
    }
}
