package com.kms.katalon.core.event;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

@SuppressWarnings({ "unchecked", "rawtypes" })
public class EventEmitter {
    public static class EventSubscriber {
        private WeakReference<EventEmitter> emitterRef;

        private String eventName;

        private CustomEventListener listener;

        public EventSubscriber(EventEmitter emitter, String eventName, CustomEventListener listener) {
            this.emitterRef = new WeakReference<>(emitter);
            this.eventName = eventName;
            this.listener = listener;
        }

        public void unsubscribe() {
            var emitter = this.emitterRef.get();
            if (emitter == null) {
                return;
            }
            emitter.off(this.eventName, this.listener);
        }
    }

    public static class CustomEvent {
        public String type;

        public Object data;
    }

    public interface CustomEventListener<EventType> {
        void call(EventType event);
    }

    Map<String, List<CustomEventListener>> listenerMap = new ConcurrentHashMap<String, List<CustomEventListener>>();

    public <EventType extends Enum<EventType>> EventSubscriber on(EventType eventType, CustomEventListener listener) {
        return on(eventType.name(), listener);
    }

    public EventSubscriber on(String eventName, CustomEventListener listener) {
        var listeners = getListeners(eventName);
        synchronized (listeners) {
            listeners.add(listener);
        }

        return new EventSubscriber(this, eventName, listener);
    }

    public <EventType extends Enum<EventType>> void off(EventType eventType, CustomEventListener listener) {
        off(eventType.name(), listener);
    }

    public void off(String eventType, CustomEventListener listener) {
        var listeners = getListeners(eventType);
        synchronized (listeners) {
            listeners.remove(listener);
        }
    }

    private synchronized List<CustomEventListener> getListeners(String type) {
        return listenerMap.computeIfAbsent(type, k -> new ArrayList<>());
    }

    public <EventType extends CustomEvent> void emit(EventType event) {
        emit(event.type, event);
    }

    public <EventType extends Enum<EventType>> void emit(EventType eventType, Object event) {
        emit(eventType.name(), event);
    }

    public void emit(String eventType, Object event) {
        var listeners = getListeners(eventType);
        synchronized (listeners) {
            listeners.forEach(listener -> listener.call(event));
        }
    }

    public static void safeUnsubscribe(EventSubscriber subscriber) {
        if (subscriber == null) {
            return;
        }
        subscriber.unsubscribe();
    }
}
