package com.kms.katalon.core.reporting.newreport;

import java.io.IOException;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Date;
import java.util.Deque;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.commons.lang3.StringUtils;

import com.kms.katalon.core.logging.LogLevel;
import com.kms.katalon.core.logging.TestSuiteXMLLogParser;
import com.kms.katalon.core.logging.XmlLogRecord;
import com.kms.katalon.core.logging.model.ILogRecord;
import com.kms.katalon.core.logging.model.MessageLogRecord;
import com.kms.katalon.core.util.ArrayUtil;
import com.kms.katalon.report.core.models.common.DataBindingVariable;
import com.kms.katalon.report.core.models.common.ExecutedTestResult;
import com.kms.katalon.report.core.models.common.ExecutedTestStatus;
import com.kms.katalon.report.core.models.entities.ExecutedTestEntity;
import com.kms.katalon.util.DateTimes;

public class NewLogRecordModelMapper {
    private static final EnumSet<LogLevel> RECORD_RESULT_LEVELS = EnumSet.of(LogLevel.PASSED, LogLevel.FAILED,
            LogLevel.WARNING, LogLevel.NOT_RUN, LogLevel.ERROR);

    private Deque<XmlLogRecord> xmlLogRecordStack = new ArrayDeque<>();

    private Deque<Object> logRecordStack = new ArrayDeque<Object>();

    private Deque<ExecutedTestEntity> entityStack = new ArrayDeque<>();

    private NewReportModelMapper reportModelMapper = new NewReportModelMapper();

    private int lastLogRecordIndex = 0;

    private Map<String, XmlLogRecord> resultXMLRecordMap = new HashMap<>();

    private Map<String, ExecutedTestEntity> entityMap = new HashMap<>();

    public XmlLogRecord findResultXMLRecord(String entityId) {
        return resultXMLRecordMap.get(entityId);
    }

    public ExecutedTestEntity findEntity(String entityId) {
        return entityMap.get(entityId);
    }

    public void reset() {
        logRecordStack.clear();
        entityStack.clear();
        xmlLogRecordStack.clear();
        lastLogRecordIndex = 0;
        resultXMLRecordMap.clear();
        entityMap.clear();
    }

    public ExecutedTestEntity mapTheWholeLogRecords(List<XmlLogRecord> logRecords, boolean flushIncompleteRecords) {
        this.reset();
        if (logRecords == null) {
            return null;
        }

        ExecutedTestEntity firstEntity = null;
        List<XmlLogRecord> clonedLogRecords = null;
        synchronized (logRecords) {
            clonedLogRecords = new ArrayList<>(logRecords);
            lastLogRecordIndex = clonedLogRecords.size();
        }
        for (var logRecordI : clonedLogRecords) {
            var entityI = this.mapLogRecord(logRecordI, true);
            if (firstEntity == null) {
                firstEntity = entityI;
            }
        }
        if (flushIncompleteRecords) {
            this.flushIncompleteLogRecords();
        }
        return firstEntity;
    }

    public synchronized List<XmlLogRecord> collectNewLogRecords(List<XmlLogRecord> logRecords) {
        if (logRecords == null) {
            return null;
        }

        List<XmlLogRecord> newRecords = null;
        synchronized (logRecords) {
            int size = logRecords.size();
            newRecords = new ArrayList<>(logRecords.subList(lastLogRecordIndex, size));
            lastLogRecordIndex = size;
        }

        return newRecords;
    }

    public ExecutedTestEntity[] mapLogRecordParts(List<XmlLogRecord> newRecords) {
        if (newRecords == null) {
            return null;
        }

        var newEntities = newRecords.stream().map(recordI -> {
            return this.mapLogRecord(recordI, false);
        }).filter((logRecord) -> logRecord != null).collect(Collectors.toList()).toArray(new ExecutedTestEntity[] {});

        var combinedEntities = new ArrayList<ExecutedTestEntity>();
        ExecutedTestEntity lastEntity = null;
        for (var entityI : newEntities) {
            if (lastEntity != null && StringUtils.equals(entityI.id, lastEntity.id)) {
                this.mergeEntity(lastEntity, entityI);
                continue;
            }
            combinedEntities.add(entityI);
            lastEntity = entityI;
        }

        return combinedEntities.toArray(new ExecutedTestEntity[] {});
    }

    public ExecutedTestEntity mapLogRecord(XmlLogRecord xmlLogRecord, boolean collectingMode) {
        var isRunData = LogLevel.RUN_DATA.toString().equals(xmlLogRecord.getLevel().getName());
        var stackDepth = logRecordStack.size();
        var logRecord = isRunData ? (ILogRecord) logRecordStack.peekLast()
                : TestSuiteXMLLogParser.mapLogRecord(xmlLogRecord, logRecordStack);
        if (logRecord == null) {
            return null;
        }

        if (isRunData) {
            var parentEntity = entityStack.peekLast();
            var properties = xmlLogRecord.getProperties();
            if (properties == null) {
                return null;
            }
            var newDataBinding = properties.entrySet().stream().map((entry) -> {
                var variable = new DataBindingVariable();
                variable.name = entry.getKey();
                variable.value = entry.getValue();
                return variable;
            }).collect(Collectors.toList());
            parentEntity.dataBinding = ArrayUtil.concat(parentEntity.dataBinding,
                    newDataBinding.toArray(new DataBindingVariable[] {}));
            return parentEntity;
        }

        var isSameLevel = logRecordStack.size() - stackDepth == 0;
        var isGoDepeper = logRecordStack.size() - stackDepth > 0;

        ExecutedTestEntity executedEntity = null;
        try {
            if (logRecord instanceof MessageLogRecord messageLog) {
                var logMessage = reportModelMapper.mapLogMessage(messageLog);
                executedEntity = entityStack.peekLast();
                executedEntity.logs = ArrayUtil.concat(executedEntity.logs, logMessage);

                LogLevel logLevel = LogLevel.valueOf(xmlLogRecord.getLevel());
                if (RECORD_RESULT_LEVELS.contains(logLevel)) {
                    resultXMLRecordMap.put(executedEntity.id, xmlLogRecord);
                }

                return executedEntity;
            } else {
                executedEntity = reportModelMapper.mapLogRecord(logRecord, 0);
            }

            if (executedEntity == null) {
                return null;
            }

            executedEntity.index = xmlLogRecord.getIndex();

            if (isSameLevel) {
                var parent = entityStack.peekLast();
                if (parent != null) {
                    executedEntity.parentId = parent.id;
                }
            } else if (isGoDepeper) {
                var startMessage = xmlLogRecord.getMessage();
                if (StringUtils.isNotBlank(startMessage)) {
                    int stepContentIndex = startMessage.indexOf(":");
                    executedEntity.message = startMessage.substring(stepContentIndex + 1).trim();
                }

                executedEntity.status = ExecutedTestStatus.RUNNING;
                executedEntity.result = null;

                var parent = entityStack.peekLast();
                entityMap.put(executedEntity.id, executedEntity);
                if (parent != null) {
                    executedEntity.parentId = parent.id;
                    if (collectingMode) {
                        parent.children = ArrayUtil.concat(parent.children, executedEntity);
                    }
                }
                entityStack.addLast(executedEntity);
                xmlLogRecordStack.addLast(xmlLogRecord);
            } else {
                xmlLogRecordStack.pollLast();
                var startEntity = entityStack.pollLast();
                executedEntity.id = startEntity.id;
                executedEntity.parentId = startEntity.parentId;
                executedEntity.logs = startEntity.logs;
                executedEntity.dataBinding = startEntity.dataBinding;
                this.mergeEntity(startEntity, executedEntity);
            }
        } catch (IOException e) {
            // Ignore as there is no IO operation
        }
        return executedEntity;
    }

    private void mergeEntity(ExecutedTestEntity entity, ExecutedTestEntity entityPart) {
        entity.endTime = entityPart.endTime;
        entity.status = entityPart.status;
        entity.result = entityPart.result;
    }

    public ExecutedTestEntity[] flushIncompleteLogRecords() {
        String endTime = DateTimes.formatISO8601(new Date());
        for (var xmlLogRecord : xmlLogRecordStack) {
            if (!StringUtils.isBlank(xmlLogRecord.getEndTime())) {
                continue;
            }
            xmlLogRecord.setEndTime(endTime);
        }

        ExecutedTestEntity[] incompleteRecords = entityStack.toArray(new ExecutedTestEntity[] {});
        for (var entity : incompleteRecords) {
            if (entity == null) {
                continue;
            }
            entity.status = ExecutedTestStatus.COMPLETED;
            entity.result = ExecutedTestResult.INCOMPLETE;
            if (StringUtils.isBlank(entity.endTime)) {
                entity.endTime = endTime;
            }
        }
        entityStack.clear();
        logRecordStack.clear();
        xmlLogRecordStack.clear();
        return incompleteRecords;
    }
}
