package com.kms.katalon.core.reporting.newreport;

import java.io.File;
import java.io.IOException;
import java.nio.file.Paths;
import java.text.MessageFormat;
import java.util.*;

import com.kms.katalon.core.reporting.service.IFailureAnalysisService;
import com.kms.katalon.core.reporting.newreport.analyzer.FailureAnalyzer;
import com.kms.katalon.report.core.models.common.*;
import com.kms.katalon.report.core.models.entities.*;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.eclipse.e4.core.contexts.EclipseContextFactory;
import org.eclipse.e4.core.contexts.IEclipseContext;
import org.osgi.framework.BundleContext;
import org.osgi.framework.FrameworkUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.helper.LogRecordHelper;
import com.kms.katalon.core.logging.model.ILogRecord;
import com.kms.katalon.core.logging.model.MessageLogRecord;
import com.kms.katalon.core.logging.model.TestCaseLogRecord;
import com.kms.katalon.core.logging.model.TestStatus.TestStatusValue;
import com.kms.katalon.core.logging.model.TestStepLogRecord;
import com.kms.katalon.core.logging.model.TestSuiteCollectionLogRecord;
import com.kms.katalon.core.logging.model.TestSuiteLogRecord;
import com.kms.katalon.core.reporting.ReportUtil;
import com.kms.katalon.core.setting.ReportSettings;
import com.kms.katalon.core.util.FileUtil;
import com.kms.katalon.core.util.MediaUtils;
import com.kms.katalon.core.util.internal.JsonUtil;
import com.kms.katalon.report.core.models.execution.ExternalReport;
import com.kms.katalon.report.core.models.execution.TestExecution;
import com.kms.katalon.report.core.models.execution.TestProjectInfo;
import com.kms.katalon.report.core.models.logmessage.LogAttachment;
import com.kms.katalon.report.core.models.logmessage.LogAttachmentType;
import com.kms.katalon.report.core.models.logmessage.LogLevel;
import com.kms.katalon.report.core.models.logmessage.LogMessage;
import com.kms.katalon.util.DateTimes;

public class NewReportModelMapper {
    private Logger logger = LoggerFactory.getLogger(NewReportModelMapper.class);

    private static final Map<TestStatusValue, ExecutedTestResult> TEST_RESULT_MAP = new HashMap<>();

    static {
        TEST_RESULT_MAP.put(TestStatusValue.PASSED, ExecutedTestResult.PASSED);
        TEST_RESULT_MAP.put(TestStatusValue.FAILED, ExecutedTestResult.FAILED);
        TEST_RESULT_MAP.put(TestStatusValue.ERROR, ExecutedTestResult.ERROR);
        TEST_RESULT_MAP.put(TestStatusValue.WARNING, ExecutedTestResult.WARNING);
        TEST_RESULT_MAP.put(TestStatusValue.INCOMPLETE, ExecutedTestResult.INCOMPLETE);
        TEST_RESULT_MAP.put(TestStatusValue.SKIPPED, ExecutedTestResult.SKIPPED);
        TEST_RESULT_MAP.put(TestStatusValue.NOT_RUN, null);
    }

    private static final Map<TestStatusValue, ExecutedTestStatus> TEST_STATUS_MAP = new HashMap<>();
    static {
        TEST_STATUS_MAP.put(TestStatusValue.PASSED, ExecutedTestStatus.COMPLETED);
        TEST_STATUS_MAP.put(TestStatusValue.FAILED, ExecutedTestStatus.COMPLETED);
        TEST_STATUS_MAP.put(TestStatusValue.ERROR, ExecutedTestStatus.COMPLETED);
        TEST_STATUS_MAP.put(TestStatusValue.WARNING, ExecutedTestStatus.COMPLETED);
        TEST_STATUS_MAP.put(TestStatusValue.INCOMPLETE, ExecutedTestStatus.COMPLETED);
        TEST_STATUS_MAP.put(TestStatusValue.SKIPPED, ExecutedTestStatus.COMPLETED);
        TEST_STATUS_MAP.put(TestStatusValue.NOT_RUN, ExecutedTestStatus.NOT_RUN);
    }

    private static final Map<TestStatusValue, LogLevel> LOG_LEVEL_MAP = new HashMap<>();
    static {
        LOG_LEVEL_MAP.put(TestStatusValue.PASSED, LogLevel.PASSED);
        LOG_LEVEL_MAP.put(TestStatusValue.FAILED, LogLevel.FAILED);
        LOG_LEVEL_MAP.put(TestStatusValue.ERROR, LogLevel.ERROR);
        LOG_LEVEL_MAP.put(TestStatusValue.WARNING, LogLevel.WARNING);
        LOG_LEVEL_MAP.put(TestStatusValue.SKIPPED, LogLevel.SKIPPED);
        LOG_LEVEL_MAP.put(TestStatusValue.INFO, LogLevel.INFO);
        LOG_LEVEL_MAP.put(TestStatusValue.NOT_RUN, LogLevel.NOT_RUN);
    }

    public static final String MAIN_PART_KEY = "main";

    public static final int INFINITE_DEPTH = -1;

    public interface ExecutionDataPartHandler {
        public void handle(String part, String partKey) throws IOException;
    }

    @SuppressWarnings("unused")
    private class SimplifiedDataPart {
        public ExecutedTestEntity[] children;

        public LogMessage[] logs;

        public SimplifiedDataPart(ExecutedTestEntity[] children, LogMessage[] logs) {
            this.children = children;
            this.logs = logs;
        }
    }

    public ReportSettings settings;

    public File reportFolder;
    
    private IFailureAnalysisService failureAnalysisService = getFailureAnalysisService();
    
    private boolean canExecuteAIFailureAnalysis;

    public NewReportModelMapper() {
    }

    public NewReportModelMapper(ReportSettings settings, File reportFolder) {
        this.settings = settings;
        this.reportFolder = reportFolder;
        this.canExecuteAIFailureAnalysis = failureAnalysisService != null && failureAnalysisService.canExecute();
    }

    public void toTestExecutionParts(TestSuiteLogRecord testSuiteLogRecord, ExecutionDataPartHandler partHandler)
            throws IOException {
        if (partHandler == null) {
            return;
        }
        partHandler.handle(JsonUtil.toJson(toTestExecution(testSuiteLogRecord, 1), false), MAIN_PART_KEY);
        var children = testSuiteLogRecord.getChildRecords();
        for (int i = 0; i < children.length; i++) {
            ExecutedTestEntity part = this.mapLogRecord(children[i]);
            handleSimplifiedDataPart(partHandler, i, part);
        }
    }

    private void handleSimplifiedDataPart(ExecutionDataPartHandler partHandler, int i, ExecutedTestEntity part)
            throws IOException {
        SimplifiedDataPart simplifiedPart = new SimplifiedDataPart(part.children, part.logs);
        partHandler.handle(JsonUtil.toJson(simplifiedPart, false), String.valueOf(i));
    }

    public TestExecution toTestExecution(TestSuiteLogRecord testSuiteLogRecord, int depth) throws IOException {
        var execution = new TestExecution();
        execution.entity = (ExecutedTestEntry) this.mapLogRecord(testSuiteLogRecord, depth);
        execution.project = getProject(testSuiteLogRecord);

        var collectionName = testSuiteLogRecord.getTestSuiteCollectionName();
        if (StringUtils.isNotBlank(collectionName)) {
            var parentReport = new ExternalReport();
            parentReport.name = collectionName;
            parentReport.path = testSuiteLogRecord.getTestSuiteCollectionPath();
            execution.parentReport = parentReport;
        }

        return execution;
    }

    private TestProjectInfo getProject(TestSuiteLogRecord testSuiteLogRecord) {
        var project = new TestProjectInfo();
        project.name = testSuiteLogRecord.getProjectName();
        return project;
    }

    public ExecutedTestEntity mapLogRecord(ILogRecord logRecord) throws IOException {
        return mapLogRecord(logRecord, INFINITE_DEPTH);
    }

    public ExecutedTestEntity mapLogRecord(ILogRecord logRecord, int depth) throws IOException {
        ExecutedTestEntity entity = null;

        if (logRecord instanceof TestSuiteLogRecord testSuite) {
            entity = this.mapTestSuite(testSuite);
        } else if (logRecord instanceof TestCaseLogRecord testCase) {
            entity = this.mapTestCase(testCase);
        } else if (logRecord instanceof TestStepLogRecord testStep) {
            entity = this.mapTestStep(testStep);
        }

        if (entity == null) {
            return null;
        }

        this.mapCommonProperties(logRecord, entity);

        this.collectChildrenAndLogMessages(entity, logRecord, depth);

        if (isExecutedTestEntry(entity)) {
            this.mapEntryProperties(logRecord, (ExecutedTestEntry) entity);
        }

        if (shouldAnalyzeFailure(entity, depth)) {
            var isAnalyzeFailedReasonByAIEnabled = settings != null && settings.useAnalyzeFailedReasonByAI
                    && canExecuteAIFailureAnalysis;
            var failureAnalyzer = new FailureAnalyzer(isAnalyzeFailedReasonByAIEnabled, failureAnalysisService);
            failureAnalyzer.analyzeAndEnrich(entity, logRecord);
        }

        return entity;
    }
    
    private boolean shouldAnalyzeFailure(ExecutedTestEntity entity, int depth) {
        // Only analyze top-level suites and test cases
        if (depth < 0) {
            return false;
        }
        
        if (!(entity instanceof ExecutedTestSuite || entity instanceof ExecutedTestCase)) {
            return false;
        }
        
        return entity.result == ExecutedTestResult.FAILED 
            || entity.result == ExecutedTestResult.ERROR;
    }

    private boolean isExecutedTestEntry(ExecutedTestEntity entity) {
        return entity instanceof ExecutedTestSuiteCollection || entity instanceof ExecutedTestSuite
                || entity instanceof ExecutedTestCase;
    }

    private ExecutedTestSuite mapTestSuite(TestSuiteLogRecord testSuiteLogRecord) {
        var testSuite = new ExecutedTestSuite();
        testSuite.startIndex = testSuiteLogRecord.getBeforeTestSuiteLogRecords().size();
        if (StringUtils.isNotBlank(testSuiteLogRecord.getTestSuiteCollectionLocation())
                && StringUtils.isNotBlank(testSuiteLogRecord.getTestSuitePath())) {
            testSuite.reportUrl = Paths.get(testSuiteLogRecord.getTestSuiteCollectionLocation())
                    .relativize(Paths.get(testSuiteLogRecord.getTestSuitePath()))
                    .toString();
        }
        return testSuite;
    }

    private ExecutedTestCase mapTestCase(TestCaseLogRecord testCaseLogRecord) {
        var testCase = new ExecutedTestCase();
        testCase.dataIterationName = testCaseLogRecord.getIterationVariableValue();
        return testCase;
    }

    private ExecutedTestStep mapTestStep(TestStepLogRecord testStepLogRecord) {
        var step = new ExecutedTestStep();
        return step;
    }

    public LogMessage mapLogMessage(MessageLogRecord logMessageRecord) throws IOException {
        var logMessage = new LogMessage();
        logMessage.time = this.formatDate(logMessageRecord.getStartTime());
        logMessage.message = logMessageRecord.getMessage();
        logMessage.level = this.mapLogLevel(logMessageRecord);

        var attachmentPath = logMessageRecord.getAttachment();
        if (StringUtils.isNotBlank(attachmentPath)) {
            logMessage.attachments = new LogAttachment[] { collectAttachment(attachmentPath) };
        }

        return logMessage;
    }

    private LogAttachment collectAttachment(String attachmentPath) throws IOException {
        var attachment = new LogAttachment();
        attachment.type = LogAttachmentType.IMAGE; // KS only support image attachment for now

        if (this.settings == null || this.settings.useHTMLImageReferences) {
            attachment.content = attachmentPath;
            return attachment;
        }

        File attachmentFile = getReportFile(attachmentPath);
        if (!attachmentFile.exists()) {
            logger.warn(MessageFormat.format(
                    "The attachment does not exist and cannot be embedded in the report. Falling back to the attachment reference: \"{0}\" (Absolute path: \"{1}\")",
                    attachmentPath, attachmentFile.getAbsolutePath()));
            attachment.content = attachmentPath;
            return attachment;
        }

        File tempFile = null;
        try {
            tempFile = File.createTempFile("katalon-", ".png");
            tempFile.deleteOnExit();
        } catch (Throwable error) {
            tempFile = null;
            logger.warn(MessageFormat.format(
                    "Cannot create a temporary file for optimizing the attachment. Skipping optimization: \"{0}\"",
                    attachmentPath), error);
        }

        if (tempFile == null) {
            attachment.content = encodeAttachmentToDataURL(attachmentFile, attachmentPath);
            return attachment;
        }

        try {
            MediaUtils.optimizeImage(attachmentFile, tempFile);
            attachment.content = FileUtil.readFileToImageDataURL(tempFile);
        } catch (Throwable error) {
            logger.warn(MessageFormat.format(
                    "Failed to compress the attachment. Falling back to processing the original attachment: \"{0}\"",
                    attachmentPath), error);
            attachment.content = encodeAttachmentToDataURL(attachmentFile, attachmentPath);
        } finally {
            FileUtils.deleteQuietly(tempFile);
        }

        return attachment;
    }

    /**
     * @return The Data URL of the attachment file, or the original attachment path if it cannot be encoded.
     */
    private String encodeAttachmentToDataURL(File attachmentFile, String originalFilePath) {
        try {
            return FileUtil.readFileToImageDataURL(attachmentFile);
        } catch (Throwable error2) {
            logger.warn(MessageFormat.format(
                    "Failed to process the attachment. Falling back to the attachment reference: \"{0}\"",
                    originalFilePath), error2);
            return originalFilePath;
        }
    }

    private <EntryType extends ExecutedTestEntry> void mapEntryProperties(ILogRecord logRecord, EntryType entry) {
        entry.entityId = logRecord.getId();
        entry.statistics = this.calculateStatistics(logRecord);
        entry.dataBinding = this.collectDataBinding(logRecord);
        entry.context = this.collectExecutionContext(logRecord);
    }

    private TestExecutionContext collectExecutionContext(ILogRecord logRecord) {
        if (!(logRecord instanceof TestSuiteLogRecord)) {
            return null;
        }

        var testSuiteLogRecord = (TestSuiteLogRecord) logRecord;
        var context = new TestExecutionContext();
        var targetContext = new TargetTestExecutionContext();

        targetContext.platform = testSuiteLogRecord.getDevicePlatform();
        targetContext.deviceName = testSuiteLogRecord.getMobileDeviceName();
        targetContext.deviceOS = testSuiteLogRecord.getDevicePlatform();
        targetContext.deviceOSVersion = testSuiteLogRecord.getDeviceVersion();
        targetContext.browserName = testSuiteLogRecord.getBrowser();

        context.local = collectLocalContext(testSuiteLogRecord);
        context.target = targetContext;
        context.executedBy = testSuiteLogRecord.getUserFullName();
        context.profile = testSuiteLogRecord.getExecutionProfile();

        return context;
    }

    private LocalTestExecutionContext collectLocalContext(TestSuiteLogRecord testSuiteLogRecord) {
        var localContext = new LocalTestExecutionContext();
        localContext.hostName = testSuiteLogRecord.getHostName();
        localContext.os = testSuiteLogRecord.getOs();
        String[] appVersion = testSuiteLogRecord.getAppVersion().split("\\.");
        if (appVersion.length >= 3) {
            localContext.katalonVersion = appVersion[0] + "." + appVersion[1] + "." + appVersion[2];
        }
        return localContext;
    }

    private DataBindingVariable[] collectDataBinding(ILogRecord logRecord) {
        if (!(logRecord instanceof TestCaseLogRecord)) {
            return null;
        }

        String rawDataBinding = ReportUtil.getTestCaseLogRecordProperty((TestCaseLogRecord) logRecord,
                StringConstants.EXECUTION_BINDING_VARIABLES);
        var dataBinding = ReportUtil.parseDataBinding(rawDataBinding);
        if (dataBinding == null) {
            return null;
        }

        List<DataBindingVariable> variables = new ArrayList<>();
        for (var variable : dataBinding) {
            var dataBindingVar = new DataBindingVariable();
            dataBindingVar.name = variable.getName();
            dataBindingVar.masked = variable.isMasked();
            dataBindingVar.value = variable.isMasked() ? "" : variable.getValue();
            variables.add(dataBindingVar);
        }

        return variables.toArray(new DataBindingVariable[] {});
    }

    private ExecutedTestStatistics calculateStatistics(ILogRecord logRecord) {
        var children = logRecord.getChildRecords();
        if (children == null) {
            return null;
        }
        ExecutedTestStatistics stats = new ExecutedTestStatistics();
        stats.total = 0;
        for (var child : children) {
            if (logRecord instanceof TestSuiteLogRecord && !(child instanceof TestCaseLogRecord)) {
                continue;
            }
            updateStatistics(stats, child.getStatus().getStatusValue());
            stats.total++;
        }
        return stats;
    }

    private <EntityType extends ExecutedTestEntity> void mapCommonProperties(ILogRecord logRecord, EntityType entity) {
        entity.id = UUID.randomUUID().toString();
        entity.name = logRecord.getName();
        entity.description = logRecord.getDescription();
        entity.startTime = logRecord.getStartTime() > 0 ? this.formatDate(logRecord.getStartTime()) : null;
        entity.endTime = logRecord.getEndTime() > 0 ? this.formatDate(logRecord.getEndTime()) : null;
        entity.status = this.mapTestStatus(logRecord);
        entity.result = this.mapTestResult(logRecord);
        entity.message = logRecord.getMessage();
        entity.retryCount = LogRecordHelper.getRetyCount(logRecord);
    }

    private void collectChildrenAndLogMessages(ExecutedTestEntity entity, ILogRecord logRecord, int depth)
            throws IOException {
        if (depth == 0) {
            return;
        }
        depth -= 1;

        List<ExecutedTestEntity> children = new ArrayList<>();
        List<LogMessage> logMessages = new ArrayList<>();
        ILogRecord[] childRecords = logRecord.getChildRecords();
        int stepIndex = 0;
        for (int i = 0; i < childRecords.length; i++) {
            var stepLogRecord = childRecords[i];

            if (stepLogRecord instanceof MessageLogRecord) {
                var log = this.mapLogMessage((MessageLogRecord) stepLogRecord);
                logMessages.add(log);
                continue;
            }

            var child = this.mapLogRecord(stepLogRecord, depth);
            if (child == null) {
                continue;
            }
            child.index = stepIndex++;
            children.add(child);
        }
        entity.children = children.toArray(new ExecutedTestEntity[] {});
        entity.logs = logMessages.toArray(new LogMessage[] {});
    }

    private ExecutedTestResult mapTestResult(ILogRecord logRecord) {
        if (logRecord instanceof TestSuiteLogRecord) {
            return TEST_RESULT_MAP.get(logRecord.getStatus().getStatusValue());
        }
        return TEST_RESULT_MAP.get(logRecord.getStatus().getStatusValue());
    }

    private ExecutedTestStatus mapTestStatus(ILogRecord logRecord) {
        return TEST_STATUS_MAP.get(logRecord.getStatus().getStatusValue());
    }

    private LogLevel mapLogLevel(ILogRecord logRecord) {
        return LOG_LEVEL_MAP.get(logRecord.getStatus().getStatusValue());
    }

    private String formatDate(long datetime) {
        return DateTimes.formatISO8601(new Date(datetime));
    }

    private File getReportFile(String path) {
        File file = new File(path);
        if (file.isAbsolute()) {
            return file;
        }
        return reportFolder != null ? new File(reportFolder, path) : file;
    }

    public void toTestExecutionParts(TestSuiteCollectionLogRecord tscLogRecord, ExecutionDataPartHandler partHandler)
            throws IOException {
        if (partHandler == null || tscLogRecord == null) {
            return;
        }

        partHandler.handle(JsonUtil.toJson(toTestExecution(tscLogRecord, 1), false), MAIN_PART_KEY);

        var children = tscLogRecord.getTestSuiteRecords();
        String testSuiteCollectionLocation = tscLogRecord.getReportFolder();
        for (int i = 0; i < children.size(); i++) {
            TestSuiteLogRecord childRecord = children.get(i);
            childRecord.setTestSuiteCollectionLocation(testSuiteCollectionLocation);
            ExecutedTestEntity part = mapLogRecord(childRecord, 2);
            handleSimplifiedDataPart(partHandler, i, part);
        }
    }

    public TestExecution toTestExecution(TestSuiteCollectionLogRecord tscLogRecord, int depth) throws IOException {
        var execution = new TestExecution();
        execution.entity = (ExecutedTestEntry) mapLogRecord(tscLogRecord, depth);
        execution.project = getProject(tscLogRecord.getTestSuiteRecords().get(0));
        return execution;
    }

    private ExecutedTestEntity mapLogRecord(TestSuiteCollectionLogRecord logRecord, int depth) throws IOException {
        ExecutedTestEntry entity = new ExecutedTestSuiteCollection();
        entity.name = logRecord.getName();
        entity.startTime = formatDate(logRecord.getStartTime());
        entity.endTime = formatDate(logRecord.getEndTime());
        entity.result = TEST_RESULT_MAP.get(logRecord.getStatus());
        entity.description = logRecord.getDescription();
        entity.entityId = logRecord.getId();
        entity.statistics = calculateStatistics(logRecord);

        collectChildrenAndLogMessages(entity, logRecord, depth);

        TestExecutionContext context = new TestExecutionContext();
        TestSuiteLogRecord testSuite = logRecord.getTestSuiteRecords().get(0);
        context.local = collectLocalContext(testSuite);
        context.executedBy = testSuite.getUserFullName();
        entity.context = context;

        return entity;
    }

    private ExecutedTestStatistics calculateStatistics(TestSuiteCollectionLogRecord logRecord) {
        var testSuites = logRecord.getTestSuiteRecords();
        ExecutedTestStatistics stats = new ExecutedTestStatistics();

        for (var testSuite : testSuites) {
            TestStatusValue status = testSuite.getStatus().getStatusValue();
            updateStatistics(stats, status);
        }

        stats.total = testSuites.size();
        return stats;
    }

    private void updateStatistics(ExecutedTestStatistics stats, TestStatusValue status) {
        switch (status) {
            case PASSED:
                stats.passed++;
                break;
            case FAILED:
                stats.failed++;
                break;
            case ERROR:
                stats.errored++;
                break;
            case WARNING:
                stats.warned++;
                break;
            case INCOMPLETE:
                stats.incomplete++;
                break;
            case SKIPPED:
                stats.skipped++;
                break;
            case NOT_RUN:
                stats.notRun++;
                break;
            default:
                break;
        }
    }

    private void collectChildrenAndLogMessages(ExecutedTestEntity entity, TestSuiteCollectionLogRecord logRecord,
            int depth) throws IOException {
        if (depth == 0) {
            return;
        }
        depth -= 1;
        List<ExecutedTestEntity> children = new ArrayList<>();
        var childRecords = logRecord.getTestSuiteRecords();
        int stepIndex = 0;

        for (var stepLogRecord : childRecords) {
            ExecutedTestEntity child = mapLogRecord(stepLogRecord, depth);
            if (child != null) {
                child.index = stepIndex++;
                children.add(child);
            }
        }

        entity.children = children.toArray(new ExecutedTestEntity[0]);
    }

    private IFailureAnalysisService getFailureAnalysisService() {
        try {
            BundleContext bundleContext = FrameworkUtil.getBundle(getClass()).getBundleContext();
            if (bundleContext != null) {
                IEclipseContext eclipseContext = EclipseContextFactory.getServiceContext(bundleContext);
                return eclipseContext.get(IFailureAnalysisService.class);
            }
        } catch (Exception e) {
            logger.error("Failed to retrieve failure analysis service", e);
        }
        return null;
    }
}
