package com.kms.katalon.core.reporting.newreport.extractor;

import java.util.Optional;

import org.apache.commons.lang3.StringUtils;

import com.kms.katalon.core.testcase.BrokenTestCaseSummary;

/**
 * Extracts failure reasons from exception messages containing "Caused by:" chains.
 * <p>
 * This extractor uses the existing {@link BrokenTestCaseSummary.Utils#getCausedBySentence(String)}
 * utility to extract the root cause from nested exception stacktraces.
 * </p>
 * 
 * <p>
 * <b>Example input:</b>
 * </p>
 * 
 * <pre>
 * com.kms.katalon.core.exception.StepFailedException: Unable to click on object
 *     at com.kms.katalon.core.webui.keyword.WebUiBuiltInKeywords.clickElement(...)
 * Caused by: org.openqa.selenium.NoSuchElementException: no such element: Unable to locate element
 * </pre>
 * 
 * <p>
 * <b>Example output:</b>
 * </p>
 * 
 * <pre>
 * org.openqa.selenium.NoSuchElementException: no such element: Unable to locate element
 * </pre>
 */
public class CausedByExtractor implements IFailureReasonExtractor {
    private static final String CAUSED_BY_MARKER = "Caused by:";

    @Override
    public Optional<String> extract(String message) {
        if (StringUtils.isBlank(message) || !message.contains(CAUSED_BY_MARKER)) {
            return Optional.empty();
        }

        try {
            String causedBy = BrokenTestCaseSummary.Utils.getCausedBySentence(message);
            if (StringUtils.isNotBlank(causedBy)) {
                return Optional.of(causedBy.trim());
            }
        } catch (Exception ignored) {}

        return Optional.empty();
    }
}
