package com.kms.katalon.core.selenium.remote.http;

import java.io.IOException;
import java.net.Proxy;
import java.net.URISyntaxException;
import java.net.URL;
import java.security.KeyManagementException;
import java.security.NoSuchAlgorithmException;
import java.security.SecureRandom;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;

import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.remote.http.ClientConfig;
import org.openqa.selenium.remote.http.HttpClient;
import org.openqa.selenium.remote.http.HttpClient.Factory;
import org.openqa.selenium.remote.http.jdk.JdkHttpClient;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kms.katalon.core.network.ProxyInformation;
import com.kms.katalon.core.network.ProxyOption;
import com.kms.katalon.core.util.internal.ProxyUtil;

public class ConfiguredHttpClientFactory implements Factory {

    private static Logger logger = LoggerFactory.getLogger(ConfiguredHttpClientFactory.class);

    private static final String JDK_HTTP_AUTH_TUNNELING_DISABLED_SCHEMES = "jdk.http.auth.tunneling.disabledSchemes";

    private JdkHttpClient.Factory factory = new JdkHttpClient.Factory();

    private Proxy proxy;

    private String proxyUsername;

    private String proxyPassword;

    private Capabilities capabilities;

    public static ConfiguredHttpClientFactory of(ProxyInformation proxyInfo, URL url, Capabilities capabilities)
            throws URISyntaxException, IOException {
        if (proxyInfo != null) {
            if (proxyInfo.getProxyOption().equals(ProxyOption.MANUAL_CONFIG.name())) {
                return new ConfiguredHttpClientFactory(ProxyUtil.getProxy(proxyInfo, url), proxyInfo.getUsername(),
                        proxyInfo.getPassword(), capabilities);
            }

            return new ConfiguredHttpClientFactory(ProxyUtil.getProxy(proxyInfo, url), capabilities);
        }

        return new ConfiguredHttpClientFactory(null, capabilities);
    }

    public ConfiguredHttpClientFactory(Proxy proxy, Capabilities capabilities) {
        this(proxy, null, null, capabilities);
    }

    public ConfiguredHttpClientFactory(Proxy proxy, String proxyUsername, String proxyPassword,
            Capabilities capabilities) {
        this.proxy = proxy;
        this.proxyUsername = proxyUsername;
        this.proxyPassword = proxyPassword;
        this.capabilities = capabilities;
    }

    @Override
    public HttpClient createClient(ClientConfig config) {
        if (proxy == null) {
            return factory.createClient(config);
        }

        ClientConfig proxyInstance = config.proxy(proxy);

        Object value = capabilities.getCapability("acceptInsecureCerts");
        if (value instanceof Boolean && (Boolean) value) {
            try {
                SSLContext sslContext = createTrustAllSSLContext();
                proxyInstance = proxyInstance.sslContext(sslContext);
            } catch (KeyManagementException | NoSuchAlgorithmException e) {
                logger.error("Failed to bypass SSL validation", e);
            }
        }

        /**
         * We have change the JVM
         * system property <b>jdk.http.auth.tunneling.disabledSchemes</b> from
         * default value <b>Basic</> to empty in order to use Basic for proxy tunnel
         * </br>
         * 
         * @see <a href=
         * "https://docs.oracle.com/en/java/javase/16/core/networking-properties.html#:~:text=jdk.http.auth.tunneling.disabledSchemes">Java
         * Networking System Properties</a>
         */
        if (StringUtils.isNotBlank(proxyUsername) && StringUtils.isNotBlank(proxyPassword)) {
            String disabledSchemesValue = System.getProperty(JDK_HTTP_AUTH_TUNNELING_DISABLED_SCHEMES);
            System.setProperty(JDK_HTTP_AUTH_TUNNELING_DISABLED_SCHEMES, "");

            HttpClient client = factory.createClient(proxyInstance);

            if (null != disabledSchemesValue) {
                System.setProperty(JDK_HTTP_AUTH_TUNNELING_DISABLED_SCHEMES, disabledSchemesValue);
            } else {
                System.clearProperty(JDK_HTTP_AUTH_TUNNELING_DISABLED_SCHEMES);
            }

            return client;
        } else {
            return factory.createClient(proxyInstance);
        }
    }

    @Override
    public void cleanupIdleClients() {
        factory.cleanupIdleClients();
    }

    private SSLContext createTrustAllSSLContext() throws NoSuchAlgorithmException, KeyManagementException {
        TrustManager[] trustAllCerts = new TrustManager[] { new X509TrustManager() {

            @Override
            public void checkClientTrusted(java.security.cert.X509Certificate[] chain, String authType)
                    throws CertificateException {
            }

            @Override
            public void checkServerTrusted(java.security.cert.X509Certificate[] chain, String authType)
                    throws CertificateException {
            }

            @Override
            public X509Certificate[] getAcceptedIssuers() {
                return new X509Certificate[0];
            }
        } };

        SSLContext sslContext = SSLContext.getInstance("TLS");
        sslContext.init(null, trustAllCerts, new SecureRandom());
        return sslContext;
    }
}
