package com.kms.katalon.core.testobject;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStreamWriter;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathFactory;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;

import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.testobject.impl.HttpTextBodyContent;
import com.kms.katalon.core.util.MimeTypeUtil;
import com.kms.katalon.core.util.internal.XMLUtil;
import com.kms.katalon.util.DocumentBuilderProvider;
import com.kms.katalon.util.TransformerFactoryProvider;

public class ResponseObject implements PerformanceResourceTiming, HttpMessage {

    private static final String DF_CHARSET = "UTF-8";
    
    private String contentType = "text/plain";
    
    private String contentDisposition = null;

    private transient final Pattern SOAPPatternEnvelope = Pattern.compile(":Envelope>?");
    
    private transient final Pattern SOAPPatternBody = Pattern.compile(":Body>?");

    @Deprecated
    private String responseText;

    private int statusCode;

    private Map<String, List<String>> headerFields;
    
    private long responseHeaderSize;
    
    private long responseBodySize;
    
    private long waitingTime;
    
    private long contentDownloadTime;
    
    private String contentCharset;

    private HttpBodyContent responseBodyContent;

    public ResponseObject() {
    }

    public ResponseObject(String responseText) {
        this.responseText = responseText;
    }

    /**
     * Get the response body content as a String
     * 
     * @return the response body content as a String
     * @throws Exception if errors happened
     */
    // TODO: Detect the source to see if it is JSON, XML, HTML or plain text
    public String getResponseBodyContent() throws Exception {
        String responseText = this.getResponseTextSafely();
        if (responseText != null) {
            if (contentType != null && contentType.startsWith("application/xml")) {
                Matcher SOAPEnvMatcher = SOAPPatternEnvelope.matcher(responseText);
                Matcher SOAPBodyMatcher = SOAPPatternBody.matcher(responseText);
                if (SOAPEnvMatcher.find() && SOAPBodyMatcher.find()) {
                    // SOAP xml
                    DocumentBuilder db = DocumentBuilderProvider.newBuilderInstance();
                    Document doc = db.parse(new InputSource(new StringReader(responseText)));
                    XPath xPath = XPathFactory.newInstance().newXPath();
                    NodeList nodes = (NodeList) xPath.evaluate("/*/*/*", doc, XPathConstants.NODESET);
                    if (nodes.getLength() > 0) {
                        // Body root node always at the last
                        Node lastNode = nodes.item(nodes.getLength() - 1);
                        return nodeToString(lastNode);
                    }
                    return "";
                } else { 
                    // REST xml
                    return responseText;
                }
            } else if (contentType != null && contentType.startsWith("application/json")) {
                return responseText;
            }
            // plain text/html
            else {
                return responseText;
            }
        }
        return "";
    }

    /**
     * Get the raw response text
     * 
     * @return the raw response text
     * @throws IOException if content could not be parsed to String.
     */
    public String getResponseText() throws IOException {
        ByteArrayOutputStream outstream = new ByteArrayOutputStream();
        responseBodyContent.writeTo(outstream);
        return outstream.toString(getContentCharset());
    }

    /**
     * Set the raw response text
     * 
     * @deprecated from 5.4
     * @param responseText the new raw response text
     */
    public void setResponseText(String responseText) {
        responseBodyContent = new HttpTextBodyContent(responseText);
        this.responseText = responseText;
    }

    /**
     * Get the content type
     * 
     * @return the content type
     */
    public String getContentType() {
        return contentType;
    }

    /**
     * Set the content type
     * 
     * @param contentType the new content type
     */
    public void setContentType(String contentType) {
        this.contentType = contentType;
    }
    
    /**
     * Get the content disposition
     * 
     * @return the content disposition
     */
    public String getContentDisposition() {
        return contentDisposition;
    }

    /**
     * Set the content disposition
     * 
     * @param contentDisposition the new content disposition
     */
    public void setContentDisposition(String contentDisposition) {
        this.contentDisposition = contentDisposition;
    }

    private String nodeToString(Node node) throws TransformerException {
        StringWriter writer = new StringWriter();
        TransformerFactory tf = TransformerFactoryProvider.newInstance();
        Transformer xform = tf.newTransformer();
        xform.setOutputProperty(OutputKeys.OMIT_XML_DECLARATION, "yes");
        xform.transform(new DOMSource(node), new StreamResult(writer));
        return writer.toString();
    }

    /**
     * Check if the content type of this response is json
     * 
     * @return true if the content type of this response is json, otherwise false
     */
    public boolean isJsonContentType() {
        return contentType != null && contentType.toLowerCase().startsWith("application/json");
    }

    /**
     * Check if the content type of this response is xml
     * 
     * @return true if the content type of this response is xml, otherwise false
     */
    public boolean isXmlContentType() {
        String contentTypeString = contentType.toLowerCase();
        return contentType != null && (contentTypeString.startsWith("application/xml")
                || contentTypeString.equals("application/soap+xml") || contentTypeString.equals("text/xml"));
    }
    
    /**
     * Check if this response is soap
     * 
     * @return true if this response is soap, otherwise false
     */
    public boolean isSOAPResponse() {
        String responseText = this.getResponseTextSafely();
        
        Matcher SOAPEnvMatcher = SOAPPatternEnvelope.matcher(responseText);
        Matcher SOAPBodyMatcher = SOAPPatternBody.matcher(responseText);
        return isXmlContentType() && SOAPEnvMatcher.find() && SOAPBodyMatcher.find();
    }

    /**
     * Check if the content type of this response is raw text
     * 
     * @return true if the content type of this response is raw text, otherwise false
     */
    public boolean isTextContentType() {
        return !isJsonContentType() && !isXmlContentType();
    }

    /**
     * Get the header fields as a {@link Map}
     * 
     * @return the header fields as a {@link Map}
     */
    public Map<String, List<String>> getHeaderFields() {
        if (headerFields == null) {
            headerFields = Collections.emptyMap();
        }
        return headerFields;
    }

    /**
     * Set the header fields
     * 
     * @param headerFields the new header fields as a {@link Map}
     */
    public void setHeaderFields(Map<String, List<String>> headerFields) {
        this.headerFields = headerFields;
    }

    /**
     * Get the status code
     * 
     * @return the status code
     */
    public int getStatusCode() {
        return statusCode;
    }

    /**
     * Set the status code
     * 
     * @param statusCode the status code
     */
    public void setStatusCode(int statusCode) {
        this.statusCode = statusCode;
    }

    /**
     * Returns size (byte) as long value of the response.</br></br>
     * Response size = Header size + Body size
     * 
     * @see #getResponseHeaderSize()
     * @see #getResponseBodySize()
     * 
     */
    public long getResponseSize() {
        return getResponseHeaderSize() + getResponseBodySize();
    }

    /**
     * Returns headers size (byte) as long value of the response.
     * 
     * @see #getResponseBodySize()
     * @see #getResponseSize()
     */
    public long getResponseHeaderSize() {
        return responseHeaderSize;
    }

    public void setResponseHeaderSize(long reponseHeaderSize) {
        this.responseHeaderSize = reponseHeaderSize;
    }

    public long getResponseBodySize() {
        return responseBodySize;
    }

    /**
     * Returns body size (byte) as long value of the response.
     * 
     * @see #getResponseHeaderSize()
     * @see #getResponseSize()
     */
    public void setResponseBodySize(long reponseBodySize) {
        this.responseBodySize = reponseBodySize;
    }

    @Override
    public long getElapsedTime() {
        return getWaitingTime() + getContentDownloadTime();
    }

    @Override
    public long getWaitingTime() {
        return waitingTime;
    }
    
    public void setWaitingTime(long waitingTime) {
        this.waitingTime = waitingTime;
    }

    @Override
    public long getContentDownloadTime() {
        return contentDownloadTime;
    }
    
    public void setContentDownloadTime(long contentDownloadTime) {
        this.contentDownloadTime = contentDownloadTime;
    }

    @Override
    public HttpBodyContent getBodyContent() {
        return responseBodyContent;
    }
    
    public void setBodyContent(HttpBodyContent bodyContent) {
        this.responseBodyContent = bodyContent;
    }

    /**
     * @return Returns the Charset specified in the Content-Type of this response or the "UTF-8" charset as a default.
     * 
     * @since 5.4
     */
    public String getContentCharset() {
        if (StringUtils.isEmpty(contentCharset)) {
            return DF_CHARSET;
        }
        return contentCharset;
    }

    public void setContentCharset(String contentCharset) {
        this.contentCharset = contentCharset;
    }
    
    @Override
    public String toString() {
        return getStatusCode() + " " + FileUtils.byteCountToDisplaySize(getResponseSize());
    }
    
    public String getHeaderField(String name) {
        List<String> headerValues = headerFields.get(name);
        if (headerValues != null && headerValues.size() > 0) {
            return headerValues.get(0);
        } else {
            return null;
        }
    }
    
    public String extractSOAPBodyAsContent() {
        String responseText = this.getResponseTextSafely();
        if (responseText != null && contentType != null && isXmlContentType()) {
            Matcher SOAPEnvMatcher = SOAPPatternEnvelope.matcher(responseText);
            Matcher SOAPBodyMatcher = SOAPPatternBody.matcher(responseText);
            if (SOAPEnvMatcher.find() && SOAPBodyMatcher.find()) {
                // SOAP xml
                return XMLUtil.extractSOAPBodyAsContent(responseText);
            } else {
                // REST xml
                return responseText;
            }
        }
        return responseText;
    }
    
    /**
     * Writes the response body content to a file at the specified path. If the file already exists, it will be overwritten.
     * <p>
     * This method handles both text and binary content types automatically based on the
     * content type of the response. For text content, the file will be written using the
     * character encoding specified by {@link #getContentCharset()}. For binary content,
     * the file will be written as raw bytes.
     * </p>
     * 
     * @param absoluteFilePath the absolute path where the file should be created and written to
     * @throws IllegalStateException if the response body content is null
     * @throws IllegalArgumentException if the file path is null or empty
     * @throws IOException if the file already exists, if parent directories cannot be created,
     *         or if any I/O error occurs during writing
     * @throws UnsupportedOperationException if the content type is neither text nor binary
     * @throws Exception if any other error occurs during the operation
     * 
     * @since 10.3.0
     */
    public void writeResponseBodyToFile(String absoluteFilePath) throws Exception {
        if (responseBodyContent == null) {
            throw new IllegalStateException("Response body content is null");
        }
        
        if (StringUtils.isBlank(absoluteFilePath)) {
            throw new IllegalArgumentException("File path cannot be null or empty");
        }
        
        File file = new java.io.File(absoluteFilePath);
        if (file.getParentFile() != null && !file.getParentFile().exists()) {
            File parentDir = file.getParentFile();
            
            if (!parentDir.exists()) {
                try {
                    if (!parentDir.mkdirs()) {
                        if (!parentDir.canWrite()) {
                            throw new IOException(MessageFormat.format(
                                    StringConstants.WS_SAVE_RESPONSE_MISSING_WRITE_PERMISSION_ERROR_MSG, parentDir.getAbsolutePath()));
                        } else {
                            
                            throw new IOException(MessageFormat.format(StringConstants.WS_SAVE_RESPONSE_DISK_SPACE_ERROR_MSG, absoluteFilePath));
                        }
                    }
                } catch (SecurityException se) {
                    throw new IOException(MessageFormat.format(StringConstants.WS_SAVE_RESPONSE_SECURITY_RESTRICTION_ERROR_MSG, parentDir.getAbsolutePath()), se);
                }
            }
        }

        if (MimeTypeUtil.isTextContentType(contentType)) {
            try (InputStream inputStream = responseBodyContent.getInputStream();
                 InputStreamReader reader = new InputStreamReader(inputStream, getContentCharset());
                 BufferedReader bufferedReader = new BufferedReader(reader);
                 FileOutputStream fos = new FileOutputStream(file);
                 OutputStreamWriter writer = new OutputStreamWriter(fos, getContentCharset());
                 BufferedWriter bufferedWriter = new BufferedWriter(writer)) {
                
                char[] buffer = new char[8192]; // 8KB buffer
                int charsRead;
                while ((charsRead = bufferedReader.read(buffer)) != -1) {
                    bufferedWriter.write(buffer, 0, charsRead);
                }
            }
        } else {
            try (InputStream inputStream = responseBodyContent.getInputStream();
                 FileOutputStream fos = new FileOutputStream(file);
                 BufferedOutputStream bos = new BufferedOutputStream(fos)) {
                
                byte[] buffer = new byte[8192]; // 8KB buffer
                int bytesRead;
                while ((bytesRead = inputStream.read(buffer)) != -1) {
                    bos.write(buffer, 0, bytesRead);
                }
            }
        }
    }

    /**
     * Creates and returns a copy of this ResponseObject with the following properties:
     * <ul>
     *   <li>All primitive fields are copied by value</li>
     *   <li>The responseText field is copied by reference (shallow copy)</li>
     *   <li>The headerFields map is deep-copied with new HashMap</li>
     *   <li>The responseBodyContent is copied by reference (shallow copy)</li>
     * </ul>
     *
     * @return A new ResponseObject instance with copied values
     * @since 10.3.0
     */
    public ResponseObject clone() {
        ResponseObject cloneObject = new ResponseObject();
        
        cloneObject.statusCode = this.statusCode;
        cloneObject.contentType = this.contentType;
        cloneObject.contentDisposition = this.contentDisposition;
        cloneObject.responseHeaderSize = this.responseHeaderSize;
        cloneObject.responseBodySize = this.responseBodySize;
        cloneObject.waitingTime = this.waitingTime;
        cloneObject.contentDownloadTime = this.contentDownloadTime;
        cloneObject.contentCharset = this.contentCharset;
        
        // Copy responseText (for backward compatibility)
        cloneObject.responseText = this.responseText;
        
        // Deep copy headerFields if not null
        if (this.headerFields != null) {
            cloneObject.headerFields = new HashMap<>();
            for (Map.Entry<String, List<String>> entry : this.headerFields.entrySet()) {
                cloneObject.headerFields.put(entry.getKey(), 
                    entry.getValue() != null ? new ArrayList<>(entry.getValue()) : null);
            }
        }
        
        if (this.responseBodyContent != null) {
            cloneObject.responseBodyContent = this.responseBodyContent;
            // Note: HttpBodyContent doesn't have a clone method, so we use the same instance
            // If deep cloning is required for HttpBodyContent, additional code would be needed
        }
        
        return cloneObject;
    }
    
    private String getResponseTextSafely() {
        try {
            return this.getResponseText();
        } catch (Exception ignored) {
            // Fallback to use this.responseText
            return this.responseText;
        }
    }
}
