package com.kms.katalon.core.testobject.authorization;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.GeneralSecurityException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import org.apache.commons.lang3.StringUtils;
import com.kms.katalon.util.CryptoUtil;

/**
 * Represents AWS Signature authorization method for the Web Service Test Object
 * 
 * @since 9.7.0
 */
public class AwsSignatureAuthorization extends BasicRequestAuthorization {
    private static final long serialVersionUID = 1L;

    private static final String ADD_AUTH_DATA_TO = "aws_add_auth_data_to";
    private static final String ACCESS_KEY = "aws_access_key";
    private static final String SECRET_KEY = "aws_secret_key";
    private static final String REGION = "aws_region";
    private static final String SERVICE_NAME = "aws_service_name";
    private static final String SESSION_TOKEN = "aws_session_token";
    public static final String AUTHORIZATION_TYPE_NAME = "AWS Signature";

    public AwsSignatureAuthorization(AwsSignatureLocation signatureLocation, String awsAccessKey, String awsSecretKey, String awsRegion, String awsServiceName, String awsSessionToken) throws UnsupportedEncodingException, GeneralSecurityException {
        super();
        var authorizationInfo = new HashMap<String, String>();
        authorizationInfo.put(ADD_AUTH_DATA_TO, Objects.isNull(signatureLocation) ? AwsSignatureLocation.REQUEST_HEADER.name() : signatureLocation.name());
        
        if (StringUtils.isNotBlank(awsAccessKey)) {
            authorizationInfo.put(ACCESS_KEY, awsAccessKey);
        }
        
        if (StringUtils.isNotBlank(awsSecretKey)) {
            authorizationInfo.put(SECRET_KEY, encrypt(awsSecretKey));
        }
        
        if (StringUtils.isNotBlank(awsRegion)) {
            authorizationInfo.put(REGION, StringUtils.isBlank(awsRegion) ? "us-east-1" : awsRegion);
        }
        
        if (StringUtils.isNotBlank(awsServiceName)) {
            authorizationInfo.put(SERVICE_NAME, awsServiceName);
        }
        
        if (StringUtils.isNotBlank(awsSessionToken)) {
            authorizationInfo.put(SESSION_TOKEN, awsSessionToken);
        }

        init(authorizationInfo);
    }
    
    private AwsSignatureAuthorization(Map<String, String> properties) {
        init(properties);
    }
    
    private AwsSignatureAuthorization(RequestAuthorization requestAuthorization) {
        this(requestAuthorization.getAuthorizationInfo());
    }
    
    public static boolean is(RequestAuthorization requestAuthorization) {
        if (Objects.nonNull(requestAuthorization)) {
            var at = requestAuthorization.getAuthorizationType();
            if (StringUtils.isNotBlank(at)) return at.equals(AUTHORIZATION_TYPE_NAME);
        }
        
        return false;
    }
    
    public static AwsSignatureAuthorization adapt(RequestAuthorization requestAuthorization) {
        if (!is(requestAuthorization)) {
            throw new IllegalArgumentException("The RequestAuthorization parameter isn't the type of AwsSignatureAuthorization");
        }
        
        return new AwsSignatureAuthorization(requestAuthorization);
    }
    
    public static AwsSignatureAuthorization adapt(Map<String, String> properties) {
        if (Objects.isNull(properties)) {
            throw new IllegalArgumentException("The Map<String, String> parameter is required but null");
        }
        
        return new AwsSignatureAuthorization(properties);
    }
    
    public Optional<String> getAwsSecretKey() throws GeneralSecurityException, IOException {
        Optional<String> encryptedKey = get(SECRET_KEY);
        
        if (encryptedKey.isEmpty()) return Optional.empty();
        var vl = encryptedKey.get();
        if (StringUtils.isBlank(vl)) return Optional.empty(); 
        CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(vl);
        return Optional.of(CryptoUtil.decode(cryptoInfo));
    }
    
    public Optional<String> getAwsAccessKey() {
        return get(ACCESS_KEY);
    }
    
    public String getAwsRegion() {
        return get(REGION).orElse("us-east-1");
    }
    
    public Optional<String> getAwsServiceName() {
        return get(SERVICE_NAME);
    }
    
    public Optional<String> getAwsSessionToken() {
        return get(SESSION_TOKEN);
    }
    
    public AwsSignatureLocation getSignatureLocation() {
        var dict = getAuthorizationInfo();
        if (dict.containsKey(ADD_AUTH_DATA_TO)) {
            return AwsSignatureLocation.valueOf(dict.get(ADD_AUTH_DATA_TO));
        }
        
        return AwsSignatureLocation.REQUEST_HEADER;
    }
    
    public void validate() {
        if (getAwsAccessKey().isEmpty()) throw new IllegalArgumentException("AWS Access Key is missing");
        if (get(SECRET_KEY).isEmpty()) throw new IllegalArgumentException("AWS Secret Key is missing");
        if (getAwsServiceName().isEmpty()) throw new IllegalArgumentException("AWS Service Name is missing");
    }
    
    private Optional<String> get(String key) {
        var dict = getAuthorizationInfo();
        if (dict.containsKey(key)) {
            return Optional.of(dict.get(key));
        }
        
        return Optional.empty();
    }
    
    private void init(Map<String, String> authInfo) {
        setAuthorizationType(AUTHORIZATION_TYPE_NAME);
        setAuthorizationInfo(authInfo);
    }
    
    private String encrypt(String vl) throws UnsupportedEncodingException, GeneralSecurityException {
        CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(StringUtils.isBlank(vl) ? StringUtils.EMPTY : vl);
        return CryptoUtil.encode(cryptoInfo);
    }
}
