package com.kms.katalon.core.testobject.authorization;

import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.security.GeneralSecurityException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;

import org.apache.commons.lang3.StringUtils;

import com.kms.katalon.core.constants.OAuth2Constants;
import com.kms.katalon.util.CryptoUtil;

public class OAuth2Authorization extends BasicRequestAuthorization {

    private static final long serialVersionUID = 1L;

    private static final String GRANT_TYPE_KEY = OAuth2Constants.GRANT_TYPE;

    private static final String USERNAME_KEY = OAuth2Constants.USERNAME;

    private static final String PASSWORD_KEY = OAuth2Constants.PW;

    private static final String CALLBACK_URL_KEY = OAuth2Constants.REDIRECT_URI;

    private static final String AUTH_URL_KEY = OAuth2Constants.AUTH_URL;

    private static final String ACCESS_TOKEN_URL_KEY = OAuth2Constants.ACCESS_TOKEN_URL;

    private static final String STATE_KEY = OAuth2Constants.STATE;

    private static final String CONSUMER_ID_KEY = OAuth2Constants.CLIENT_ID;

    private static final String CONSUMER_SECRET_KEY = OAuth2Constants.CLIENT_SECRET;

    private static final String AUTHORIZATION_CODE_KEY = OAuth2Constants.AUTHORIZATION_CODE;

    private static final String SCOPE_KEY = OAuth2Constants.SCOPE;

    private static final String ACCESS_TOKEN_KEY = OAuth2Constants.ACCESS_TOKEN;

    private static final String REFRESH_TOKEN_KEY = OAuth2Constants.REFRESH_TOKEN;

    private static final String TOKEN_TYPE_KEY = OAuth2Constants.TOKEN_TYPE;

    public static final String AUTHORIZATION_TYPE = "OAuth2";
    
    public static final String AUTHORIZATION_TYPE_OAUTH_2_0 = OAuth2Constants.AUTHORIZATION_TYPE_OAUTH_2_0;

    public OAuth2Authorization(RequestAuthorization source)
            throws UnsupportedEncodingException, GeneralSecurityException {
        this(source.getAuthorizationInfo());
    }

    public OAuth2Authorization(Map<String, String> props) {
        this.init(props);
    }

    public OAuth2Authorization(OAuth2GrantType grantType, String username, String password, String callbackUrl,
            String authUrl, String accessTokenUrl, String state, String clientId, String clientSecret,
            String authorizationCode, String scope, String accessToken, String refreshToken, String tokenType)
            throws UnsupportedEncodingException, GeneralSecurityException {

        var authorizationInfo = new HashMap<String, String>();

        authorizationInfo.put(GRANT_TYPE_KEY, grantType.name());

        if (StringUtils.isNotBlank(username)) {
            authorizationInfo.put(USERNAME_KEY, username);
        }

        if (StringUtils.isNotBlank(password)) {
            authorizationInfo.put(PASSWORD_KEY, encrypt(password));
        }

        if (StringUtils.isNotBlank(callbackUrl)) {
            authorizationInfo.put(CALLBACK_URL_KEY, callbackUrl);
        }

        if (StringUtils.isNotBlank(authUrl)) {
            authorizationInfo.put(AUTH_URL_KEY, authUrl);
        }

        if (StringUtils.isNotBlank(accessTokenUrl)) {
            authorizationInfo.put(ACCESS_TOKEN_URL_KEY, accessTokenUrl);
        }

        if (StringUtils.isNotBlank(state)) {
            authorizationInfo.put(STATE_KEY, state);
        }

        if (StringUtils.isNotBlank(clientId)) {
            authorizationInfo.put(CONSUMER_ID_KEY, clientId);
        }

        if (StringUtils.isNotBlank(clientSecret)) {
            authorizationInfo.put(CONSUMER_SECRET_KEY, encrypt(clientSecret));
        }

        if (StringUtils.isNotBlank(authorizationCode)) {
            authorizationInfo.put(AUTHORIZATION_CODE_KEY, authorizationCode);
        }

        if (StringUtils.isNotBlank(scope)) {
            authorizationInfo.put(SCOPE_KEY, scope);
        }

        if (StringUtils.isNotBlank(accessToken)) {
            authorizationInfo.put(ACCESS_TOKEN_KEY, accessToken);
        }

        if (StringUtils.isNotBlank(refreshToken)) {
            authorizationInfo.put(REFRESH_TOKEN_KEY, refreshToken);
        }

        if (StringUtils.isNotBlank(tokenType)) {
            authorizationInfo.put(TOKEN_TYPE_KEY, tokenType);
        }

        init(authorizationInfo);
    }

    public static OAuth2Authorization adapt(Map<String, String> properties) {
        if (Objects.isNull(properties)) {
            throw new IllegalArgumentException("The Map<String, String> parameter is required but null");
        }

        return new OAuth2Authorization(properties);
    }
    
    public static OAuth2Authorization adapt(RequestAuthorization requestAuthorization)
            throws UnsupportedEncodingException, GeneralSecurityException {
        if (Objects.isNull(requestAuthorization)) {
            throw new IllegalArgumentException("The RequestAuthorization parameter is required but null");
        }

        return new OAuth2Authorization(requestAuthorization);
    }

    public String getUsername() {
        return getWithEmpty(USERNAME_KEY);
    }

    public String getAuthorizationCode() {
        return getWithEmpty(AUTHORIZATION_CODE_KEY);
    }

    public String getPassword() throws GeneralSecurityException, IOException {
        String encryptedValue = getWithEmpty(PASSWORD_KEY);
        if (StringUtils.isNotBlank(encryptedValue)) {
            CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(encryptedValue);
            return CryptoUtil.decode(cryptoInfo);
        }
        return StringUtils.EMPTY;
    }

    public String getGrantType() {
        return getWithEmpty(GRANT_TYPE_KEY);
    }

    public String getClientId() {
        return getWithEmpty(CONSUMER_ID_KEY);
    }

    public String getClientSecret() throws GeneralSecurityException, IOException {
        String encryptedValue = getWithEmpty(CONSUMER_SECRET_KEY);
        if (StringUtils.isNotBlank(encryptedValue)) {
            CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(encryptedValue);
            return CryptoUtil.decode(cryptoInfo);
        }
        return StringUtils.EMPTY;
    }

    public String getAccessTokenUrl() {
        return getWithEmpty(ACCESS_TOKEN_URL_KEY);
    }

    public String getAuthUrl() {
        return getWithEmpty(AUTH_URL_KEY);
    }

    public String getRedirectUri() {
        return getWithEmpty(CALLBACK_URL_KEY);
    }

    public String getScope() {
        return getWithEmpty(SCOPE_KEY);
    }

    public String getState() {
        return getWithEmpty(STATE_KEY);
    }

    public String getAccessToken() {
        return getWithEmpty(ACCESS_TOKEN_KEY);
    }

    public String getRefreshToken() {
        return getWithEmpty(REFRESH_TOKEN_KEY);
    }

    public String getTokenType() {
        return getWithEmpty(TOKEN_TYPE_KEY);
    }

    public void validate() {
        if (StringUtils.isBlank(getGrantType())) {
            throw new IllegalArgumentException("The parameter 'grant_type' is required but unavailable");
        }
    }

    public static boolean is(RequestAuthorization requestAuthorization) {
        if (Objects.nonNull(requestAuthorization)) {
            var authType = requestAuthorization.getAuthorizationType();
            if (StringUtils.isNotBlank(authType))
                return authType.equalsIgnoreCase(AUTHORIZATION_TYPE) || authType.equalsIgnoreCase(AUTHORIZATION_TYPE_OAUTH_2_0);
        }
        return false;
    }

    public static String encrypt(String vl) throws UnsupportedEncodingException, GeneralSecurityException {
        if (StringUtils.isBlank(vl)) {
            return StringUtils.EMPTY;
        }
        CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(vl);
        return CryptoUtil.encode(cryptoInfo);
    }

    private void init(Map<String, String> info) {
        this.setAuthorizationType(AUTHORIZATION_TYPE_OAUTH_2_0);
        this.setAuthorizationInfo(info);
    }

    private String getWithEmpty(String key) {
        var dict = getAuthorizationInfo();
        if (!dict.containsKey(key))
            return StringUtils.EMPTY;

        var value = dict.get(key);
        return StringUtils.isBlank(value) ? StringUtils.EMPTY : value;
    }

}