package com.kms.katalon.core.webservice.common;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;

import net.lightbody.bmp.core.har.HarEntry;
import net.lightbody.bmp.core.har.HarNameValuePair;
import net.lightbody.bmp.core.har.HarPostData;
import net.lightbody.bmp.core.har.HarRequest;

/**
 * A utility class for generating curl command strings from HAR (HTTP Archive) entries.
 * 
 * <p>This class converts HTTP request information captured in HAR format into executable
 * curl commands that can be used for testing, debugging, or reproducing HTTP requests.
 * It handles various request types including:
 * <ul>
 *   <li>GET, POST, PUT, DELETE, and other HTTP methods</li>
 *   <li>JSON, XML, and other text-based request bodies</li>
 *   <li>URL-encoded form data</li>
 *   <li>Multipart/form-data (including file uploads)</li>
 *   <li>File uploads via HAR comments</li>
 * </ul>
 * 
 * <p>The generated curl commands are properly escaped for shell execution and include
 * all necessary headers, method specifications, and request body data. Large request
 * bodies (exceeding 50KB) are truncated in the output to prevent command line length issues.
 * 
 * <p>This is a utility class with only static methods. It cannot be instantiated.
 * 
 * <p><b>Main Method:</b>
 * <ul>
 *   <li>{@link #generateCurlCommand(HarEntry)} - Generates a curl command from a HAR entry</li>
 * </ul>
 * 
 * <p><b>Usage Example:</b>
 * <pre>{@code
 * // Given a HAR entry from a captured HTTP request
 * HarEntry entry = ...; // obtained from HAR logger or proxy
 * 
 * // Generate the curl command
 * String curlCommand = CurlCommandGenerator.generateCurlCommand(entry);
 * 
 * // Output: curl -X POST -H 'Content-Type: application/json' -d '{"key":"value"}' 'https://api.example.com/endpoint'
 * System.out.println(curlCommand);
 * }</pre>
 * 
 * <p><b>Example with file upload:</b>
 * <pre>{@code
 * // For multipart form data with file uploads, the generated command will use -F flags:
 * // curl -X POST -F 'file=@/path/to/file.txt' -H 'Authorization: Bearer token' 'https://api.example.com/upload'
 * }</pre>
 * 
 * @author Katalon Studio
 * @since [10.4.2]
 */
public class CurlCommandGenerator {

    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final int MAX_BODY_SIZE_BYTES = 50 * 1024; // 50KB limit
    private static final String FILE_UPLOAD_PREFIX = "FILE_UPLOAD:";
    private static final String NEWLINE_PATTERN = "\r?\n";
    private static final String BOUNDARY_PREFIX = "--";
    private static final String BOUNDARY_SUFFIX = "--";
    private static final String SHELL_ESCAPED_SINGLE_QUOTE = "'\\''";


    private CurlCommandGenerator(){}  // Private constructor to prevent instantiation

    public static String generateCurlCommand(HarEntry harEntry) {
        if (harEntry == null || harEntry.getRequest() == null) {
            return "";
        }
 
        HarRequest request = harEntry.getRequest();
        List<String> curlParts = new ArrayList<>();
        curlParts.add("curl");
        curlParts.add("--location");
 
        addMethodIfNotGet(curlParts, request.getMethod());
 
        Map<String, String> uniqueHeaders = collectHeaders(request.getHeaders());
 
        boolean handled = handlePostData(curlParts, uniqueHeaders, request);
 
        addHeadersAndUrl(curlParts, uniqueHeaders, request);
        return String.join(" ", curlParts);
    }

    private static void addMethodIfNotGet(List<String> curlParts, String method) {
        if (!"GET".equalsIgnoreCase(method)) {
            curlParts.add("--request");
            curlParts.add(method);
        }
    }

    private static Map<String, String> collectHeaders(List<HarNameValuePair> headers) {
        Map<String, String> uniqueHeaders = new LinkedHashMap<>();
        for (HarNameValuePair header : headers) {
            if (header.getName() == null || header.getValue() == null) {
                continue;
            }

            String key = header.getName().toLowerCase();
            String sanitized = header.getValue()
                    .replace("\n", "")
                    .replace("\r", "")
                    .replace("'", SHELL_ESCAPED_SINGLE_QUOTE);
            uniqueHeaders.putIfAbsent(key, String.format("%s: %s", header.getName(), sanitized));
        }
        return uniqueHeaders;
    }

    private static boolean handlePostData(List<String> curlParts, Map<String, String> uniqueHeaders, HarRequest request) {
        HarPostData postData = request.getPostData();
        if (postData == null) {
            return false;
        }

        // PRIORITY 1: Handle file upload from HAR comment
        if (handleFileUploadComment(curlParts, uniqueHeaders, postData, request)) {
            return true;
        }

        // PRIORITY 2: Handle multipart/form-data
        if (handleMultipartFormData(curlParts, uniqueHeaders, postData)) {
            return true;
        }

        // Handle URL-encoded form data
        if (handleUrlEncodedFormData(curlParts, postData)) {
            return true;
        }

        // Handle text-based body (JSON/XML/SOAP)
        handleTextBasedBody(curlParts, postData);
        return true;
    }

    private static boolean handleFileUploadComment(List<String> curlParts, Map<String, String> uniqueHeaders, 
                                                    HarPostData postData, HarRequest request) {
        String comment = postData.getComment();
        if (comment == null || !comment.startsWith(FILE_UPLOAD_PREFIX)) {
            return false;
        }

        String filePath = comment.substring(FILE_UPLOAD_PREFIX.length()).trim();
        if (filePath.isEmpty()) {
            return false;
        }

        curlParts.add("--form");
        curlParts.add(String.format("'file=@%s'", filePath.replace("'", SHELL_ESCAPED_SINGLE_QUOTE))); 
        uniqueHeaders.remove("content-type");
        return true;
    }

    private static boolean handleMultipartFormData(List<String> curlParts, Map<String, String> uniqueHeaders, 
                                                    HarPostData postData) {
        String mimeType = postData.getMimeType();
        if (mimeType == null || !mimeType.toLowerCase().startsWith("multipart/form-data")) {
            return false;
        }

        String body = postData.getText();
        if (body == null || body.isEmpty()) {
            return false;
        }

        List<MultipartField> fields = parseMultipartFormData(body);
        for (MultipartField field : fields) {
            curlParts.add("--form");
            String escapedName = safe(field.name());
            String escapedValue = safe(field.value());
            if (field.isFile()) {
                curlParts.add(String.format("'%s=@%s'", escapedName, escapedValue));
            } else {
                curlParts.add(String.format("'%s=%s'", escapedName, escapedValue));
            }
        }

        return true;
    }

    private static boolean handleUrlEncodedFormData(List<String> curlParts, HarPostData postData) {
        String mimeType = postData.getMimeType();
        boolean isUrlEncoded = mimeType != null && 
                               mimeType.toLowerCase().contains("application/x-www-form-urlencoded");
        
        if (!isUrlEncoded) {
            return false;
        }

        // First, try to use params if available
        if (postData.getParams() != null && !postData.getParams().isEmpty()) {
            String formData = postData.getParams().stream()
                    .filter(p -> p.getName() != null && p.getValue() != null)
                    .map(p -> String.format("%s=%s", p.getName(), p.getValue()))
                    .collect(Collectors.joining("&"));
            
            if (!formData.isEmpty()) {
                curlParts.add("--data-urlencode");
                curlParts.add(String.format("'%s'", formData.replace("'", SHELL_ESCAPED_SINGLE_QUOTE)));
                return true;
            }
        }

        // Fallback: use text field if params are not available
        String text = postData.getText();
        if (text != null && !text.trim().isEmpty()) {
            curlParts.add("--data-urlencode");
            curlParts.add(String.format("'%s'", text.replace("'", SHELL_ESCAPED_SINGLE_QUOTE)));
            return true;
        }

        return false;
    }

    private static void handleTextBasedBody(List<String> curlParts, HarPostData postData) {
        String body = postData.getText();
        if (body == null || body.isEmpty()) {
            return;
        }

        if (body.getBytes().length > MAX_BODY_SIZE_BYTES) {
            curlParts.add("--data");
            curlParts.add("'[Request body truncated: exceeds 50KB limit]'");
            return;
        }

        String processedBody = processBodyContent(body, postData.getMimeType());
        curlParts.add("--data");
        curlParts.add(String.format("'%s'", processedBody.replace("'", SHELL_ESCAPED_SINGLE_QUOTE))); 
    }

    private static String processBodyContent(String body, String mimeType) {
        if ((mimeType != null && mimeType.toLowerCase().contains("json")) || looksLikeJson(body)) {
            return minifyJson(body);
        }
        return body;
    }
    
    // shell escaping
    private static String safe(String value) {
        return value == null ? "" : value.replace("'", SHELL_ESCAPED_SINGLE_QUOTE);
    }

    //Utility: add headers and URL
    private static void addHeadersAndUrl(List<String> curlParts, Map<String, String> headers, HarRequest request) {
        for (String headerValue : headers.values()) {
            curlParts.add("--header");
            curlParts.add(String.format("'%s'", headerValue));
        }

        String url = request.getUrl();
        if (url != null) {
            curlParts.add(String.format("'%s'", url.replace("'", SHELL_ESCAPED_SINGLE_QUOTE)));
        }
    }
    
    //Utility: detect JSON
    private static boolean looksLikeJson(String body) {
        if (body == null) return false;
        String trimmed = body.trim();
        return trimmed.startsWith("{") || trimmed.startsWith("[");
    }

    //Utility: minify JSON using Jackson
    private static String minifyJson(String json) {
        try {
            Object parsed = objectMapper.readValue(json, Object.class);
            return objectMapper.writeValueAsString(parsed);
        } catch (JsonProcessingException e) {
            return json.replaceAll("\\s+", " ").trim();
        }
    }

    //Minimal multipart parser
    private static List<MultipartField> parseMultipartFormData(String body) {
        List<MultipartField> fields = new ArrayList<>();
        String boundary = null;
        String[] lines = body.split(NEWLINE_PATTERN);
        for (String line : lines) {
            if (line.startsWith(BOUNDARY_PREFIX)) {
                boundary = line.substring(2);
                break;
            }
        }
        if (boundary == null) return fields;
        String[] parts = body.split(BOUNDARY_PREFIX + java.util.regex.Pattern.quote(boundary));
        for (String part : parts) {
            part = part.trim();
            if (part.isEmpty() || part.equals(BOUNDARY_SUFFIX)) continue;
            String[] partLines = part.split(NEWLINE_PATTERN);
            String contentDisposition = partLines[0];
            if (!contentDisposition.startsWith("Content-Disposition:")) continue;
            String name = extractFieldName(contentDisposition);
            boolean isFile = contentDisposition.contains("filename=");
            String value = part.substring(part.indexOf("\r\n\r\n") + 4).trim();
            fields.add(new MultipartField(name, value, isFile));
        }
        return fields;
    }

    private static String extractFieldName(String contentDisposition) {
        int nameIndex = contentDisposition.indexOf("name=\"");
        if (nameIndex == -1) return null;
        int start = nameIndex + 6;
        int end = contentDisposition.indexOf("\"", start);
        if (end == -1) return null;
        return contentDisposition.substring(start, end);
    }
    
    private record MultipartField(String name, String value, boolean isFile) {}
}
