package com.kms.katalon.core.webui.driver.chrome;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.manager.SeleniumManager;
import org.openqa.selenium.manager.SeleniumManagerOutput.Result;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.webui.common.internal.SeleniumBrowserInfo;
import com.kms.katalon.core.webui.driver.DriverFactory;
import com.kms.katalon.core.webui.exception.SeleniumManagerException;
import com.kms.katalon.core.webui.util.FileExcutableUtil;
import com.kms.katalon.core.webui.util.OSUtil;

public class ChromeDriverUtil {
    protected static final KeywordLogger logger = KeywordLogger.getInstance(DriverFactory.class);
    private static final Logger LOGGER = LoggerFactory.getLogger(ChromeDriverUtil.class);

    public static String getChromeDriverPath() {
        if (OSUtil.isWindows()) {
            if (OSUtil.is64Bit()) {
                File location = new File(RunConfiguration.getProjectDir(),
                        "Include/drivers/chromedriver_win64/chromedriver.exe");
                if (location.exists()) {
                    logger.logInfo("Custom Chrome detected at location: " + location.getAbsolutePath());
                    return location.getAbsolutePath();
                }
            } else {
                File location = new File(RunConfiguration.getProjectDir(),
                        "Include/drivers/chromedriver_win32/chromedriver.exe");
                if (location.exists()) {
                    logger.logInfo(
                            "Custom Chrome detected at location: " + location.getAbsolutePath());
                    return location.getAbsolutePath();
                }
            }
        } else if (OSUtil.isMac()) {
            File location = new File(RunConfiguration.getProjectDir(),
                    "Include/drivers/chromedriver_mac64/chromedriver");
            String chromeDriverPath = location.getAbsolutePath();
            if (location.exists()) {
                try {
                    logger.logInfo("Custom Chrome detected at location: " + location.getAbsolutePath());
                    FileExcutableUtil.makeFileExecutable(chromeDriverPath);
                } catch (IOException e) {
                    logger.logInfo("Cannot make Chrome driver file: " + location.getAbsolutePath()
                            + "executable");
                }
                return location.getAbsolutePath();
            }
        } else {
            if (OSUtil.is64Bit()) {
                File location = new File(RunConfiguration.getProjectDir(),
                        "Include/drivers/chromedriver_linux64/chromedriver");
                String chromeDriverPath = location.getAbsolutePath();
                if (location.exists()) {
                    try {
                        logger.logInfo(
                                "Custom Chrome detected at location: " + location.getAbsolutePath());
                        FileExcutableUtil.makeFileExecutable(chromeDriverPath);
                    } catch (IOException e) {
                        logger.logInfo("Cannot make Chrome driver file:" + location.getAbsolutePath()
                                + " executable");
                    }
                    return location.getAbsolutePath();
                }
            } else {
                File location = new File(RunConfiguration.getProjectDir(),
                        "Include/drivers/chromedriver_linux32/chromedriver");
                String chromeDriverPath = location.getAbsolutePath();
                if (location.exists()) {
                    try {
                        logger.logInfo(
                                "Custom Chrome detected at location: " + location.getAbsolutePath());
                        FileExcutableUtil.makeFileExecutable(chromeDriverPath);
                    } catch (IOException e) {
                        logger.logInfo("Cannot make Chrome driver file:" + location.getAbsolutePath()
                                + "executable");
                    }
                    return location.getAbsolutePath();
                }
            }
        }

        return RunConfiguration.getDriverSystemProperty(DriverFactory.WEB_UI_DRIVER_PROPERTY, DriverFactory.CHROME_DRIVER_PATH_PROPERTY);
    }

    public static SeleniumBrowserInfo getSeleniumBrowserInfo(String browserVersion) {
        LOGGER.info("Checking and download if needed Chrome browser and driver for Chrome for Testing version %s (This may take a while in case of downloading needed)".formatted(StringUtils.isNotBlank(browserVersion) ? browserVersion : "latest"));
        List<String> args = new ArrayList<>();
        args.add("--browser");
        args.add("chrome");

        if (StringUtils.isNotBlank(browserVersion)) {
            args.add("--browser-version");
            args.add(browserVersion);
        }

        /*
         * Force to download browser even when a browser is already installed in the system
         */
        args.add("--force-browser-download");

        try {
            long startTime = System.currentTimeMillis();
            SeleniumManager instance = SeleniumManager.getInstance();
            Result result = instance.getBinaryPaths(args);

            LOGGER.info("Checking Chrome browser and driver completed in %s ms".formatted(System.currentTimeMillis() - startTime));

            if (null == result) {
                throw new SeleniumManagerException("Unknown error occurred during checking Chrome browser and driver");
            }

            if (0 == result.getCode()) {
                String driverPath = result.getDriverPath();
                String browserPath = result.getBrowserPath();
                return new SeleniumBrowserInfo(driverPath, browserPath);
            } else {
                throw new SeleniumManagerException("Checking Chrome browser and driver got error | code = %s | message = %s".formatted(result.getCode(), result.getMessage()));
            }
        } catch (Exception e) {
            LOGGER.error("Exception while checking checking Chrome browser and driver", e);
            throw e;
        }
    }
}
