package com.kms.katalon.core.webui.keyword.builtin

import org.openqa.selenium.JavascriptExecutor
import org.openqa.selenium.WebDriver
import org.openqa.selenium.support.ui.ExpectedCondition
import org.openqa.selenium.support.ui.WebDriverWait

import com.google.common.base.Predicates.InstanceOfPredicate
import com.kms.katalon.core.annotation.internal.Action
import com.kms.katalon.core.configuration.RunConfiguration
import com.kms.katalon.core.exception.StepFailedException
import com.kms.katalon.core.model.FailureHandling
import com.kms.katalon.core.webui.common.WebUiCommonHelper
import com.kms.katalon.core.webui.constants.CoreWebuiMessageConstants
import com.kms.katalon.core.webui.constants.StringConstants
import com.kms.katalon.core.webui.driver.DriverFactory
import com.kms.katalon.core.webui.keyword.internal.WebUIAbstractKeyword
import com.kms.katalon.core.webui.keyword.internal.WebUIKeywordMain

import groovy.transform.CompileStatic

import java.time.Duration

/**
 * Katalon keyword for waiting until Angular applications are fully loaded and stable.
 * Supports both AngularJS (1.x) and Angular (2+) frameworks.
 * Handles main document and iframe Angular applications with cross-origin security.
 */
@Action(value = "waitForAngularLoad")
public class WaitForAngularLoadKeyword extends WebUIAbstractKeyword {

    /**
     * Main entry point - handles different parameter combinations
     * Supports overloaded calls with timeout and/or failure handling
     */
    @Override
    public Object execute(Object... params) {
        switch (params.length) {
            case 0:
                // No parameters - use default timeout and failure handling
                return waitForAgularLoad(RunConfiguration.getElementTimeoutForWeb(), RunConfiguration.getDefaultFailureHandling())
            case 1:
                // Single parameter - either timeout (Integer) or failure handling (FailureHandling)
                if (params[0] instanceof Integer) {
                    return waitForAgularLoad(params[0], RunConfiguration.getDefaultFailureHandling())
                }
                if (params[0] instanceof FailureHandling) {
                    return waitForAgularLoad(RunConfiguration.getElementTimeoutForWeb(), params[0])
                }
            case 2:
                // Two parameters - timeout and failure handling
                return waitForAgularLoad(params[0], params[1])
        }
    }

    /**
     * Waits until Angular elements loaded in a <code>timeout</code> seconds.
     * @param timeout timeout value in seconds.
     * @param flowControl
     * @return true if Angular/AJAX is ready. Otherwise, false.
     * @throws StepFailedException If browser has not started yet or jQuery is not ready
     */
    public boolean waitForAgularLoad(int timeout, FailureHandling flowControl) throws StepFailedException {
        return WebUIKeywordMain.runKeyword({
            // Get the current WebDriver instance
            WebDriver webDriver = DriverFactory.getWebDriver()
            if (webDriver == null) {
                throw new StepFailedException(CoreWebuiMessageConstants.EXC_BROWSER_IS_NOT_OPENED)
            }

            // AngularJS (1.x) detection script - checks for window.angular object
            String angularJSDetectScript = '''return (window.angular != null && window.angular.version != null && window.angular.version.full != null)'''
            
            // Modern Angular (2+) detection script - comprehensive detection for all versions
            String angularDetectScript = '''
                // Angular detection for ALL versions - main document and iframes
                function checkMainDocumentAngular() {
                    try{
                         // Method 1: Check for Angular global objects (ng namespace)
                        if (window.ng != null) {
                            console.log('Angular detect => window.ng found');
                            return true;
                        }
                    
                        // Method 2: Check DOM for Angular version attributes
                        var angularSelectors = [
                            '[ng-version]',              // Standard Angular version attribute
                            '[data-ng-version]'          // Data attribute version (alternative)
                        ];
                    
                        for (var i = 0; i < angularSelectors.length; i++) {
                            var elements = document.querySelectorAll(angularSelectors[i]);
                            if (elements.length > 0) {
                                console.log('Angular detect => Found: ' + angularSelectors[i] + ' (' + elements.length + ' elements)');
                                return true;
                            }
                        }

                        // Method 3: Fallback - scan HTML content for Angular patterns
                        var htmlContent = document.documentElement.outerHTML || '';
                        if (htmlContent.includes('ng-version') || 
                            htmlContent.includes('data-ng-version')) {
                            console.log('Angular detect => Angular patterns found in HTML');
                            return true;
                        }

                        console.log('Angular detect => No Angular patterns found in main document'); 
                    } catch (error) {
                        console.log('Angular detect => Error accessing main document:', error);
                    }

                    return false;            
                }
                
                // Security check - verify iframe is same-origin before accessing
                function isSameOrigin(iframe) {
                    try {
                        // Attempt to access href property - throws error for cross-origin
                        var href = iframe.contentWindow.location.href;
                        return true;
                    } catch (error) {
                        console.log('IFrame Angular detect => Cross-origin access denied for iframe:', iframe, error);
                        return false;
                    }
                }
                
                // IFrame Angular detection - safely handles cross-origin security
                function checkIFrameAngular() {
                    var iframes = document.querySelectorAll('iframe');
                    var foundAngular = false;
                    
                    for (var i = 0; i < iframes.length; i++) {
                        var iframe = iframes[i];
                        
                        // Skip iframes without content
                        if (!iframe.src && !iframe.srcdoc) continue;
                        
                        try {
                            // Security check - only process same-origin iframes
                            if (!isSameOrigin(iframe)) {
                                // Cross-origin iframe - cannot access due to security restrictions
                                continue;
                            }
                            
                            // Safe to access same-origin iframe content
                            var iframeDoc = null;
                            try {
                                iframeDoc = iframe.contentDocument || (iframe.contentWindow && iframe.contentWindow.document);
                            } catch (accessError) {
                                // Document access failed - skip this iframe
                                console.log('IFrame Angular detect => Cannot access iframe document:', iframe, accessError);
                                continue;
                            }
                            
                            if (iframeDoc && iframeDoc.querySelector) {
                                var hasAngularWindow = false;
                                var hasAngularElements = false;
                                
                                // Check iframe window objects for Angular
                                try {
                                    var iframeWindow = iframe.contentWindow;
                                    hasAngularWindow = iframeWindow && iframeWindow.ng;
                                } catch (accessWindowError) {
                                    console.log('IFrame Angular detect => Cannot access iframe window:', iframe, accessWindowError);
                                }
                                
                                // Check iframe DOM for Angular elements
                                try {
                                    var iframeSelectors = [
                                        '[ng-version]', '[data-ng-version]'
                                    ];
        
                                    for (var j = 0; j < iframeSelectors.length; j++) {
                                        var elements = iframeDoc.querySelectorAll(iframeSelectors[j]);
                                        if (elements.length > 0) {
                                            console.log('IFrame Angular detect => Found: ' + iframeSelectors[j] + ' (' + elements.length + ' elements)');
                                            hasAngularElements = true;
                                            break;
                                        }
                                    }
                                } catch (elementsError) {
                                    console.log('IFrame Angular detect => Cannot query iframe document:', iframe, elementsError);
                                }
                                
                                if (hasAngularWindow || hasAngularElements) {
                                    foundAngular = true;
                                }
                            }
                        } catch (exceptionError) {
                            // General error handling - skip problematic iframes
                            console.log('IFrame Angular detect => Error processing iframe:', iframe, exceptionError);
                            continue;
                        }
                    }
                    
                    return foundAngular;
                }

                // Execute detection in order: main document first, then iframes
                if (checkMainDocumentAngular()) return true;
                if (checkIFrameAngular()) return true;

                return false;
            '''

            // Execute detection scripts
            JavascriptExecutor jsExec = (JavascriptExecutor) webDriver
            boolean isAngularJS = (Boolean) jsExec.executeScript(angularJSDetectScript)
            boolean isAngular = (Boolean) jsExec.executeScript(angularDetectScript)

            // If no Angular framework detected, log warning and return
            if (!isAngularJS && !isAngular) {
                logger.logWarning(CoreWebuiMessageConstants.KW_MSG_ANGULAR_NOT_USED)
                return false
            }

            // Setup WebDriverWait with validated timeout
            timeout = WebUiCommonHelper.checkTimeout(timeout)
            WebDriverWait wait = new WebDriverWait(webDriver, Duration.ofSeconds(timeout))
            
            // Custom ExpectedCondition to wait for Angular stability
            ExpectedCondition jQueryLoadExpectation = new ExpectedCondition<Boolean>() {
                        public Boolean apply(WebDriver driver) {
                            // AngularJS (1.x) stability check - wait for HTTP requests to complete
                            String waitForAngularJSAllRequestsEnded = '''
                                return document.readyState === 'complete'
                                    && window.angular.element(document.body)
                                        .injector()
                                        .get('$http')
                                        .pendingRequests.length === 0
                            '''
                            
                            // AngularJS testability API - wait for all async operations
                            String waitForAngularJSStabled = '''
                                var callback = arguments[arguments.length - 1];
                                try {
                                    var testability = window.angular.getTestability(document);
                                    testability.whenStable(() => callback(true));
                                } catch (error) {
                                    callback(null)
                                }
                            '''
                            
                            // Angular (2+) stability check - comprehensive document and iframe checking
                            String waitForAngularStabled = '''
                                // Check Angular stability in main document
                                function checkAngularStabilityInDocument() {
                                    try {
                                        // Ensure document is fully loaded
                                        if (document.readyState !== 'complete') return false;
                                        
                                        // Find Angular version elements (indicates Angular app presence)
                                       var ngVersionElements = document.querySelectorAll('[ng-version], [data-ng-version]');
                                        if (ngVersionElements.length > 0) {
                                            // Check if Angular has bootstrapped content (has child elements)
                                            var hasBootstrappedContent = false;
                                            for (var i = 0; i < ngVersionElements.length; i++) {
                                                if (ngVersionElements[i].children.length > 0) {
                                                    hasBootstrappedContent = true;
                                                    break;
                                                }
                                            }
                                            return hasBootstrappedContent;
                                        }
 
                                        // Fallback - if no Angular elements found, check document ready state
                                        return document.readyState === 'complete';
                                    } catch (error) {
                                        console.log('Error checking Angular stability in document:', error);                    
                                    }

                                    return false;
                                }

                                // Check Angular stability in iframes (same-origin only)
                                function checkAngularStabilityInIFrame() {
                                     // Iterate through all iframes
                                    var iframes = document.querySelectorAll('iframe');
                                    for (var i = 0; i < iframes.length; i++) {
                                        try {
                                            // Access iframe document (same-origin only)
                                            var iframeDoc = iframes[i].contentDocument || iframes[i].contentWindow.document;
                                            if (iframeDoc) {
                                                // Check if this iframe contains Angular
                                                var hasAngular = (iframeDoc.defaultView && iframeDoc.defaultView.ng != null) || 
                                                               (iframeDoc.querySelector('[ng-version], [data-ng-version]') != null);
                                                
                                                if (hasAngular) {
                                                    return true;
                                                }
                                            }
                                        } catch (error) {
                                            // Cross-origin iframe or access error - skip safely
                                            console.log('Error checking Angular stability in IFrame:', error);
                                        }   
                                    }

                                    return false;
                                }
                                
                                // Execute stability checks
                                if (checkAngularStabilityInDocument()) return true;
                                if (checkAngularStabilityInIFrame()) return true;

                                return false;
                            '''

                            // Execute appropriate stability check based on Angular version
                            if (isAngularJS) {
                                // AngularJS - check both HTTP requests and testability API
                                Boolean isAllRequestsEnded = jsExec.executeScript(waitForAngularJSAllRequestsEnded)
                                Boolean isAngularStabled = jsExec.executeAsyncScript(waitForAngularJSStabled)
                                return isAllRequestsEnded && isAngularStabled
                            }
                            if (isAngular) {
                                // Angular (2+) - check application bootstrap stability
                                Boolean isAngularStabled = jsExec.executeScript(waitForAngularStabled)
                                return isAngularStabled
                            }

                            return false
                        }
                    }

            // Wait for Angular stability with timeout
            boolean isAngularLoaded = wait.until(jQueryLoadExpectation)
            if (!isAngularLoaded) {
                // Angular not ready within timeout - handle failure
                WebUIKeywordMain.stepFailedWithReason(CoreWebuiMessageConstants.KW_LOG_ANGULAR_NOT_READY, flowControl, "timeout", true)
                return false
            }
            
            // Success - Angular is ready
            logger.logPassed(CoreWebuiMessageConstants.KW_LOG_ANGULAR_READY)
            return isAngularLoaded
        }, flowControl, RunConfiguration.getTakeScreenshotOption(), CoreWebuiMessageConstants.KW_MSG_CANNOT_WAIT_FOR_ANGULAR_LOAD)
    }
}
