package com.kms.katalon.core.webui.model;

import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import com.kms.katalon.core.webui.common.WebUiCommonHelper;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.WebElement;

import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.keyword.internal.KeywordExecutionContext;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.testobject.BrokenTestObject;
import com.kms.katalon.core.testobject.SelectorMethod;
import com.kms.katalon.core.testobject.TestObject;
import com.kms.katalon.core.webui.common.FindElementsResult;
import com.kms.katalon.core.webui.common.WebUICommonScripts;
import com.kms.katalon.core.webui.common.internal.SelfHealingController;
import com.kms.katalon.core.webui.constants.StringConstants;
import com.kms.katalon.core.webui.driver.DriverFactory;
import com.kms.katalon.core.webui.util.FileUtil;

public class WebUiElementFinderWithSelfHealing implements SeleniumElementFinder<FindElementParams> {
    private final KeywordLogger logger = KeywordLogger.getInstance(WebUiElementFinderWithSelfHealing.class);

    private WebUiElementFinder basicElementFinder = new WebUiElementFinder();

    @Override
    public WebElement findElement(FindElementParams params) throws Exception {
        List<WebElement> foundElements = findElements(params);
        return foundElements != null && !foundElements.isEmpty() ? foundElements.get(0) : null;
    }

    @Override
    public List<WebElement> findElements(FindElementParams params) throws Exception {
        SelfHealingController.setLogger(logger);
        WebDriver webDriver = params.getWebDriver();
        TestObject testObject = params.getTestObject();
        try {
            boolean isWebPlatform = KeywordExecutionContext.isRunningWebUI();
            String runningKeyword = KeywordExecutionContext.getRunningKeyword();

            List<String> excludeKeywords = RunConfiguration.getExcludedWebUIKeywordsFromSelfHealing();
            if (!isWebPlatform || excludeKeywords.contains(runningKeyword)) {
                return Collections.emptyList();
            }

            SelfHealingController.logWarning(MessageFormat.format(
                    StringConstants.KW_LOG_INFO_DEFAULT_LOCATOR_FAILED_TRY_SELF_HEALING, testObject.getObjectId()));

            if (isInShadowDOM(testObject)) {
                return findElementsInShadowDOM(webDriver, testObject);
            }

            return performSelfHealing(webDriver, testObject);
        } catch (Exception exception) {
            SelfHealingController.logWarning(exception.getMessage(), exception);
        }
        return Collections.emptyList();
    }

    private List<WebElement> performSelfHealing(WebDriver webDriver, TestObject testObject) throws Exception {
        prepareSelfHealingDataFile();
        
        List<WebElement> elementsFromCandidates = tryHealingWithCandidates(webDriver, testObject);
        if (elementsFromCandidates != null && !elementsFromCandidates.isEmpty()) {
            return elementsFromCandidates;
        }
        
        return tryHealingWithAlternativeLocators(webDriver, testObject);
    }

    private void prepareSelfHealingDataFile() {
        String dataFileAtReportFolder = SelfHealingController
                .getSelfHealingDataFilePath(RunConfiguration.getReportFolder());
        SelfHealingController.prepareDataFile(dataFileAtReportFolder);
    }

    private List<WebElement> tryHealingWithCandidates(WebDriver webDriver, TestObject testObject) throws Exception {
        Set<BrokenTestObject> healingCandidates = SelfHealingController.findBrokenTestObjects(testObject);
        if (healingCandidates == null || healingCandidates.isEmpty()) {
            return null;
        }

        for (BrokenTestObject healingCandidate : healingCandidates) {
            List<WebElement> foundElements = tryHealingCandidate(webDriver, testObject, healingCandidate);
            if (foundElements != null && !foundElements.isEmpty()) {
                return foundElements;
            }
        }
        return null;
    }

    private List<WebElement> tryHealingCandidate(WebDriver webDriver, TestObject testObject, 
            BrokenTestObject healingCandidate) throws Exception {
        TestObject healedTestObject = SelfHealingController.healTestObject(testObject, healingCandidate);
        
        if (!isValidHealedLocator(healedTestObject, testObject)) {
            return null;
        }
        
        FindElementParams findHealedObjectParams = new FindElementParams(webDriver, healedTestObject);
        List<WebElement> foundElements = basicElementFinder.findElements(findHealedObjectParams);

        if (foundElements != null && !foundElements.isEmpty()) {
            FindElementsResult findResultForReport = createFindResultWithScreenshot(foundElements, 
                    healedTestObject, healingCandidate);
            registerBrokenTestObjectAndLog(findResultForReport, testObject);
            return foundElements;
        }
        
        return null;
    }

    private boolean isValidHealedLocator(TestObject healedTestObject, TestObject originalTestObject) {
        SelectorMethod healedMethod = healedTestObject.getSelectorMethod();
        String healedLocator = healedTestObject.getSelectorCollection().get(healedMethod);
        
        if (StringUtils.isBlank(healedLocator)) {
            logger.logDebug(MessageFormat.format(StringConstants.KW_LOG_DEBUG_HEALING_CANDIDATE_EMPTY_LOCATOR, 
                    originalTestObject.getObjectId()));
            return false;
        }
        return true;
    }

    private FindElementsResult createFindResultWithScreenshot(List<WebElement> foundElements, 
            TestObject healedTestObject, BrokenTestObject healingCandidate) {
        FindElementsResult findResultForReport = FindElementsResult.from(foundElements, healedTestObject);
        String screenshotPath = SelfHealingController
                .getScreenshotAbsolutePath(healingCandidate.getPathToScreenshot());
        findResultForReport.setScreenshot(screenshotPath);
        return findResultForReport;
    }

    private List<WebElement> tryHealingWithAlternativeLocators(WebDriver webDriver, TestObject testObject) {
        SelectorMethod elementMethod = testObject.getSelectorMethod();
        WebUiLocatorProvider locatorProvider = new WebUiLocatorProvider();

        while (locatorProvider.hasNext()) {
            SelectorMethod currentMethod = locatorProvider.getNextLocator();
            
            if (shouldTryLocatorMethod(currentMethod, elementMethod)) {
                List<WebElement> foundElements = tryAlternativeLocatorMethod(webDriver, testObject, 
                        currentMethod, elementMethod);
                if (foundElements != null && !foundElements.isEmpty()) {
                    return foundElements;
                }
            }
        }
        return Collections.emptyList();
    }

    private boolean shouldTryLocatorMethod(SelectorMethod currentMethod, SelectorMethod elementMethod) {
        return currentMethod != elementMethod || currentMethod == SelectorMethod.XPATH;
    }

    private List<WebElement> tryAlternativeLocatorMethod(WebDriver webDriver, TestObject testObject, 
            SelectorMethod currentMethod, SelectorMethod elementMethod) {
        try {
            String locatorValue = testObject.getSelectorCollection().get(currentMethod);
            if (StringUtils.isBlank(locatorValue)) {
                logger.logDebug(MessageFormat.format(StringConstants.KW_LOG_DEBUG_SKIPPING_METHOD_EMPTY_LOCATOR,
                        currentMethod.name(), testObject.getObjectId()));
                return null;
            }
            
            FindElementsResult findResult = findElementsWithMethod(webDriver, testObject, 
                    currentMethod, elementMethod);

            if (findResult != null && !findResult.isEmpty()) {
                registerBrokenTestObjectAndLog(findResult, testObject);
                return findResult.getElements();
            }
        } catch (Exception exception) {
            SelfHealingController.logWarning(exception.getMessage(), exception);
        }
        return null;
    }

    private FindElementsResult findElementsWithMethod(WebDriver webDriver, TestObject testObject, 
            SelectorMethod currentMethod, SelectorMethod elementMethod) throws Exception {
        FindElementsResult findResult = null;
        boolean hasFindWithDefaultXPath = elementMethod == SelectorMethod.XPATH;
        
        if (currentMethod == SelectorMethod.XPATH && !hasFindWithDefaultXPath) {
            findResult = basicElementFinder.findElementsBySelectedMethod(webDriver, testObject,
                    currentMethod, false);
        }
        
        if (findResult == null || findResult.isEmpty()) {
            findResult = basicElementFinder.findElementsBySelectedMethod(webDriver, testObject,
                    currentMethod, true);
        }
        
        return findResult;
    }


    private boolean isInShadowDOM(TestObject testObject) {
        return testObject.getParentObject() != null && testObject.isParentObjectShadowRoot();
    }


    private List<WebElement> findElementsInShadowDOM(WebDriver webDriver, TestObject testObject) throws Exception {
        try {
            FindElementsResult result = basicElementFinder.findElementsBySelectedMethod(webDriver, testObject);
            if (result != null && !result.isEmpty()) {
                return result.getElements();
            }
        } catch (Exception e) {
            logger.logDebug(MessageFormat.format(StringConstants.KW_LOG_DEBUG_SHADOW_DOM_FIND_FAILED,
                    testObject.getObjectId(), e.getMessage()));
        }
        
        List<WebElement> result = findElementsWithParentHealing(webDriver, testObject);
        if (result != null && !result.isEmpty()) {
            return result;
        }

        // Fallback to healing child elements with a smart locator if parent healing fails
        WebUiLocatorProvider locatorProvider = new WebUiLocatorProvider();
        if (WebUiCommonHelper.ableExecuteWithSmartLocator(webDriver)
                && locatorProvider.isLocatorEnabled(SelectorMethod.SMART_LOCATOR)) {
            FindElementsResult findElementsResult = basicElementFinder.findElementsBySelectedMethod(webDriver,
                    testObject, SelectorMethod.SMART_LOCATOR, false);
            if (findElementsResult != null && !findElementsResult.isEmpty()) {
                registerBrokenTestObjectAndLog(findElementsResult, testObject);
                return findElementsResult.getElements();
            }
        }

        return Collections.emptyList();
    }


    private List<WebElement> findElementsWithParentHealing(WebDriver webDriver, TestObject testObject) throws Exception {
        final TestObject parentObject = testObject.getParentObject();
        if (parentObject == null) {
            return Collections.emptyList();
        }

        SelectorMethod parentMethod = parentObject.getSelectorMethod();
        String parentLocator = parentObject.getSelectorCollection().get(parentMethod);
        if (StringUtils.isBlank(parentLocator)) {
            logger.logDebug(StringConstants.KW_LOG_DEBUG_PARENT_EMPTY_LOCATOR_CANNOT_HEAL);
            return Collections.emptyList();
        }

        logger.logDebug(MessageFormat.format(StringConstants.KW_LOG_DEBUG_APPLYING_SELF_HEALING_TO_SHADOW_PARENT,
                parentObject.getObjectId()));

        List<WebElement> healedParentElements = performSelfHealing(webDriver, parentObject);
        
        if (healedParentElements != null && !healedParentElements.isEmpty()) {
            WebElement healedParentElement = healedParentElements.get(0);
            
            TestObject testObjectWithHealedParent = createTestObjectWithHealedParent(testObject, healedParentElement);
            
            try {
                FindElementsResult childResult = basicElementFinder.findElementsBySelectedMethod(webDriver, testObjectWithHealedParent);
                
                if (childResult != null && !childResult.isEmpty()) {
                    registerBrokenTestObjectAndLog(childResult, testObject);
                    return childResult.getElements();
                }
            } catch (Exception e) {
                logger.logDebug(MessageFormat.format(StringConstants.KW_LOG_DEBUG_SHADOW_DOM_CHILD_FIND_FAILED, 
                        e.getMessage()));
            }
        }

        logger.logDebug(MessageFormat.format(StringConstants.KW_LOG_DEBUG_FAILED_TO_HEAL_SHADOW_PARENT, 
                parentObject.getObjectId()));
        return Collections.emptyList();
    }


    private TestObject createTestObjectWithHealedParent(TestObject testObject, WebElement healedParentElement) {
        TestObject parentObject = testObject.getParentObject();
        if (parentObject != null) {
            parentObject.setCachedWebElement(healedParentElement);
        }
        return testObject;
    }

    private BrokenTestObject registerBrokenTestObjectAndLog(FindElementsResult findResult, TestObject originalTestObject) {
        BrokenTestObject brokenTestObject = registerBrokenTestObject(findResult, originalTestObject, RunConfiguration.getReportFolder());
        registerBrokenTestObject(findResult, originalTestObject, SelfHealingController.getSelfHealingFolderPath());
        
        SelfHealingController.logWarning(
                MessageFormat.format(StringConstants.KW_LOG_INFO_PROPOSE_ALTERNATE_LOCATOR,
                        originalTestObject.getObjectId(), brokenTestObject.getProposedLocator()));
        
        KeywordExecutionContext.setHasHealedSomeObjects(true);
        return brokenTestObject;
    }

    private BrokenTestObject registerBrokenTestObject(FindElementsResult findResult, TestObject brokenTestObject,
            String dataFolder) {
        List<WebElement> foundElements = findResult.getElements();
        WebElement foundElement = foundElements.get(0);

        SelectorMethod recoveryMethod = findResult.getLocatorMethod();
        SelectorMethod proposedMethod = recoveryMethod == SelectorMethod.IMAGE ? SelectorMethod.XPATH : recoveryMethod;
        String proposedLocator = recoveryMethod == SelectorMethod.IMAGE ? generateNewXPath(foundElement)
                : findResult.getLocator();

        String elementScreenshot = findResult.getScreenshot();
        if (StringUtils.isBlank(elementScreenshot)) {
            WebDriver webDriver = DriverFactory.getWebDriver();
            elementScreenshot = SelfHealingController.takeScreenShot(webDriver, foundElement, brokenTestObject,
                    recoveryMethod.name());
            findResult.setScreenshot(elementScreenshot);
        }

        if (StringUtils.isNotBlank(elementScreenshot)) {
            try {
                String screenshotRelativePath = SelfHealingController
                        .getRelativePathToSelfHealindDir(elementScreenshot);
                String destScreenshotPath = FilenameUtils.concat(dataFolder, screenshotRelativePath);
                File destScreenshot = new File(destScreenshotPath);
                if (!destScreenshot.exists()) {
                    FileUtils.copyFile(new File(elementScreenshot), destScreenshot);
                }
                elementScreenshot = destScreenshotPath;
            } catch (IOException exception) {
                SelfHealingController
                        .logWarning(MessageFormat.format(StringConstants.KW_LOG_INFO_COULD_NOT_SAVE_SCREENSHOT,
                                dataFolder, exception.getMessage()), exception);
            }
        }

        if (StringUtils.isNotBlank(elementScreenshot)) {
            String projectDir = RunConfiguration.getProjectDir();
            if (FileUtil.isInBaseFolder(elementScreenshot, projectDir)) {
                elementScreenshot = FileUtil.getRelativePath(elementScreenshot, projectDir);
            }
        }

        return SelfHealingController.registerBrokenTestObject(brokenTestObject, proposedLocator, proposedMethod,
                recoveryMethod, elementScreenshot, dataFolder);
    }

    private String generateNewXPath(WebElement element) {
        WebDriver webDriver = DriverFactory.getWebDriver();

        Map<String, List<String>> generatedXPaths = WebUICommonScripts.generateXPaths(webDriver, element);

        List<Pair<String, Boolean>> xpathsPriority = RunConfiguration.getXPathsPriority();
        for (Pair<String, Boolean> xpath : xpathsPriority) {
            if (generatedXPaths.containsKey(xpath.getLeft()) && !generatedXPaths.get(xpath.getLeft()).isEmpty()) {
                return generatedXPaths.get(xpath.getLeft()).get(0);
            }
        }

        return WebUICommonScripts.generateXPath(webDriver, element);
    }
}
