package com.kms.katalon.core.reporting;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.net.InetAddress;
import java.net.URISyntaxException;
import java.net.URLDecoder;
import java.net.UnknownHostException;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

import javax.xml.stream.XMLStreamException;

import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.io.output.StringBuilderWriter;
import org.apache.commons.lang.StringUtils;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.Element;
import org.dom4j.io.SAXReader;
import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.NullProgressMonitor;

import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.logging.TestSuiteXMLLogParser;
import com.kms.katalon.core.logging.XMLParserException;
import com.kms.katalon.core.logging.model.ILogRecord;
import com.kms.katalon.core.logging.model.TestCaseLogRecord;
import com.kms.katalon.core.logging.model.TestSuiteCollectionLogRecord;
import com.kms.katalon.core.logging.model.TestSuiteLogRecord;
import com.kms.katalon.core.reporting.html.JsSuiteModel;
import com.kms.katalon.core.reporting.html.ResourceLoader;
import com.kms.katalon.core.reporting.newreport.NewHTMLReportDataWriter;
import com.kms.katalon.core.reporting.newreport.NewReportModelMapper;
import com.kms.katalon.core.reporting.pdf.TestSuitePdfGenerator;
import com.kms.katalon.core.reporting.pdf.exception.JasperReportException;
import com.kms.katalon.core.reporting.util.ResourceUtil;
import com.kms.katalon.core.setting.ReportSettings;
import com.kms.katalon.core.testdata.reader.CsvWriter;
import com.kms.katalon.core.util.ObjectUtil;
import com.kms.katalon.core.util.internal.JsonUtil;

public class ReportWriterUtil {
    public static final String COLLECTION_INFO_FILE_NAME = "collection.json";

    private static final String FAILED_STATUS_BACKGROUND_COLOR = "#f19696";

    private static final String INCOMPLETE_STATUS_BACKGROUND_COLOR = "#f2bc70";

    private static final String DEFAULT_COMPOSITE_BACKGROUND_COLOR = "#f0f0f0";

    public static abstract class BaseReportGenerationOptions {
        private ReportSettings settings;

        private File projectDir;

        private File reportDir;

        private File outputFile;

        private File outputDir;

        public ReportSettings getSettings() {
            return settings;
        }

        public void setSettings(ReportSettings settings) {
            this.settings = settings;
        }

        public File getProjectDir() {
            return projectDir;
        }

        public void setProjectDir(File projectDir) {
            this.projectDir = projectDir;
        }

        public File getReportDir() {
            return reportDir;
        }

        public void setReportDir(File reportDir) {
            this.reportDir = reportDir;
        }

        public File getOutputFile() {
            if (outputFile != null) {
                return outputFile;
            }
            var outputDir = this.getOutputDir();
            return outputDir != null ? new File(outputDir, outputDir.getName() + ".html") : null;
        }

        public void setOutputFile(File outputFile) {
            this.outputFile = outputFile;
        }

        public File getOutputDir() {
            return outputDir != null ? outputDir : this.getReportDir();
        }

        public void setOutputDir(File outputDir) {
            this.outputDir = outputDir;
        }
    }

    public static class SuiteReportGenerationOptions extends BaseReportGenerationOptions {
        private TestSuiteLogRecord suiteLogRecord;

        private List<ILogRecord> filteredTestCases;

        public TestSuiteLogRecord getSuiteLogRecord() {
            return suiteLogRecord;
        }

        public void setSuiteLogRecord(TestSuiteLogRecord suiteLogRecord) {
            this.suiteLogRecord = suiteLogRecord;
        }

        public List<ILogRecord> getFilteredTestCases() {
            return filteredTestCases;
        }

        public void setFilteredTestCases(List<ILogRecord> filteredTestCases) {
            this.filteredTestCases = filteredTestCases;
        }
    }

    public static class CollectionReportGenerationOptions extends BaseReportGenerationOptions {
        private TestSuiteCollectionLogRecord collectionLogRecord;

        public TestSuiteCollectionLogRecord getCollectionLogRecord() {
            return collectionLogRecord;
        }

        public void setCollectionLogRecord(TestSuiteCollectionLogRecord collectionLogRecord) {
            this.collectionLogRecord = collectionLogRecord;
        }
    }

    public static abstract class BaseReportGenerationOptionsBuilder<ReportOptionsType extends BaseReportGenerationOptions, BuilderType extends BaseReportGenerationOptionsBuilder<ReportOptionsType, BuilderType>> {
        private ReportSettings settings;

        private File projectDir;

        private File reportDir;

        private File outputFile;

        private File outputDir;

        @SuppressWarnings("unchecked")
        private BuilderType _this() {
            return (BuilderType) this;
        }

        public BuilderType projectDir(File projectDir) {
            this.projectDir = projectDir;
            return _this();
        }

        public BuilderType settings(ReportSettings settings) {
            this.settings = settings;
            return _this();
        }

        public BuilderType reportDir(File reportDir) {
            this.reportDir = reportDir;
            return _this();
        }

        public BuilderType outputFile(File outputFile) {
            this.outputFile = outputFile;
            return _this();
        }

        public BuilderType outputDir(File outputDir) {
            this.outputDir = outputDir;
            return _this();
        }

        protected abstract ReportOptionsType createOptions();

        public ReportOptionsType build() {
            var options = createOptions();
            options.setSettings(settings);
            options.setProjectDir(projectDir);
            options.setReportDir(reportDir);
            options.setOutputFile(outputFile);
            options.setOutputDir(outputDir);
            return options;
        }
    }

    public static class SuiteReportGenerationOptionsBuilder extends
            BaseReportGenerationOptionsBuilder<SuiteReportGenerationOptions, SuiteReportGenerationOptionsBuilder> {
        private TestSuiteLogRecord suiteLogRecord;

        private List<ILogRecord> filteredTestCases;

        public SuiteReportGenerationOptionsBuilder suiteLogRecord(TestSuiteLogRecord suiteLogRecord) {
            this.suiteLogRecord = suiteLogRecord;
            return this;
        }

        public SuiteReportGenerationOptionsBuilder filteredTestCases(List<ILogRecord> filteredTestCases) {
            this.filteredTestCases = filteredTestCases;
            return this;
        }

        public static SuiteReportGenerationOptionsBuilder create() {
            return new SuiteReportGenerationOptionsBuilder();
        }

        @Override
        protected SuiteReportGenerationOptions createOptions() {
            return new SuiteReportGenerationOptions();
        }

        @Override
        public SuiteReportGenerationOptions build() {
            var options = super.build();
            options.setSuiteLogRecord(suiteLogRecord);
            options.setFilteredTestCases(filteredTestCases);
            return options;
        }
    }

    public static class CollectionReportGenerationOptionsBuilder extends
            BaseReportGenerationOptionsBuilder<CollectionReportGenerationOptions, CollectionReportGenerationOptionsBuilder> {
        private TestSuiteCollectionLogRecord collectionLogRecord;

        public CollectionReportGenerationOptionsBuilder collectionLogRecord(
                TestSuiteCollectionLogRecord collectionLogRecord) {
            this.collectionLogRecord = collectionLogRecord;
            return this;
        }

        public static CollectionReportGenerationOptionsBuilder create() {
            return new CollectionReportGenerationOptionsBuilder();
        }

        @Override
        protected CollectionReportGenerationOptions createOptions() {
            return new CollectionReportGenerationOptions();
        }

        @Override
        public CollectionReportGenerationOptions build() {
            var options = super.build();
            options.setCollectionLogRecord(collectionLogRecord);
            return options;
        }
    }

    private static interface ReportWritingPhase {
        public void write(Writer writer) throws IOException, URISyntaxException;
    }

    private static void appendReportConstantValuesWithWriter(List<String> constantValues, Writer writer)
            throws IOException {
        int size = constantValues.size();
        for (int i = 0; i < size; i++) {
            writer.write(constantValues.get(i));
            if (i < size - 1) {
                writer.write(",");
            }
        }
    }

    private static void generateVarsWithWriter(List<String> strings, TestSuiteLogRecord suiteLogEntity,
            StringBuilder model, Writer writer) throws IOException {
        List<String> lines = IOUtils.readLines(
                ResourceUtil.getResourceAsInputStream(ResourceLoader.class, ResourceLoader.HTML_TEMPLATE_VARS));
        for (String line : lines) {
            if (line.equals(ResourceLoader.HTML_TEMPLATE_SUITE_MODEL_TOKEN)) {
                writer.write(model.toString());
            } else if (line.equals(ResourceLoader.HTML_TEMPLATE_STRINGS_CONSTANT_TOKEN)) {
                appendReportConstantValuesWithWriter(strings, writer);
            } else if (line.equals(ResourceLoader.HTML_TEMPLATE_EXEC_ENV_TOKEN)) {
                StringBuilder envInfoSb = new StringBuilder();
                envInfoSb.append("{");
                envInfoSb.append(String.format("\"host\" : \"%s\", ", suiteLogEntity.getHostName()));
                envInfoSb.append(String.format("\"os\" : \"%s\", ", suiteLogEntity.getOs()));
                envInfoSb.append(String.format("\"" + StringConstants.APP_VERSION + "\" : \"%s\", ",
                        suiteLogEntity.getAppVersion()));
                String mobileDeviceName = suiteLogEntity.getMobileDeviceName();
                envInfoSb.append(String.format("\"deviceName\" : \"%s\", ", mobileDeviceName));

                String browserName = suiteLogEntity.getBrowser();
                envInfoSb.append(String.format("\"browserName\" : \"%s\", ", browserName));
                envInfoSb.append("\"\" : \"\"");
                envInfoSb.append("}");
                writer.write(envInfoSb.toString());
            } else {
                writer.write(line);
                writer.write("\n");
            }
        }
    }

    public static String getOs() {
        return System.getProperty("os.name") + " " + System.getProperty("sun.arch.data.model") + "bit";
    }

    public static String getHostName() {
        String hostName = "Unknown";
        try {
            InetAddress addr;
            addr = InetAddress.getLocalHost();
            hostName = addr.getCanonicalHostName();
        } catch (UnknownHostException ex) {}
        return hostName;
    }

    public static void writeLogRecordToCSVFile(TestSuiteLogRecord suiteLogEntity, File destFile,
            List<ILogRecord> filteredTestCases) throws IOException {
        writeLogRecordToCSVFile(suiteLogEntity, destFile, filteredTestCases, true);
    }

    public static void writeLogRecordToCSVFile(TestSuiteLogRecord suiteLogEntity, File destFile,
            List<ILogRecord> filteredTestCases, boolean stepsIncluded) throws IOException {
        CsvWriter.writeCsvReport(suiteLogEntity, destFile, filteredTestCases, stepsIncluded);
    }

    public static File writePdfReport(TestSuiteLogRecord suiteLogEntity, File logFolder)
            throws JasperReportException, IOException {
        List<TestCaseLogRecord> testCases = suiteLogEntity.getAllTestCaseLogRecords();
        List<String> originalTestCaseNames = new ArrayList<>();

        for (TestCaseLogRecord testCase : testCases) {
            originalTestCaseNames.add(testCase.getName());
            testCase.setName(testCase.getTestCaseNameWithIteration());
        }
        TestSuitePdfGenerator pdfGenerator = new TestSuitePdfGenerator(suiteLogEntity);
        File exportLocation = new File(logFolder, logFolder.getName() + ".pdf");
        pdfGenerator.exportToPDF(exportLocation.getAbsolutePath());

        // Restore original names after PDF generation
        for (int i = 0; i < originalTestCaseNames.size(); i++) {
            testCases.get(i).setName(originalTestCaseNames.get(i));
        }
        return exportLocation;
    }

    public static void writeCollectionHTMLReport(CollectionReportGenerationOptions options)
            throws IOException, URISyntaxException, XMLParserException, XMLStreamException, DocumentException {
        if (options.getSettings().applyNewReport) {
            writeNewCollectionHTMLReport(options);
        } else {
            writeOldCollectionHTMLReport(options);
        }
    }

    private static void writeNewCollectionHTMLReport(CollectionReportGenerationOptions options)
            throws IOException, URISyntaxException, XMLParserException, XMLStreamException, DocumentException {
        var collectionLogRecord = options.getCollectionLogRecord();
        var outputFile = options.getOutputFile();
        var settings = options.getSettings();
        var collectionReportFolder = options.getReportDir();
        var projectFolder = options.getProjectDir();

        if (collectionLogRecord == null) {
            collectionLogRecord = collectCollectionLogRecord(collectionReportFolder, projectFolder);
        }

        String template = readFileToStringBuilder(ResourceLoader.NEW_HTML_TEST_SUITE_COLLECTION_TEMPLATE_FILE);
        String[] parts = template.split(ResourceLoader.NEW_HTML_DATA_PLACEHOLDER);
        String header = parts[0];
        String footer = parts[1];

        var suiteLogRecords = collectionLogRecord.getTestSuiteRecords();
        if (suiteLogRecords == null) {
            suiteLogRecords = collectCollectionSuites(collectionReportFolder, projectFolder);
            collectionLogRecord.setTestSuiteRecords(suiteLogRecords);
        }

        for (var testSuite : suiteLogRecords) {
            testSuite.setTestSuiteCollectionLocation(outputFile.getParent());
            List<ILogRecord> newChildRecords = new ArrayList<>();
            newChildRecords.addAll(testSuite.getBeforeTestSuiteLogRecords());
            newChildRecords.addAll(Arrays.asList(testSuite.getFinalTestCases()));
            newChildRecords.addAll(testSuite.getAfterTestSuiteLogRecords());
            testSuite.setChildRecords(newChildRecords.toArray(new ILogRecord[] {}));
        }

        final var finalCollectionLogRecord = collectionLogRecord;
        writeReport(outputFile, (writer) -> {
            writer.write(header);
        }, (writer) -> {
            var newHTMLWriter = new NewHTMLReportDataWriter(writer, outputFile.getParentFile(), false);
            new NewReportModelMapper(settings, collectionReportFolder).toTestExecutionParts(finalCollectionLogRecord,
                    (String part, String partKey) -> {
                        newHTMLWriter.writePart(part, partKey);
                    });
        }, (writer) -> {
            writer.write(footer);
        });
    }

    private static List<TestSuiteLogRecord> collectCollectionSuites(File collectionReportFolder, File projectFolder)
            throws XMLParserException, IOException, XMLStreamException, DocumentException {
        List<TestSuiteLogRecord> suiteLogRecords = new ArrayList<TestSuiteLogRecord>();
        List<String> testSuiteReportIds = collectCollectionSuiteIds(collectionReportFolder);

        for (String reportRelativeLocation : testSuiteReportIds) {
            var suiteReportDir = new File(projectFolder, reportRelativeLocation);
            TestSuiteLogRecord testSuiteLogRecord = ReportWriterUtil
                    .parseCollectionSuiteLog(suiteReportDir.getAbsolutePath());
            suiteLogRecords.add(testSuiteLogRecord);
        }

        return suiteLogRecords;
    }

    private static void writeOldCollectionHTMLReport(CollectionReportGenerationOptions options)
            throws IOException, URISyntaxException, DocumentException {
        var tsInfoItems = writeOldCollectionSuiteHTMLReport(options);
        var rawTsReports = JsonUtil.toJson(tsInfoItems, false);

        var colelctionIndexFile = options.getOutputFile();
        var reportDir = options.getReportDir();
        writeOldCollectionHTMLIndexFile(reportDir, colelctionIndexFile, rawTsReports);
    }

    private static void writeOldCollectionHTMLIndexFile(File reportDir, File colelctionIndexFile, String rawTsReports)
            throws IOException, URISyntaxException {
        var reportTitle = reportDir.getName();

        var destDir = colelctionIndexFile.getParentFile();
        String template = readFileToStringBuilder(ResourceLoader.HTML_COLLECTION_INDEX_TEMPLATE);
        template = StringUtils.replace(template, "REPORT_TITLE", reportTitle);
        template = StringUtils.replace(template, "TEST_SUITE_REPORT_LIST", rawTsReports);

        FileUtils.writeStringToFile(colelctionIndexFile, template, StringConstants.DF_CHARSET);

        template = readFileToStringBuilder(ResourceLoader.HTML_COLLECTION_FRAME_TEMPLATE);
        template = StringUtils.replace(template, "REPORT_TITLE", reportTitle);
        template = StringUtils.replace(template, "TEST_SUITE_REPORT_LIST", rawTsReports);
        FileUtils.writeStringToFile(new File(destDir, "index-frame-view.html"), template, StringConstants.DF_CHARSET);
    }

    private static List<String> collectCollectionSuiteIds(File reportDir) throws DocumentException {
        var reportEntityFile = new File(reportDir, reportDir.getName() + ".rp");

        SAXReader reader = new SAXReader();
        Document document = reader.read(reportEntityFile);
        Element rootElement = document.getRootElement();
        List<String> testSuiteReportIds = new ArrayList<>();
        List<Element> reportItemDescriptionItems = rootElement.element("reportItemDescriptions")
                .elements("ReportItemDescription");
        reportItemDescriptionItems.stream().forEach(e -> {
            testSuiteReportIds.add(e.elementText("reportLocation"));
        });

        return testSuiteReportIds;
    }

    private static List<Map<String, String>> writeOldCollectionSuiteHTMLReport(
            CollectionReportGenerationOptions options) throws DocumentException, IOException, URISyntaxException {
        var reportDir = options.getReportDir();
        var exportLocation = options.getOutputDir();
        var outputFile = options.getOutputFile();
        var settings = options.getSettings();
        var projectDir = options.getProjectDir();

        List<String> testSuiteReportIds = collectCollectionSuiteIds(reportDir);

        Map<String, String> tsInfoItem;
        List<Map<String, String>> tsInfoItems = new ArrayList<>();
        for (String reportRelativeLocation : testSuiteReportIds) {
            try {
                var suiteReportDir = new File(projectDir, reportRelativeLocation);

                TestSuiteLogRecord testSuiteLogRecord = ReportWriterUtil
                        .parseCollectionSuiteLog(suiteReportDir.getAbsolutePath());

                String htmlFileName = new File(reportRelativeLocation).getName() + ".html";

                File suiteOutputFile = new File(exportLocation, htmlFileName);

                String testSuiteCollectionId = StringUtils.defaultString(testSuiteLogRecord.getTestSuiteCollectionId());
                String testSuiteCollectionName = new File(testSuiteCollectionId).getName();
                testSuiteLogRecord.setTestSuiteCollectionName(testSuiteCollectionName);
                testSuiteLogRecord
                        .setTestSuiteCollectionPath(exportLocation.toPath().relativize(outputFile.toPath()).toString());

                ReportWriterUtil.writeHTMLReport(SuiteReportGenerationOptionsBuilder.create()
                        .suiteLogRecord(testSuiteLogRecord)
                        .settings(settings)
                        .reportDir(reportDir)
                        .outputFile(suiteOutputFile)
                        .build());

                tsInfoItem = new HashMap<>();
                tsInfoItem.put("report_location", reportRelativeLocation);
                tsInfoItem.put("report", htmlFileName);
                tsInfoItem.put("id", getTestSuiteId(reportRelativeLocation));
                tsInfoItem.put("environment", testSuiteLogRecord.getBrowser());
                String status = getStatus(testSuiteLogRecord);
                tsInfoItem.put("status", status);
                tsInfoItem.put("fail_on_total", getFailOnTotal(testSuiteLogRecord));
                tsInfoItem.put("status_color", getStatusColor(status));
                tsInfoItem.put("fail_color", getFailColor(testSuiteLogRecord));
                tsInfoItems.add(tsInfoItem);
            } catch (XMLParserException | XMLStreamException e) {
                throw new IOException(e);
            }
        }

        return tsInfoItems;
    }

    public static void writeHTMLReport(SuiteReportGenerationOptions options) throws IOException, URISyntaxException {
        if (options.getSettings().applyNewReport) {
            writeNewTestSuiteHTMLReport(options);
        } else {
            writeOldTestSuiteHTMLReport(options);
        }
    }

    private static void writeOldTestSuiteHTMLReport(SuiteReportGenerationOptions options)
            throws IOException, URISyntaxException {
        var testSuiteLogRecord = options.suiteLogRecord;
        var filteredTestCases = options.filteredTestCases;
        var outputFile = options.getOutputFile();

        List<String> strings = new ArrayList<String>();

        JsSuiteModel jsSuiteModel = filteredTestCases != null
                ? new JsSuiteModel(testSuiteLogRecord, strings, filteredTestCases)
                : new JsSuiteModel(testSuiteLogRecord, strings);
        StringBuilder sbModel = jsSuiteModel.toArrayString();

        writeReport(outputFile, (writer) -> {
            writer.write(readFileToStringBuilder(ResourceLoader.HTML_TEMPLATE_FILE));
        }, (writer) -> {
            generateVarsWithWriter(strings, testSuiteLogRecord, sbModel, writer);
        }, (writer) -> {
            writer.write(readFileToStringBuilder(ResourceLoader.HTML_TEMPLATE_CONTENT));
        });
    }

    private static void writeNewTestSuiteHTMLReport(SuiteReportGenerationOptions options)
            throws IOException, URISyntaxException {
        String newHTMLTemplate = readFileToStringBuilder(ResourceLoader.NEW_HTML_TEMPLATE_FILE);
        String[] parts = newHTMLTemplate.split(ResourceLoader.NEW_HTML_DATA_PLACEHOLDER);
        String header = parts[0];
        String footer = parts[1];

        var testSuiteLogRecord = options.getSuiteLogRecord();
        var filteredTestCases = options.getFilteredTestCases();
        var finalTestCases = filteredTestCases != null ? filteredTestCases
                : Arrays.asList(testSuiteLogRecord.getFinalTestCases());
        var outputFile = options.getOutputFile();
        
        testSuiteLogRecord.setTestSuitePath(outputFile.getPath());

        final var testSuiteLogRecordClone = new TestSuiteLogRecord("", "");
        ObjectUtil.clone(testSuiteLogRecord, testSuiteLogRecordClone);

        List<ILogRecord> newChildRecords = new ArrayList<>();
        newChildRecords.addAll(testSuiteLogRecordClone.getBeforeTestSuiteLogRecords());
        newChildRecords.addAll(finalTestCases);
        newChildRecords.addAll(testSuiteLogRecordClone.getAfterTestSuiteLogRecords());
        testSuiteLogRecordClone.setChildRecords(newChildRecords.toArray(new ILogRecord[] {}));

        var reportDir = options.getReportDir();
        
        var settings = options.getSettings();
        
        writeReport(outputFile, (writer) -> {
            writer.write(header);
        }, (writer) -> {
            var newHTMLWriter = new NewHTMLReportDataWriter(writer, outputFile.getParentFile(),
                    settings.useSplitReportData);
            new NewReportModelMapper(settings, reportDir).toTestExecutionParts(testSuiteLogRecordClone,
                    (String part, String partKey) -> {
                        newHTMLWriter.writePart(part, partKey);
                    });
        }, (writer) -> {
            writer.write(footer);
        });
    }

    private static void writeReport(File destFile, ReportWritingPhase... phases)
            throws IOException, URISyntaxException {
        try (var outputStream = new FileOutputStream(destFile)) {
            try (var writer = new OutputStreamWriter(outputStream, StringConstants.DF_CHARSET)) {
                for (var phase : phases) {
                    phase.write(writer);
                }
            }
        }
    }

    public static File writeCSVReport(TestSuiteLogRecord suiteLogEntity, File folder) throws IOException {
        File file = new File(folder, folder.getName() + ".csv");
        if (!file.exists()) {
            CsvWriter.writeCsvReport(suiteLogEntity, file, Arrays.asList(suiteLogEntity.filterFinalTestCasesResult()));
        }

        return file;
    }

    public static TestSuiteLogRecord parseTestSuiteLog(String logFolder, IProgressMonitor progressMonitor)
            throws XMLParserException, IOException, XMLStreamException {
        return new TestSuiteXMLLogParser().readTestSuiteLogFromXMLFiles(logFolder, progressMonitor);
    }

    public static TestSuiteLogRecord parseTestSuiteLog(String logFolder)
            throws XMLParserException, IOException, XMLStreamException {
        return parseTestSuiteLog(logFolder, new NullProgressMonitor());
    }

    public static TestSuiteLogRecord parseCollectionSuiteLog(String logFolder, IProgressMonitor progressMonitor)
            throws XMLParserException, IOException, XMLStreamException {
        TestSuiteLogRecord suiteLogRecord = parseTestSuiteLog(logFolder, progressMonitor);
        String testSuiteCollectionId = getTestSuiteCollectionId(new File(logFolder));
        if (StringUtils.isNotBlank(testSuiteCollectionId) && suiteLogRecord != null) {
            suiteLogRecord.setTestSuiteCollectionId(testSuiteCollectionId);
        }
        return suiteLogRecord;
    }

    public static TestSuiteLogRecord parseCollectionSuiteLog(String logFolder)
            throws XMLParserException, IOException, XMLStreamException {
        return parseCollectionSuiteLog(logFolder, new NullProgressMonitor());
    }

    private static String getTestSuiteCollectionId(File logFolder) throws IOException {
        File idFile = getOrCreateTestSuiteCollectionIdFile(logFolder);
        return FileUtils.readFileToString(idFile, StringConstants.DF_CHARSET);
    }

    public static void writeTestSuiteCollectionIdToFile(String testSuiteCollectionId, File logFolder)
            throws IOException {
        File idFile = getOrCreateTestSuiteCollectionIdFile(logFolder);
        FileUtils.writeStringToFile(idFile, testSuiteCollectionId, StringConstants.DF_CHARSET);
    }

    private static File getOrCreateTestSuiteCollectionIdFile(File logFolder) throws IOException {
        File idFile = new File(logFolder, "tsc_id.txt");
        if (!idFile.exists()) {
            idFile.createNewFile();
        }
        return idFile;
    }

    private static String readFileToStringBuilder(String filePath) throws IOException, URISyntaxException {
        StringBuilder sb = new StringBuilder();
        String path = ResourceLoader.class.getProtectionDomain().getCodeSource().getLocation().getFile();
        path = URLDecoder.decode(path, "utf-8");
        File jarFile = new File(path);
        if (jarFile.isFile()) {
            JarFile jar = new JarFile(jarFile);
            Enumeration<JarEntry> entries = jar.entries();
            while (entries.hasMoreElements()) {
                JarEntry jarEntry = entries.nextElement();
                String name = jarEntry.getName();
                if (name.endsWith(filePath)) {
                    StringBuilderWriter sbWriter = new StringBuilderWriter(new StringBuilder());
                    IOUtils.copy(jar.getInputStream(jarEntry), sbWriter);
                    sbWriter.flush();
                    sbWriter.close();
                    sb.append(sbWriter.getBuilder());
                    break;
                }
            }
            jar.close();
        } else { // Run with IDE
            InputStream is = new FileInputStream(new File(jarFile, filePath).getAbsolutePath());
            sb.append(IOUtils.toString(is, "UTF-8"));
        }

        return sb.toString();
    }

    public static void saveCollectionLogRecord(TestSuiteCollectionLogRecord collectionLogRecord, File reportFolder)
            throws IOException {
        var suiteLogRecords = collectionLogRecord.getTestSuiteRecords();
        collectionLogRecord.setTestSuiteRecords(null);
        FileUtils.writeStringToFile(getCollectionInfoFile(reportFolder), JsonUtil.toJson(collectionLogRecord), "UTF-8");
        collectionLogRecord.setTestSuiteRecords(suiteLogRecords);
    }

    public static TestSuiteCollectionLogRecord collectCollectionLogRecord(File reportFolder, File projectFolder)
            throws IOException, DocumentException, XMLParserException, XMLStreamException {
        TestSuiteCollectionLogRecord collection;

        var tscInfoFile = getCollectionInfoFile(reportFolder);
        if (tscInfoFile.exists()) {
            var rawCollectionInfo = FileUtils.readFileToString(tscInfoFile, "UTF-8");
            collection = JsonUtil.fromJson(rawCollectionInfo, TestSuiteCollectionLogRecord.class);
        } else {
            collection = parseCollectionLogRecordFromRPFile(reportFolder);
        }

        if (collection != null) {
            collection.setTestSuiteRecords(collectCollectionSuites(reportFolder, projectFolder));

            var suites = collection.getTestSuiteRecords();
            if (suites != null && suites.size() > 0) {
                if (collection.getStartTime() == 0) {
                    collection.setStartTime(suites.get(0).getStartTime());
                }
                if (collection.getEndTime() == 0) {
                    collection.setEndTime(suites.get(suites.size() - 1).getEndTime());
                }
            }
        }

        return collection;
    }

    private static TestSuiteCollectionLogRecord parseCollectionLogRecordFromRPFile(File reportFolder)
            throws DocumentException {
        var reportEntityFile = new File(reportFolder, reportFolder.getName() + ".rp");

        SAXReader reader = new SAXReader();
        Document document = reader.read(reportEntityFile);
        Element rootElement = document.getRootElement();

        var tscLogRecord = new TestSuiteCollectionLogRecord();

        var collectionId = rootElement.elementText("testSuiteCollectionId");
        tscLogRecord.setDescription(rootElement.elementText("description"));
        tscLogRecord.setName(new File(collectionId).getName());
        tscLogRecord.setReportFolder(reportFolder.getAbsolutePath());
        tscLogRecord.setId(collectionId);

        return tscLogRecord;
    }

    public static File getCollectionInfoFile(File reportFolder) {
        return new File(reportFolder, COLLECTION_INFO_FILE_NAME);
    }

    private static int getTotalFailedTestCases(TestSuiteLogRecord testSuiteLogRecord) {
        if (testSuiteLogRecord == null) {
            return 0;
        }

        return testSuiteLogRecord.getTotalFailedTestCases() + testSuiteLogRecord.getTotalErrorTestCases();
    }

    private static String getFailOnTotal(TestSuiteLogRecord testSuiteLogRecord) {
        if (testSuiteLogRecord == null) {
            return StringUtils.EMPTY;
        }

        return getTotalFailedTestCases(testSuiteLogRecord) + " / " + testSuiteLogRecord.getTotalTestCases();
    }

    private static String getFailColor(TestSuiteLogRecord testSuiteLogRecord) {
        if (testSuiteLogRecord == null) {
            return StringUtils.EMPTY;
        }

        if (getTotalFailedTestCases(testSuiteLogRecord) > 0) {
            return FAILED_STATUS_BACKGROUND_COLOR;
        }

        return StringUtils.EMPTY;
    }

    private static String getTestSuiteId(String reportRelativeLocation) {
        String testSuiteId = reportRelativeLocation.replaceFirst("Reports", "Test Suites");
        testSuiteId = StringUtils.substringBeforeLast(testSuiteId, "/");
        return testSuiteId;
    }

    private static String getStatus(TestSuiteLogRecord logRecord) {
        if (logRecord == null) {
            return "NOT_STARTED";
        }

        if (logRecord.getTotalIncompleteTestCases() > 0) {
            return "INCOMPLETE";
        }

        return "COMPLETE";
    }

    private static String getStatusColor(String status) {
        if (StringUtils.equals(status, "NOT_STARTED")) {
            return DEFAULT_COMPOSITE_BACKGROUND_COLOR;
        }

        if (StringUtils.equals(status, "INCOMPLETE")) {
            return INCOMPLETE_STATUS_BACKGROUND_COLOR;
        }

        return StringUtils.EMPTY;
    }
}
