package com.kms.katalon.core.util;

import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.Strings;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.jsoup.nodes.Element;
import org.jsoup.nodes.TextNode;

import in.wilsonl.minifyhtml.Configuration;
import in.wilsonl.minifyhtml.MinifyHtml;

public class HTMLMinifier {
    private String originalHTML;

    private Document document;

    private boolean debuggingEnabled = true;

    private static final String HIDDEN_ELEMENT_SELECTOR = StringUtils.join(Arrays.asList("script", "style", "noscript",
            "link", "meta", "head", "base", "template", "slot", "iframe", "object", "embed", "track", "param", "source",
            "picture", "*[style*=\"display: none\"]", "*[style*=\"display:none\"]", "*[style*=\"visibility: hidden\"]",
            "*[style*=\"visibility:hidden\"]", "*[style*=\"opacity: 0\"]", "*[style*=\"opacity:0\"]", "svg", "footer"),
            ",");

    private static final int MAX_ATTRIBUTE_VALUE_LENGTH = 50;

    private static final int MAX_TEXT_CONTENT_LENGTH = 100;

    public HTMLMinifier(String html) {
        this.originalHTML = html;
    }

    private void parseHTML() {
        this.document = Jsoup.parse(originalHTML);
    }

    public String minify() {
        return minify(null);
    }

    public String minify(String locator) {
        var minifiedDOM = this.minifyDOM(locator);
        var minifiedHTML = this.minifyHTML(minifiedDOM);

        var originalLength = this.originalHTML.length();
        var minifiedLength = this.buildHTML().length();
        System.out.println(MessageFormat.format(">>> Total: {0} -> {1} (-{2,number,#.##}%)", originalLength,
                minifiedLength, ((double) originalLength - minifiedLength) / originalLength * 100));

        return minifiedHTML;
    }

    private String minifyDOM(String locator) {
        this.parseHTML();
        runDOMMinizationPhase("Remove hidden elements", () -> {
            this.removeHiddenElements();
        });
        runDOMMinizationPhase("Remove next siblings", () -> {
            this.removeUnusedSiblings(locator);
        });
        runDOMMinizationPhase("Remove redundant parents", () -> {
            this.removeRedundantParents();
        });
        runDOMMinizationPhase("Remove event handler attributes", () -> {
            this.removeEventHandlerAttributes();
        });
        runDOMMinizationPhase("Trim long attributes", () -> {
            this.trimLongAttributes();
        });
        runDOMMinizationPhase("Trim long text", () -> {
            this.trimLongText();
        });

        return this.buildHTML();
    }

    private void runDOMMinizationPhase(String name, Runnable phase) {
        if (this.debuggingEnabled) {
            var beforeSize = this.buildHTML().length();
            phase.run();
            var afterSize = this.buildHTML().length();
            this.log(name, beforeSize, afterSize);
        } else {
            phase.run();
        }
    }

    private void log(String label, int beforeSize, int afterSize) {
        var originalSize = this.originalHTML.length();
        System.out.println(MessageFormat.format("> {0}: {1} -> {2} (-{3,number,#.##}%)", label, beforeSize, afterSize,
                ((double) beforeSize - afterSize) / originalSize * 100));
    }

    private String buildHTML() {
        return document.toString();
    }

    private String minifyHTML(String html) {
        var beforeSize = html.length();
        Configuration configs = new Configuration.Builder().setKeepHtmlAndHeadOpeningTags(false)
                .setMinifyCss(true)
                .setMinifyJs(true)
                .setMinifyDoctype(true)
                .setKeepInputTypeTextAttr(false)
                .setKeepComments(false)
                .setKeepSsiComments(false)
                .setKeepClosingTags(false)
                .setAllowNoncompliantUnquotedAttributeValues(true)
                .setAllowRemovingSpacesBetweenAttributes(true)
                .setAllowOptimalEntities(false)
                .setPreserveBraceTemplateSyntax(false)
                .setPreserveChevronPercentTemplateSyntax(false)
                .setRemoveProcessingInstructions(true)
                .setRemoveBangs(true)
                .build();
        var minimizedHTML = MinifyHtml.minify(html, configs);
        var afterSize = minimizedHTML.length();
        if (this.debuggingEnabled) {
            this.log("Minify HTML syntax", beforeSize, afterSize);
        }
        return minimizedHTML;
    }

    private void removeHiddenElements() {
        var elements = document.select(HIDDEN_ELEMENT_SELECTOR);
        for (var element : elements) {
            element.remove();
        }
    }

    /**
     * If we are going to generate a stable locator for some element => Remove all
     * the next siblings and previous siblings' contents.
     */
    private void removeUnusedSiblings(String targetLocator) {
        if (StringUtils.isBlank(targetLocator)) {
            return;
        }
        var matchedElements = document.select(targetLocator);
        if (CollectionUtils.isEmpty(matchedElements)) {
            return;
        }
        matchedElements.forEach((matchedElement) -> {
            this.removeUnusedSiblings(matchedElement);
        });
    }

    private void removeUnusedSiblings(Element element) {
        var parents = element.parents();
        var anchorElement = element;
        for (var parent : parents) {
            var anchorIndex = anchorElement.elementSiblingIndex();
            var children = parent.children();
            for (var child : children) {
                if (child.elementSiblingIndex() > anchorIndex) {
                    child.remove();
                }
            }
            anchorElement = parent;
        }
    }

    private void removeRedundantParents() {
        removeRedundantParents(getBodyElement());
    }

    private void removeRedundantParents(Element root) {
        var children = root.children();
        if (CollectionUtils.isEmpty(children)) {
            return;
        }

        var firstChild = children.get(0);
        var isOnlyChild = children.size() == 1 && firstChild.children().size() == 1
                && "div".equalsIgnoreCase(firstChild.tagName()) && firstChild.attributes().size() == 0;
        if (isOnlyChild) {
            root.replaceWith(firstChild);
            removeRedundantParents(firstChild);
            return;
        }

        children.forEach((child) -> {
            removeRedundantParents(child);
        });
    }

    private void removeEventHandlerAttributes() {
        this.removeEventHandlerAttributes(getBodyElement());
    }

    private void removeEventHandlerAttributes(Element root) {
        var attributes = root.attributes();
        var tobeRemovedAttributes = new ArrayList<String>();
        for (var attribute : attributes) {
            var attributeKey = attribute.getKey();
            if (attributeKey.startsWith("on")) {
                tobeRemovedAttributes.add(attributeKey);
            }
        }
        tobeRemovedAttributes.forEach((attributeName) -> {
            root.removeAttr(attributeName);
        });
        root.children().forEach((child) -> {
            removeEventHandlerAttributes(child);
        });
    }

    private void trimLongAttributes() {
        trimLongAttributes(getBodyElement());
    }

    private void trimLongAttributes(Element root) {
        root.removeAttr("style");

        var attributes = root.attributes();
        for (var attribute : attributes) {
            var attributeKey = attribute.getKey();
            if (Strings.CS.equalsAny(attributeKey, "class", "value", "id", "name")
                    || Strings.CS.equalsAny(attributeKey, "aria-")) {
                continue;
            }
            var attributeValue = attribute.getValue();
            if (StringUtils.isNotBlank(attributeValue) && attributeValue.length() > MAX_ATTRIBUTE_VALUE_LENGTH) {
                var trimmedAttributeValue = trimLongText(attributeValue, MAX_ATTRIBUTE_VALUE_LENGTH);
                root.attr(attributeKey, trimmedAttributeValue);
            }
        }

        root.children().forEach((child) -> {
            trimLongAttributes(child);
        });
    }

    private void trimLongText() {
        trimLongText(getBodyElement());
    }

    private void trimLongText(Element root) {
        root.childNodes().forEach((childNode) -> {
            if (childNode instanceof TextNode textNode) {
                var text = textNode.text().trim();
                textNode.text(trimLongText(text, MAX_TEXT_CONTENT_LENGTH));
                return;
            }
            if (childNode instanceof Element childElement) {
                trimLongText(childElement);
            }
        });
    }

    private Element getBodyElement() {
        return document.selectFirst("body");
    }

    private String trimLongText(String text, int maxLength) {
        return text.length() > maxLength ? StringUtils.substring(text.trim(), 0, maxLength - 3) + "..." : text.trim();
    }
}
