package com.kms.katalon.core.mobile.helper;

import java.text.MessageFormat;
import java.time.Duration;
import java.util.Arrays;
import java.util.List;
import java.util.Map;

import org.openqa.selenium.WebDriverException;
import org.openqa.selenium.WebElement;
import org.xmlunit.builder.DiffBuilder;
import org.xmlunit.diff.DefaultNodeMatcher;
import org.xmlunit.diff.Diff;
import org.xmlunit.diff.ElementSelectors;

import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.mobile.constants.StringConstants;
import com.kms.katalon.core.mobile.exception.WebElementNotFoundException;
import com.kms.katalon.core.mobile.keyword.internal.MobileDriverFactory;
import com.kms.katalon.core.testobject.TestObject;
import com.kms.katalon.core.util.internal.JsonUtil;

import io.appium.java_client.AppiumBy;
import io.appium.java_client.AppiumDriver;
import io.appium.java_client.android.AndroidDriver;
import io.appium.java_client.android.HasAndroidDeviceDetails;

public final class AndroidHelper {

    private AndroidHelper() {
    }

    private static final KeywordLogger logger = KeywordLogger.getInstance(AndroidHelper.class);

    public static int getStatusBarHeight(AppiumDriver driver) {
        int statusBar = 0;
        try {
            statusBar = getStatusBarByProperty(driver);
        } catch (WebDriverException e) {
            logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_FAILED_GET_OS_STATUSBAR, "getSystemBar"));
        }

        if (statusBar == 0) {
            try {
                return getStatusBarHeightByCommand(driver);
            } catch (WebDriverException e) {
                logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_FAILED_GET_OS_STATUSBAR, "viewportRect"));
            }
        }
        return statusBar;
    }

    private static final List<String> SCROLLABLE_CLASSES = Arrays.asList("androidx.recyclerview.widget.RecyclerView",
            "android.widget.ScrollView", "android.widget.ListView", "androidx.core.widget.NestedScrollView");

    /**
     * Scrolls to an element with the given text starting from the TestObject's view.
     *
     * @param testObject TestObject to locate the scrollable area
     * @param text Text to scroll to
     * @param timeoutInSeconds Timeout duration
     * @return true if the element is found, false otherwise
     * @throws WebElementNotFoundException
     */

    public static WebElement scrollToText(TestObject testObject, String text, int timeoutInSeconds)
            throws WebElementNotFoundException {
        long endTime = System.currentTimeMillis() + timeoutInSeconds * 1000L;

        AndroidDriver driver = (AndroidDriver) MobileDriverFactory.getDriver();

        Duration originalImplicitWait = driver.manage().timeouts().getImplicitWaitTimeout();

        WebElement scrollableElement = null;
        if (testObject != null) {
            try {
                scrollableElement = MobileCommonHelper.findElement(driver, testObject, 0);
            } catch (Exception e) {}
        }

        if (scrollableElement == null) {
            scrollableElement = findScrollableElement(driver);
        }

        if (scrollableElement == null) {
            throw new WebElementNotFoundException("Scrollable element not found");
        }

        long remainingTime = endTime - System.currentTimeMillis();
        if (remainingTime <= 0) {
            logger.logWarning("Timeout.");
            return null;
        }

        WebElement targetElement = null;

        try {
            // Step 1: Try to find the element by text without scrolling
            try {
                driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));
                targetElement = findElementContainingText(scrollableElement, text);
                if (targetElement != null) {
                    return targetElement;
                }
            } catch (Exception ignore) {}

            remainingTime = endTime - System.currentTimeMillis();
            if (remainingTime <= 0) {
                logger.logWarning("Timeout.");
                return null;
            }

            // Step 3: Scroll to top
            String className = scrollableElement.getAttribute("class");
            String scrollBackward = String
                    .format("new UiScrollable(new UiSelector().className(\"%s\")).scrollBackward()", className);
            String resourceId = scrollableElement.getAttribute("resourceId");
            if (resourceId != null && !resourceId.equals("null")) {
                scrollBackward = String.format(
                        "new UiScrollable(new UiSelector().className(\"%s\").resourceId(\"%s\")).scrollBackward()",
                        className, resourceId);
            }

            String previousSource, currentSource;
            previousSource = driver.getPageSource();
            boolean canScroll = true;
            while (canScroll) {
                remainingTime = endTime - System.currentTimeMillis();
                if (remainingTime <= 0) {
                    logger.logWarning("Timeout.");
                    return null;
                }

                try {
                    // Each call will scroll one step up
                    driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));
                    driver.findElement(AppiumBy.androidUIAutomator(scrollBackward));

                    currentSource = driver.getPageSource();
                    if (arePageSourcesEqual(previousSource, currentSource)) {
                        canScroll = false; // no more scrolling happened
                    }

                    previousSource = currentSource;
                } catch (Exception e) {
                    logger.logInfo("Scrolling to top.");
                    canScroll = false;
                }
            }

            // Step 4: Scroll forward repeatedly and search for the target text
            canScroll = true;
            while (canScroll) {
                remainingTime = endTime - System.currentTimeMillis();
                if (remainingTime <= 0) {
                    logger.logWarning("Timeout.");
                    break;
                }

                try {
                    driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));
                    targetElement = findElementContainingText(scrollableElement, text);
                    if (targetElement != null) {
                        return targetElement;
                    }
                } catch (Exception ignore) {}

                try {
                    String scrollForward = String
                            .format("new UiScrollable(new UiSelector().className(\"%s\")).scrollForward()", className);
                    if (resourceId != null && !resourceId.equals("null")) {
                        scrollForward = String.format(
                                "new UiScrollable(new UiSelector().className(\"%s\").resourceId(\"%s\")).scrollForward()",
                                className, resourceId);
                    }

                    driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));
                    driver.findElement(AppiumBy.androidUIAutomator(scrollForward));

                    currentSource = driver.getPageSource();
                    if (arePageSourcesEqual(previousSource, currentSource)) {
                        canScroll = false; // no more scrolling happened
                    }

                    previousSource = currentSource;
                } catch (Exception e) {
                    logger.logInfo("Scrolling ended or failed.");
                    canScroll = false;
                }
            }

            throw new WebElementNotFoundException(String.format("Element with text '%s' not found", text));
        } finally {
            // Restore implicit wait
            driver.manage().timeouts().implicitlyWait(originalImplicitWait);
        }
    }

    public static WebElement findScrollableElement(AppiumDriver driver) {
        List<WebElement> scrollables = driver
                .findElements(AppiumBy.androidUIAutomator("new UiSelector().scrollable(true)"));
        for (WebElement el : scrollables) {
            if ("true".equals(el.getAttribute("enabled")) && "true".equals(el.getAttribute("scrollable"))
                    && el.isDisplayed()) {
                return el;
            }
        }

        // Fallback to known scrollable classes
        for (String clazz : SCROLLABLE_CLASSES) {
            List<WebElement> candidates = driver.findElements(AppiumBy.className(clazz));
            for (WebElement el : candidates) {
                if ("true".equals(el.getAttribute("enabled"))) {
                    return el; // don’t check scrollable attr since it may be "false"
                }
            }
        }

        return null;
    }

    public static WebElement findElementContainingText(WebElement scrollableView, String text) {
        List<WebElement> elements = scrollableView
                .findElements(AppiumBy.xpath(".//*[contains(@text, '" + text + "')]"));
        for (WebElement el : elements) {
            if ("true".equals(el.getAttribute("enabled")) && el.isDisplayed()) {
                return el;
            }
        }

        return null;
    }

    private static int getStatusBarByProperty(AppiumDriver driver) throws WebDriverException {
        Map<String, Map<String, Object>> bars = ((HasAndroidDeviceDetails) driver).getSystemBars();
        AndroidSystemBarProperties barsProps = JsonUtil.fromJson(JsonUtil.toJson(bars),
                AndroidSystemBarProperties.class);
        OSBarProperties statusBar = barsProps.getStatusBar();
        if (statusBar != null && statusBar.getWidth() > 0 && statusBar.getHeight() > 0) {
            return statusBar.getHeight();
        }
        return 0;
    }

    private static int getStatusBarHeightByCommand(AppiumDriver driver) {
        Object res = driver.executeScript("mobile:viewportRect");
        ViewportRect vp = JsonUtil.fromJson(res.toString(), ViewportRect.class);
        return vp.top == 0 ? 0 : (int) vp.top;
    }

    private static boolean arePageSourcesEqual(String pageSourceA, String pageSourceB) {
        Diff diff = DiffBuilder.compare(pageSourceA)
                .withTest(pageSourceB)
                .withNodeMatcher(new DefaultNodeMatcher(ElementSelectors.byNameAndAllAttributes))
                .ignoreWhitespace()
                .checkForSimilar()
                .build();
        return !diff.hasDifferences();
    }
}

class ViewportRect {
    public int left;

    public int top;

    public int height;

    public int width;

    public int getLeft() {
        return left;
    }

    public void setLeft(int left) {
        this.left = left;
    }

    public int getTop() {
        return top;
    }

    public void setTop(int top) {
        this.top = top;
    }

    public int getHeight() {
        return height;
    }

    public void setHeight(int height) {
        this.height = height;
    }

    public int getWidth() {
        return width;
    }

    public void setWidth(int width) {
        this.width = width;
    }

}

class AndroidSystemBarProperties {
    private OSBarProperties statusBar;

    private OSBarProperties navigationBar;

    public OSBarProperties getStatusBar() {
        return statusBar;
    }

    public void setStatusBar(OSBarProperties statusBar) {
        this.statusBar = statusBar;
    }

    public OSBarProperties getNavigationBar() {
        return navigationBar;
    }

    public void setNavigationBar(OSBarProperties navigationBar) {
        this.navigationBar = navigationBar;
    }

}

class OSBarProperties {
    private int x;

    private int y;

    private int width;

    private int height;

    private boolean visible;

    public int getX() {
        return x;
    }

    public void setX(int x) {
        this.x = x;
    }

    public int getY() {
        return y;
    }

    public void setY(int y) {
        this.y = y;
    }

    public int getWidth() {
        return width;
    }

    public void setWidth(int width) {
        this.width = width;
    }

    public int getHeight() {
        return height;
    }

    public void setHeight(int height) {
        this.height = height;
    }

    public boolean isVisible() {
        return visible;
    }

    public void setVisible(boolean visible) {
        this.visible = visible;
    }

    public OSBarProperties(int x, int y, int width, int height, boolean visible) {
        super();
        this.x = x;
        this.y = y;
        this.width = width;
        this.height = height;
        this.visible = visible;
    }
}
