package com.kms.katalon.core.webui.keyword.internal

import com.kms.katalon.core.aut.WebAUT
import com.kms.katalon.core.configuration.RunConfiguration
import com.kms.katalon.core.constants.CoreConstants
import com.kms.katalon.core.exception.StepFailedException
import com.kms.katalon.core.keyword.internal.AbstractKeyword
import com.kms.katalon.core.keyword.internal.KeywordExecutionContext
import com.kms.katalon.core.keyword.internal.KeywordExecutor
import com.kms.katalon.core.keyword.internal.SupportLevel
import com.kms.katalon.core.testobject.SelectorMethod
import com.kms.katalon.core.testobject.TestObject
import com.kms.katalon.core.webui.common.FindElementsResult
import com.kms.katalon.core.webui.common.WebUiCommonHelper
import com.kms.katalon.core.webui.common.controller.*
import com.kms.katalon.core.webui.constants.CoreWebuiMessageConstants
import com.kms.katalon.core.webui.driver.DriverFactory
import com.kms.katalon.core.webui.exception.WebElementNotFoundException
import com.kms.katalon.core.webui.model.*
import com.kms.katalon.core.webui.model.SeleniumActionRetryController.RetryContext
import com.kms.katalon.execution.selfhealing.healers.SelfHealingPipeline
import com.kms.katalon.execution.selfhealing.healers.SelfHealingPipeline.SelfHealingParams

import groovy.transform.CompileStatic
import org.apache.commons.collections4.CollectionUtils
import org.openqa.selenium.JavascriptExecutor
import org.openqa.selenium.ScriptTimeoutException
import org.openqa.selenium.WebDriver
import org.openqa.selenium.WebElement

public abstract class WebUIAbstractKeyword extends AbstractKeyword {

    @Override
    public SupportLevel getSupportLevel(Object ...params) {
        return SupportLevel.NOT_SUPPORT;
    }

    protected List<ElementWaitingPhase> getElementWaitingPhases(JavascriptExecutor jsExecutor) {
        List<ElementWaitingPhase> result = new ArrayList<>();
        result.add(new WaitForVisiblePhase(jsExecutor));
        result.add(new WaitForStablePhase(jsExecutor));
        result.add(new WaitForEnabledPhase(jsExecutor));
        result.add(new WaitForTopmostPhase(jsExecutor));
        return result;
    }

    @CompileStatic
    public static WebElement findWebElement(TestObject to, int timeOut = RunConfiguration.getElementTimeoutForWeb()) throws IllegalArgumentException, WebElementNotFoundException, StepFailedException {
        return WebUiCommonHelper.findWebElement(to, timeOut);
    }

    @CompileStatic
    public static List<WebElement> findWebElements(TestObject to, int timeOut) throws WebElementNotFoundException {
        return WebUiCommonHelper.findWebElements(to, timeOut);
    }

    @CompileStatic
    public static List<WebElement> findWebElements(WebDriver webDriver, TestObject testObject, long timeoutInMillis) throws WebElementNotFoundException {
        Exception defaultException = null;
        def params = new FindElementWithTimeoutParams(webDriver, testObject, timeoutInMillis);

        boolean shouldApplySelfHealing = RunConfiguration.shouldApplySelfHealing();
        if (!shouldApplySelfHealing) {
            return new WebUiElementFinderWithTimeout().findElements(params);
        }

        def ctrl = new SeleniumActionRetryController();
        def retryTimes = shouldApplySelfHealing ? 1 : 0;
        return ctrl.retry({retryContext ->
            if (retryContext.retryCount == 0) {
                try {
                    return new WebUiElementFinderWithTimeout().findElements(params);
                } catch (Exception error) {
                    defaultException = error;
                    throw error;
                }
            }

            def foundElements = this.findElemnentsWithSelfHealing(params);
            if (foundElements == null || foundElements.isEmpty()) {
                throw defaultException;
            }

            return foundElements;
        }, retryTimes);
    }

    @CompileStatic
    public static WebElement findWebElement(WebDriver driver, TestObject testObject, long timeoutInMillis) throws WebElementNotFoundException {
        Exception defaultException = null;
        def params = new FindElementWithTimeoutParams(driver, testObject, timeoutInMillis);

        boolean shouldApplySelfHealing = RunConfiguration.shouldApplySelfHealing();
        if (!shouldApplySelfHealing) {
            return new WebUiElementFinderWithTimeout().findElement(params);
        }

        def ctrl = new SeleniumActionRetryController();
        def retryTimes = shouldApplySelfHealing ? 1 : 0;
        return ctrl.retry({retryContext ->
            if (retryContext.retryCount == 0) {
                try {
                    return new WebUiElementFinderWithTimeout().findElement(params);
                } catch (Exception error) {
                    defaultException = error;
                    throw error;
                }
            }

            def foundElements = this.findElemnentsWithSelfHealing(params);
            def firstElement = foundElements != null && !foundElements.isEmpty() ? foundElements.get(0) : null;
            if (firstElement == null) {
                throw defaultException;
            }

            return firstElement;
        }, retryTimes);
    }

    private static List<WebElement> findElemnentsWithSelfHealing(FindElementWithTimeoutParams findElementParams) {
        def driver = findElementParams.getWebDriver();
        def testObject = findElementParams.getTestObject();
        def selfHealingParams = new SelfHealingParams();

        boolean isWebPlatform = KeywordExecutionContext.isRunningWebUI();
        String runningKeyword = KeywordExecutionContext.getRunningKeyword();
        List<String> excludedKeywords = RunConfiguration.getExcludedWebUIKeywordsFromSelfHealing();

        def isSelfHealingEnabled = isWebPlatform && !excludedKeywords.contains(runningKeyword) && RunConfiguration.shouldApplySelfHealing();
        if (!isSelfHealingEnabled) {
            return null;
        }

        selfHealingParams.useBasicSelfHealing = true;
        selfHealingParams.useAISelfHealing = RunConfiguration.getAISelfHealingEnabled();
        selfHealingParams.aiSelfHealingInputSources = RunConfiguration.getAISelfHealingInputSources();
        selfHealingParams.currentAction = runningKeyword;

        selfHealingParams.aut = new WebAUT(driver);
        selfHealingParams.testObject = testObject;

        selfHealingParams.basicSelfHealer = { tempTestObject ->
            def foundElement = new WebUiElementFinderWithSelfHealing().findElement(findElementParams);
            return foundElement != null ? Arrays.asList(foundElement) : null;
        };

        selfHealingParams.elementFinder = { tempTestObject ->
            def result = new WebUiElementFinder().findElementsBySelectedMethod(driver, tempTestObject);
            return result.elements;
        };

        def selfHealingPipeline = new SelfHealingPipeline();
        selfHealingPipeline.registerPhase({ context ->
            def foundElements = context.data.results.foundElements;

            if (foundElements == null || foundElements.isEmpty()) {
                return;
            }
            
            def firstElement = foundElements.get(0);

            def healedXPath = WebUiElementFinderWithSelfHealing.generateNewXPath(firstElement, driver);

            def healingResult = FindElementsResult.from(firstElement, healedXPath, SelectorMethod.XPATH, null);
            WebUiElementFinderWithSelfHealing.registerBrokenTestObject(healingResult, testObject, driver);
        });

        def context = selfHealingPipeline.execute(selfHealingParams);
        return context.results.foundElements;
    }

    @CompileStatic
    public static WebElement findWebElementWithoutRetry(WebDriver webDriver, TestObject testObject, long timeoutInMillis) throws WebElementNotFoundException {
        getKeywordExecutor().setCurrentKeywordTimeout(timeoutInMillis);
        def params = new FindElementParams(webDriver, testObject);
        return new WebUiElementFinder().findElement(params);
    }

    /**
     * Checks if the specified web element is interactable within a given timeout.
     *
     * @param scriptExecutor the JavaScript executor to run scripts
     * @param element the web element to wait for
     * @param timeoutInMillis the maximum time to wait in milliseconds
     * @param retryContext the context for retrying actions
     * @return true if the element is interactable, false otherwise
     */
    @CompileStatic
    public boolean waitElementInteractable(JavascriptExecutor scriptExecutor, WebElement element, long timeoutInMillis, RetryContext retryContext) {
        if (!RunConfiguration.getEnhancedWaitingEnabled()) {
            return true;
        }

        ElementWaitingPhase[] waitingPhases = getElementWaitingPhases(scriptExecutor);
        for (ElementWaitingPhase waitingPhase : waitingPhases) {
            try {
                boolean waitingResult = waitingPhase.wait(element, timeoutInMillis - retryContext.getElapsedTime());
                if (waitingResult == false) {
                    return false;
                }
            } catch (IllegalArgumentException | IllegalStateException | ScriptTimeoutException e) {
                logger.logWarning(e.getMessage());
                return false;
            }
        }
        return true;
    }

    protected WebDriver getWebDriver() {
        WebDriver webDriver = DriverFactory.getWebDriver()
        if (webDriver == null) {
            throw new StepFailedException(CoreWebuiMessageConstants.EXC_BROWSER_IS_NOT_OPENED)
        }
        return webDriver
    }

    private static KeywordExecutor getKeywordExecutor() {
        return KeywordExecutor.getInstance(CoreConstants.PLATFORM_WEB);
    }
}
