package com.kms.katalon.core.context.internal;

import java.io.File;
import java.io.IOException;
import java.text.MessageFormat;
import java.util.HashMap;
import java.util.Map;
import java.util.Stack;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.StringUtils;
import org.greenrobot.eventbus.Subscribe;
import org.greenrobot.eventbus.ThreadMode;
import org.openqa.selenium.WebDriver;

import com.kms.katalon.core.constants.CoreMessageConstants;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.event.EventBusSingleton;
import com.kms.katalon.core.event.TestingEvent;
import com.kms.katalon.core.helper.screenrecorder.CDTVideoRecorder;
import com.kms.katalon.core.helper.screenrecorder.VideoConfiguration;
import com.kms.katalon.core.helper.screenrecorder.VideoRecorder;
import com.kms.katalon.core.helper.screenrecorder.VideoRecorderBuilder;
import com.kms.katalon.core.helper.screenrecorder.VideoRecorderException;
import com.kms.katalon.core.helper.screenrecorder.VideoSubtitleWriter;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.logging.model.TestStatus;
import com.kms.katalon.core.setting.VideoRecorderSettings;
import com.kms.katalon.core.util.VideoRecorderUtil;
import com.kms.katalon.core.util.internal.ExceptionsUtil;
import com.kms.katalon.core.util.internal.TestOpsUtil;

public class VideoRecorderService implements ExecutionListenerEventHandler {

    private static final String VIDEO_KEYWORD_NAME = "Video";

    private static final String VIDEOS_FOLDER_NAME = "videos";

    private final KeywordLogger logger = KeywordLogger.getInstance(this.getClass());

    private String reportFolder;

    private VideoRecorderSettings videoRecorderSettings;

    private long actionStartTime = 0;

    private VideoRecorder videoRecorder;

    private VideoSubtitleWriter screenRecorderSubWriter;

    private int prevStepIndex = 0;

    private String prevStepName;

    private String prevStepDescription;

    private InternalTestCaseContext lastestTestCaseContext;

    public VideoRecorderService(String reportFolder, VideoRecorderSettings videoRecorderSettings) {
        this.reportFolder = reportFolder;
        this.videoRecorderSettings = videoRecorderSettings;
        EventBusSingleton.getInstance().getEventBus().register(this);
    }

    private boolean shouldRecord() {
        return isEnableScreenRecorder() || isEnableBrowserRecorder();
    }

    private boolean isEnableScreenRecorder() {
        return videoRecorderSettings.isEnable() && !videoRecorderSettings.isUseBrowserRecorder();
    }

    private boolean isEnableBrowserRecorder() {
        return videoRecorderSettings.isEnable() && videoRecorderSettings.isUseBrowserRecorder();
    }

    private boolean isEnableRecorder() {
        return videoRecorderSettings.isEnable();
    }

    @Subscribe(threadMode = ThreadMode.MAIN)
    public void onTestingEvent(TestingEvent event) {
        switch (event.getType()) {
            case BROWSER_OPENED:
            case URL_NAVIGATED:
                if (isEnableBrowserRecorder() && lastestTestCaseContext != null) {
                    startRecorder((WebDriver) event.getData());
                }
                break;
            default:
                break;
        }
    }

    @Override
    public void handleListenerEvent(ExecutionListenerEvent listenerEvent, Object[] testContext) {
        if (!shouldRecord()) {
            return;
        }
        switch (listenerEvent) {
            case BEFORE_TEST_CASE: {
                lastestTestCaseContext = (InternalTestCaseContext) testContext[0];
                if (!isMainTestCase() || isSkippedTestCase()) {
                    return;
                }
                startRecorder(null);
                break;
            }
            case AFTER_TEST_CASE: {
                lastestTestCaseContext = (InternalTestCaseContext) testContext[0];
                if (!isMainTestCase() || isSkippedTestCase()) {
                    return;
                }
                try {
                    flushSubTitles();
                    if (isEnableRecorder()) {
                        stopRecording();
                    }
                } catch (VideoRecorderException error) {
                    logger.logError(ExceptionsUtil.getStackTraceForThrowable(error));
                }
                break;
            }
            case BEFORE_TEST_STEP: {
                if (!isMainTestCase() || isSkippedTestCase()) {
                    return;
                }
                safeWriteSub((int) testContext[0], (String) testContext[1], (String) testContext[2]);
                break;
            }
            case AFTER_TEST_STEP: {
                break;
            }
            default:
                break;
        }
    }

    private void startRecorder(WebDriver webDriver) {
        try {
            // If the browser is still on the initial blank page, we will not start the video recorder
            if (isOnBlankPage(webDriver)) {
                return;
            }
            boolean browserRecording = isEnableBrowserRecorder();

            String videoFileName = genVideoFileName();
            String folderPath = getVideoFolder();

            VideoConfiguration videoConfiguration = videoRecorderSettings.toVideoConfiguration();

            VideoRecorder newRecorder = VideoRecorderBuilder.get()
                    .setVideoConfig(videoConfiguration)
                    .setOutputDirLocation(folderPath)
                    .setOutputVideoName(videoFileName)
                    .create();

            if (!browserRecording) {
                videoRecorder = newRecorder;
                startVideoSubWriter(folderPath, videoFileName);
                videoRecorder.start();
            } else {
                boolean isNewTestCase = videoRecorder == null || !StringUtils
                        .equals(videoRecorder.getCurrentVideoLocation(), newRecorder.getCurrentVideoLocation());

                if (webDriver == null && videoRecorder != null
                        && videoRecorder.getDelegate() instanceof CDTVideoRecorder) {
                    webDriver = ((CDTVideoRecorder) videoRecorder.getDelegate()).getDriver();
                }

                if (isNewTestCase) {
                    videoRecorder = newRecorder;
                    ((CDTVideoRecorder) videoRecorder.getDelegate()).setDriver(webDriver);
                    videoRecorder.start();

                    startVideoSubWriter(folderPath, videoFileName);
                } else {
                    ((CDTVideoRecorder) videoRecorder.getDelegate()).setDriver(webDriver);
                    videoRecorder.resume();
                }
            }
        } catch (VideoRecorderException error) {
            logger.logError(ExceptionsUtil.getStackTraceForThrowable(error));
        }
    }
    
    private boolean isOnBlankPage(WebDriver webDriver) {
        return webDriver != null && webDriver.getCurrentUrl() != null
                && webDriver.getCurrentUrl().equals("about:blank");
    }

    private String genVideoFileName() throws VideoRecorderException {
        String videoFolderName = getVideoFolder();
        int maxOSPathLength = VideoRecorderUtil.MAX_OS_FILE_PATH_LENGTH;
        if (videoFolderName.length() > maxOSPathLength) {
            throw new VideoRecorderException(String.format("Video folder path length exceeds max length %d, for: %s",
                    maxOSPathLength, videoFolderName));
        }

        int testCaseIndex = lastestTestCaseContext.getTestCaseIndex();
        String testCaseName = VideoRecorderUtil.getTestCaseNameFromFullPath(lastestTestCaseContext.getTestCaseName());
        testCaseName = VideoRecorderUtil.cleanTestCaseName(testCaseName);
        int retryIdx = lastestTestCaseContext.getRetryIndex();
        String suffix = VideoRecorderUtil.getScreenRecordingNameSuffix(testCaseIndex + 1, retryIdx);
        // Add the file extension
        int suffixLength = suffix.length();

        // remaining: The number of characters left for the file name
        int remaining = maxOSPathLength - videoFolderName.length();

        // minimumNeeded: The minimum file name
        int minimumNeeded = suffixLength + Math.min(VideoRecorderUtil.FIX_FILE_NAME_LENGTH, testCaseName.length());
        if (remaining < minimumNeeded) {
            throw new VideoRecorderException(
                    String.format("The combine path + filename exceeding %d characters.", maxOSPathLength));
        }

        return truncate(VideoRecorderUtil.FIX_FILE_NAME_LENGTH);
    }

    private String truncate(int maxTCNameLength) throws VideoRecorderException {
        return VideoRecorderUtil.getScreenRecordingNameWithFormat(lastestTestCaseContext, maxTCNameLength);
    }

    private void startVideoSubWriter(String videoFolderName, String videoFileName) {
        screenRecorderSubWriter = new VideoSubtitleWriter(new File(videoFolderName, videoFileName).getAbsolutePath());
    }

    private void flushSubTitles() {
        safeWriteSub(0, null, null);
    }

    private void safeWriteSub(int stepIndex, String stepDescription, String stepName) {
        try {
            writeSub(stepIndex, stepDescription, stepName);
        } catch (IOException error) {
            logger.logError(ExceptionsUtil.getStackTraceForThrowable(error));
        }
    }

    private void writeSub(int stepIndex, String stepDescription, String stepName) throws IOException {
        String description = StringUtils.defaultIfEmpty(prevStepDescription, prevStepName);
        if (StringUtils.isNotBlank(description) && screenRecorderSubWriter != null) {
            long subStartTime = Math.max(actionStartTime - getStartTime(), 0);
            long subEndTime = Math.max(System.currentTimeMillis() - getStartTime(), 0);
            if (getStartTime() == 0) { // Video Recorder was not started yet
                subStartTime = 0;
                subEndTime = 0;
            }
            screenRecorderSubWriter.writeSub(subStartTime, subEndTime,
                    String.format("%d. %s", prevStepIndex + 1, description));
        }

        prevStepIndex = stepIndex;
        prevStepName = stepName;
        prevStepDescription = stepDescription;
        actionStartTime = System.currentTimeMillis();
    }

    private void stopRecording() throws VideoRecorderException {
        boolean browserRecording = isEnableBrowserRecorder();
        if (videoRecorder == null || !videoRecorder.isStarted()) {
            return;
        }
        videoRecorder.stop();

        if (browserRecording) {
            if (isTestCaseSkipped() || (isTestCasePassed() && !videoRecorderSettings.isRecordAllTestCases())) {
                videoRecorder.delete();
                if (screenRecorderSubWriter != null) {
                    screenRecorderSubWriter.delete();
                }
            } else {
                String videoLocation = videoRecorder.getCurrentVideoLocation();
                logVideoRecordingStep(videoLocation);
            }
        } else {
            if (isTestCaseSkipped() || (isTestCasePassed() && !videoRecorderSettings.isRecordAllTestCases())) {
                videoRecorder.delete();
                if (screenRecorderSubWriter != null) {
                    screenRecorderSubWriter.delete();
                }
            } else {
                String videoLocation = videoRecorder.getCurrentVideoLocation();
                logVideoRecordingStep(videoLocation);
            }
        }
    }

    private long getStartTime() {
        if (isEnableBrowserRecorder()) {
            return videoRecorder.getStartTime();
        }
        if (isEnableScreenRecorder()) {
            return videoRecorder.getStartTime();
        }
        return 0;
    }

    private boolean isSkippedTestCase() {
        return lastestTestCaseContext != null && lastestTestCaseContext.isSkipped();
    }

    private boolean isMainTestCase() {
        return lastestTestCaseContext != null && lastestTestCaseContext.isMainTestCase();
    }

    private boolean isTestCaseFailed() {
        String testCaseStatus = lastestTestCaseContext.getTestCaseStatus();
        return TestStatus.TestStatusValue.valueOf(testCaseStatus) == TestStatus.TestStatusValue.FAILED;
    }

    private boolean isTestCasePassed() {
        String testCaseStatus = lastestTestCaseContext.getTestCaseStatus();
        return TestStatus.TestStatusValue.valueOf(testCaseStatus) == TestStatus.TestStatusValue.PASSED;
    }

    private boolean isTestCaseSkipped() {
        String testCaseStatus = lastestTestCaseContext.getTestCaseStatus();
        return TestStatus.TestStatusValue.valueOf(testCaseStatus) == TestStatus.TestStatusValue.SKIPPED;
    }

    private void logVideoRecordingStep(String videoLocation) {
        if (StringUtils.isBlank(videoLocation)) {
            return;
        }
        Map<String, String> attributes = new HashMap<>();
        attributes.put(StringConstants.XML_LOG_VIDEO_ATTACHMENT_PROPERTY,
                TestOpsUtil.getRelativePathForLog(videoLocation));
        String message = MessageFormat.format(CoreMessageConstants.EXEC_LOG_VIDEO_RECORDING_COMPLETED,
                lastestTestCaseContext.getTestCaseId());
        logger.startKeyword(VIDEO_KEYWORD_NAME, new HashMap<>(), new Stack<>());
        logger.logInfo(message, attributes);
        logger.endKeyword(VIDEO_KEYWORD_NAME, new HashMap<>(), new Stack<>());
    }

    private String getVideoFolder() {
        String rawPath = FilenameUtils.concat(reportFolder, VIDEOS_FOLDER_NAME);
        return FilenameUtils.separatorsToSystem(rawPath);
    }
}
