package com.kms.katalon.core.reporting.pdf;

import java.io.File;
import java.io.IOException;
import java.nio.file.Files;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.io.FileUtils;

import com.kms.katalon.core.logging.model.ILogRecord;
import com.kms.katalon.core.logging.model.TestCaseLogRecord;
import com.kms.katalon.core.logging.model.TestSuiteLogRecord;
import com.kms.katalon.core.reporting.pdf.exception.JasperReportException;
import com.kms.katalon.core.reporting.util.PDFUtils;
// import com.kms.katalon.core.reporting.util.PDFUtils;
import com.kms.katalon.logging.LogUtil;
import com.kms.katalon.util.ExecutorUtils;

public class TestSuitePdfGenerator extends AbstractPdfReportGenerator {
    private static final int REPORT_PART_GENERATION_POOL_SIZE = 10;

    private TestSuiteLogRecord fTestSuiteLogRecord;

    public TestSuitePdfGenerator(TestSuiteLogRecord testSuiteLogRecord) {
        fTestSuiteLogRecord = testSuiteLogRecord;
    }

    @Override
    protected ILogRecord[] getLogRecords() {
        return fTestSuiteLogRecord.filterFinalTestCasesResult();
    }

    @Override
    protected String getPrimaryTemplateLocation() {
        return TEST_SUITE_TPL;
    }

    @Override
    protected Map<String, Object> getAdditionalParams() {
        Map<String, Object> jasperParams = new HashMap<String, Object>();
        jasperParams.put("TEST_SUITE", fTestSuiteLogRecord);
        return jasperParams;
    }

    @Override
    public File exportToPDF(String fileLocation) throws JasperReportException, IOException {
        final String reportFolder = fTestSuiteLogRecord.getLogFolder();
        var tempReportFolder = Files.createTempDirectory("katalon-report-").toFile();

        List<String> reportParts = new ArrayList<String>();

        try {
            var suiteLocation = new File(tempReportFolder, "ts.pdf").getAbsolutePath();
            super.exportToPDF(suiteLocation, TEST_SUITE_TPL);
            reportParts.add(suiteLocation);

            List<String> testCaseParts = this.generateTestCaseParts(tempReportFolder, reportFolder);
            reportParts.addAll(testCaseParts);

            PDFUtils.mergePDFs(fileLocation, reportParts.toArray(new String[] {}));
        } finally {
            FileUtils.deleteQuietly(tempReportFolder);
        }

        return new File(fileLocation);
    }

    private List<String> generateTestCaseParts(File tempReportFolder, String reportFolder) {
        List<String> reportParts = Collections.synchronizedList(new ArrayList<String>());

        var testCaseLogRecords = fTestSuiteLogRecord.filterFinalTestCasesResult();
        var generateTestCaseDetailsPool = ExecutorUtils.createForkJoinPool(REPORT_PART_GENERATION_POOL_SIZE);

        int testCaseIndex = 0;
        for (var testCaseLogRecord : testCaseLogRecords) {
            if (!(testCaseLogRecord instanceof TestCaseLogRecord)) {
                continue;
            }
            var tempTestCaseName = MessageFormat.format("tc-{0}.pdf", testCaseIndex++);
            var tempTestCasePath = new File(tempReportFolder, tempTestCaseName).getAbsolutePath();
            reportParts.add(tempTestCasePath);
            generateTestCaseDetailsPool.submit(() -> {
                var testCasePDFGenerator = new TestCasePdfGenerator((TestCaseLogRecord) testCaseLogRecord,
                        reportFolder);
                try {
                    testCasePDFGenerator.exportToPDF(tempTestCasePath);
                } catch (JasperReportException | IOException error) {
                    LogUtil.logError(error);
                    reportParts.remove(tempTestCasePath);
                }
            });
        }

        ExecutorUtils.awaitWithGracefulShutdown(generateTestCaseDetailsPool);

        return reportParts;
    }
}
