package com.kms.katalon.core.webui.driver.firefox;

import static org.openqa.selenium.Platform.MAC;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.Platform;
import org.openqa.selenium.WebDriver;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.firefox.FirefoxProfile;
import org.openqa.selenium.firefox.GeckoDriverService;
import org.openqa.selenium.remote.AbstractDriverOptions;

import com.kms.katalon.core.webui.driver.AbstractDriverBuilder;
import com.kms.katalon.core.webui.driver.DriverFactory;
import com.kms.katalon.core.webui.util.FileUtil;
import com.kms.katalon.core.webui.util.FirefoxExecutable;
import com.kms.katalon.selenium.constant.SeleniumW3CCapabilityConstant;
import com.kms.katalon.selenium.driver.CFirefoxDriver;

public class FirefoxDriverBuilder extends AbstractDriverBuilder {

    private static final String SMART_LOCATOR_ADDON_PATH = "Firefox" + File.separator + "smartlocator.xpi";
    private static final String SMART_WAIT_ADDON_PATH = "Firefox" + File.separator + "smartwait.xpi";

    protected FirefoxOptions options;
    protected FirefoxProfile profile;
    protected GeckoDriverService driverService;

    public FirefoxDriverBuilder options(FirefoxOptions options) {
        this.options = options;
        return this;
    }

    public FirefoxDriverBuilder profile(FirefoxProfile profile) {
        this.profile = profile;
        return this;
    }

    public FirefoxDriverBuilder driverService(GeckoDriverService driverService) {
        this.driverService = driverService;
        return this;
    }

    @Override
    protected boolean canUseBiDi() {
        return true;
    }

    @Override
    protected AbstractDriverOptions<?> createOptions(Capabilities capabilities) {
        var options = this.options;
        if (options == null) {
            options = new FirefoxOptions(capabilities);
        }

        if (profile != null) {
            options.setProfile(profile);
        }

        return options;
    }

    @Override
    protected AbstractDriverOptions<?> addSmartWaitExtension(AbstractDriverOptions<?> options) throws IOException {
        return addExtension(options, SMART_WAIT_ADDON_PATH);
    }

    @Override
    protected AbstractDriverOptions<?> addSmartLocatorExtension(AbstractDriverOptions<?> options) throws IOException {
        return addExtension(options, SMART_LOCATOR_ADDON_PATH);
    }

    @Override
    protected WebDriver createDriver(AbstractDriverOptions<?> options) {
        if (!(options instanceof FirefoxOptions firefoxOptions)) {
            return null;
        }

        if (Platform.getCurrent().is(MAC)) {
            firefoxOptions.setBinary(FirefoxExecutable.MAC_FIREFOX_BINARY_FILE_LOCATION);
        }

        var driverPath = FirefoxDriverUtil.getDriverPath();
        if (StringUtils.isNotEmpty(driverPath)) {
            System.setProperty(DriverFactory.FIREFOX_DRIVER_PATH_PROPERTY_KEY, driverPath);
        }

        if (driverService == null) {
            return new CFirefoxDriver(firefoxOptions, driverConfigProvider.getActionDelayInMilisecond());
        }

        return new CFirefoxDriver(driverService, firefoxOptions, driverConfigProvider.getActionDelayInMilisecond());
    }

    private AbstractDriverOptions<?> addExtension(AbstractDriverOptions<?> options, String extensionPath) throws IOException {
        if (!(options instanceof FirefoxOptions firefoxOptions)) {
            return options;
        }

        var profile = firefoxOptions.getProfile();
        if (profile == null) {
            profile = new FirefoxProfile();
            firefoxOptions.setProfile(profile);
        }

        var extension = new File(FileUtil.getExtensionsDirectory(), extensionPath);
        profile.addExtension(extension);
        firefoxOptions.setProfile(profile);

        return firefoxOptions;
    }
    
    @Override
    protected Capabilities ignoreUnhandledPrompt(Capabilities capabilities) {
        var currentBehavior = capabilities.getCapability(SeleniumW3CCapabilityConstant.UNHANDLED_PROMPT_BEHAVIOR_CAP);

        if (null != currentBehavior) {
            return capabilities;
        }

        Map<String, String> userPromptHandler = Map.of(
                "alert", "ignore", 
                "beforeUnload", "ignore", 
                "confirm", "ignore",
                "default", "ignore", 
                "prompt", "ignore");

        var newCaps = new MutableCapabilities(capabilities);
        newCaps.setCapability(SeleniumW3CCapabilityConstant.UNHANDLED_PROMPT_BEHAVIOR_CAP, userPromptHandler);
        return newCaps;
    }
}
