package com.kms.katalon.core.util.internal;

import java.io.File;
import java.nio.file.InvalidPathException;
import java.nio.file.Path;
import java.text.MessageFormat;

import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;

public class NamingUtil {
    public static final String ELLIPSIS = "...";

    public static final int MAX_FILE_PATH_LENGTH = 255;

    public static final int DEFAULT_MAX_FILE_NAME_LENGTH = 25;

    public static File truncateAndSanitizePath(File file) throws InvalidPathException {
        return truncateAndSanitizePath(file, DEFAULT_MAX_FILE_NAME_LENGTH);
    }

    public static File truncateAndSanitizePath(File file, int maxNameLength) throws InvalidPathException {
        return truncateAndSanitizePath(file.toPath(), maxNameLength).toFile();
    }

    public static Path truncateAndSanitizePath(Path path) throws InvalidPathException {
        return truncateAndSanitizePath(path, DEFAULT_MAX_FILE_NAME_LENGTH);
    }

    public static Path truncateAndSanitizePath(Path path, int maxNameLength) throws InvalidPathException {
        return sanitizeFileName(makeEllipsisPath(path, maxNameLength));
    }

    public static String makeEllipsisPath(String path) throws InvalidPathException {
        return makeEllipsisPath(path, DEFAULT_MAX_FILE_NAME_LENGTH);
    }

    public static String makeEllipsisPath(String path, int maxNameLength) throws InvalidPathException {
        if (StringUtils.isBlank(path)) {
            return path;
        }
        return makeEllipsisPath(Path.of(path), maxNameLength).toString();
    }

    public static File makeEllipsisPath(File file) throws InvalidPathException {
        return makeEllipsisPath(file, DEFAULT_MAX_FILE_NAME_LENGTH);
    }

    public static File makeEllipsisPath(File file, int maxNameLength) throws InvalidPathException {
        if (file == null) {
            return file;
        }
        return makeEllipsisPath(file.toPath(), maxNameLength).toFile();
    }

    public static Path makeEllipsisPath(Path path) throws InvalidPathException {
        return makeEllipsisPath(path, DEFAULT_MAX_FILE_NAME_LENGTH);
    }

    public static Path makeEllipsisPath(Path path, int maxNameLength) throws InvalidPathException {
        if (path == null) {
            return path;
        }
        String absolutePath = path.getParent().toAbsolutePath().toString();
        int remaining = MAX_FILE_PATH_LENGTH - absolutePath.length();

        if (remaining <= 0) {
            throw new InvalidPathException(absolutePath, MessageFormat.format("Maximum length reached ({0}/{1})",
                    absolutePath.length(), MAX_FILE_PATH_LENGTH));
        }

        String ellipsisName = makeEllipsisFileName(path.getFileName().toString(), Math.min(maxNameLength, remaining));
        return path.getParent().resolve(ellipsisName);
    }

    public static String makeEllipsisFileName(String fileName) {
        return makeEllipsisFileName(fileName, DEFAULT_MAX_FILE_NAME_LENGTH);
    }

    public static String makeEllipsisFileName(String fileName, int maxLength) {
        String baseName = FilenameUtils.getBaseName(fileName);
        String extension = FilenameUtils.getExtension(fileName);
        String extensionPart = StringUtils.isBlank(extension) ? "" : ("." + extension);
        return makeEllipsisName(baseName, maxLength - extensionPart.length()) + extensionPart;
    }

    public static String makeEllipsisName(String name) {
        return makeEllipsisName(name, DEFAULT_MAX_FILE_NAME_LENGTH);
    }

    /**
     * @param name a non-extension name (e.g. "abc" - and not "abc.txt").
     * If the name includes the extension, then use "NamingUtil.makeEllipsisFileName(...)" instead.
     * This way, we can reuse this method for other kind of names.
     */
    public static String makeEllipsisName(String name, int maxLength) {
        if (StringUtils.defaultString(name).length() <= maxLength) {
            return name;
        }

        int remaining = maxLength - ELLIPSIS.length();
        if (remaining <= 1) { // Not enough space to insert the ellipsis
            return name.substring(0, maxLength);
        }

        int halfLength = Math.floorDiv(remaining, 2);

        String firstPart = StringUtils.trimToEmpty(name.substring(0, halfLength));
        int firstExtraLength = halfLength - firstPart.length();

        // If we can trim out some whitespace from the first part, then give it to the second part.
        int secondPartLength = halfLength + firstExtraLength;
        String secondPart = StringUtils.trimToEmpty(name.substring(name.length() - secondPartLength, name.length()));
        int secondExtraLength = secondPartLength - secondPart.length();

        // If we can trim out some whitespace from the second part, then give it to the first part.
        // If the first part doesn't need that, then it will be ignored.
        if (secondExtraLength > 0) {
            firstPart = StringUtils.trimToEmpty(name.substring(0, halfLength + secondExtraLength));
        }

        return firstPart + NamingUtil.ELLIPSIS + secondPart;
    }

    public static File sanitizeFileName(File file) {
        return new File(file.getParentFile(), sanitizeFileName(file.getName()));
    }

    public static Path sanitizeFileName(Path path) {
        return path.getParent().resolve(sanitizeFileName(path.getFileName().toString()));
    }

    public static String sanitizeFileName(String name) {
        return name.replaceAll("\\s+", "-");
    }

    public static String getUniqueFileName(String preferredName, String location) {
        return getUniqueFileName(preferredName, location, null);
    }

    public static String getUniqueFileName(String preferredName, String location, Integer sequenceNumber) {
        File parent = new File(location);
        if (!parent.exists()) {
            return preferredName;
        }

        do {
            String fileNameToBeTested = buildSequenceName(preferredName, sequenceNumber);
            File file = new File(FilenameUtils.concat(parent.getAbsolutePath(), fileNameToBeTested));
            if (file.exists()) {
                sequenceNumber = sequenceNumber != null
                        ? sequenceNumber + 1
                        : 1;
                continue;
            }
            return file.getName();
        } while (true);
    }

    private static String buildSequenceName(String preferredName, Integer sequenceNumber) {
        return sequenceNumber != null
                ? MessageFormat.format("{0} ({1})", preferredName, sequenceNumber)
                : preferredName;
    }
}
