package com.kms.katalon.core.mobile.helper;

import java.text.MessageFormat;
import java.time.Duration;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.openqa.selenium.By;
import org.openqa.selenium.Dimension;
import org.openqa.selenium.Rectangle;
import org.openqa.selenium.TimeoutException;
import org.openqa.selenium.WebElement;
import org.openqa.selenium.remote.RemoteWebElement;

import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.mobile.constants.StringConstants;
import com.kms.katalon.core.mobile.driver.AppiumSessionCollector;
import com.kms.katalon.core.mobile.exception.MobileException;
import com.kms.katalon.core.mobile.exception.WebElementNotFoundException;
import com.kms.katalon.core.mobile.keyword.internal.MobileDriverFactory;
import com.kms.katalon.core.testobject.TestObject;
import com.kms.katalon.core.util.internal.JsonUtil;

import io.appium.java_client.AppiumBy;
import io.appium.java_client.AppiumDriver;
import io.appium.java_client.ios.IOSDriver;

public final class IOSHelper {

    private IOSHelper() {
    }

    private static final KeywordLogger logger = KeywordLogger.getInstance(IOSHelper.class);

    private static final List<String> IOS_SCROLLABLE_CLASSES = Arrays.asList("XCUIElementTypeScrollView",
            "XCUIElementTypeTable", "XCUIElementTypeCollectionView", "XCUIElementTypeWebView");

    public static int getStatusBarHeight(AppiumDriver driver) {
        int statusBarHeight = 0;
        IOSDriver iosDriver = (IOSDriver) driver;
        try {
            DeviceScreenInfo deviceScreenInfo = getDeviceScreenInfo(iosDriver);
            StatusBarSize statusBarSize = deviceScreenInfo.getStatusBarSize();
            statusBarHeight = (int) (statusBarSize.getHeight() * deviceScreenInfo.getScale());
        } catch (Exception e) {
            logger.logInfo(StringConstants.KW_LOG_FAILED_GET_DEVICE_SCREEN_INFO + " " + e.getMessage());
            try {
                Dimension screenSize = driver.manage().window().getSize();
                DeviceViewPortRect deviceViewPortRect = getDeviceViewPortRect(iosDriver);
                int scale = deviceViewPortRect.getWidth() / screenSize.getWidth();
                statusBarHeight = deviceViewPortRect.getTop() / scale;
            } catch (Exception ex) {
                logger.logInfo(StringConstants.KW_LOG_FAILED_GET_DEVICE_VIEWPORT_RECT + " " + ex.getMessage());
                logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_FAILED_GET_OS_STATUSBAR, "Screenshot"));
            }
        }
        return statusBarHeight;
    }

    public static float getScaleFactor(AppiumDriver driver) {
        float scaleFactor = 1;
        try {
            DeviceScreenInfo deviceScreenInfo = IOSHelper.getDeviceScreenInfo(driver);
            scaleFactor = deviceScreenInfo.getScale();
        } catch (Exception e) {
            logger.logInfo(StringConstants.KW_LOG_FAILED_GET_DEVICE_SCREEN_INFO + " " + e.getMessage());
            try {
                Dimension screenSize = driver.manage().window().getSize();
                DeviceViewPortRect deviceViewPortRect;
                deviceViewPortRect = IOSHelper.getDeviceViewPortRect(driver);
                scaleFactor = deviceViewPortRect.getWidth() / screenSize.getWidth();
            } catch (Exception ex) {
                logger.logInfo(StringConstants.KW_LOG_FAILED_GET_DEVICE_VIEWPORT_RECT + " " + e.getMessage());
                logger.logInfo(StringConstants.KW_LOG_FAILED_GET_SCALE_FACTOR);
            }
        }
        return scaleFactor;
    }

    public static String getActiveAppBundleIdFromSession(AppiumDriver driver) {
        return (String) AppiumSessionCollector.getSession(driver)
                .getProperties()
                .get(MobileCommonHelper.PROPERTY_NAME_IOS_BUNDLE_ID);
    }

    public static WebElement scrollToText(TestObject testObject, String text, int timeoutInSeconds)
            throws WebElementNotFoundException {
        long endTime = System.currentTimeMillis() + timeoutInSeconds * 1000L;

        IOSDriver driver = (IOSDriver) MobileDriverFactory.getDriver();

        Duration originalImplicitWait = driver.manage().timeouts().getImplicitWaitTimeout();

        WebElement scrollableElement = null;
        if (testObject != null) {
            try {
                scrollableElement = MobileCommonHelper.findElement(driver, testObject, 0);
            } catch (Exception e) {}
        }

        if (scrollableElement == null) {
            scrollableElement = findScrollableElement(driver);
        }

        if (scrollableElement == null || !IOS_SCROLLABLE_CLASSES.contains(scrollableElement.getAttribute("type"))) {
            throw new WebElementNotFoundException("Scrollable element not found");
        }

        WebElement targetElement = null;

        try {
            // Step 1: Try to find the element directly
            try {
                targetElement = findElementContainingText(driver, text);
                if (targetElement != null) {
                    return targetElement;
                }
            } catch (Exception ignore) {}

            long remainingTime = endTime - System.currentTimeMillis();
            if (remainingTime <= 0) {
                logger.logWarning("Timeout.");
                return null;
            }

            // Step 3: Scroll up to top (try several times)
            int lastY = 0;
            WebElement firstCell = findFirstCell(scrollableElement);
            lastY = firstCell.getLocation().getY();
            while (true) {
                remainingTime = endTime - System.currentTimeMillis();
                if (remainingTime <= 0) {
                    throw new TimeoutException();
                }

                scrollUp(driver, scrollableElement);

                firstCell = findFirstCell(scrollableElement);
                int currentY = firstCell.getLocation().getY();
                if (currentY >= 0 && currentY == lastY) {
                    logger.logInfo("Reached top of table");
                    break;
                }

                lastY = currentY;
            }

            // Step 3: Scroll and search
            WebElement lastCell = findLastCell(scrollableElement);
            lastY = lastCell.getLocation().getY();
            while (true) {
                remainingTime = endTime - System.currentTimeMillis();
                if (remainingTime <= 0) {
                    throw new TimeoutException();
                }

                try {
                    targetElement = findElementContainingText(driver, text);
                    if (targetElement != null) {
                        return targetElement;
                    }
                } catch (Exception ignore) {}

                scrollDown(driver, scrollableElement);

                lastCell = findLastCell(scrollableElement);
                int currentY = lastCell.getLocation().getY();
                if (currentY == lastY) {
                    logger.logInfo("Reached bottom of table");
                    break;
                }

                lastY = currentY;
            }

            throw new WebElementNotFoundException(String.format("Element with text '%s' not found", text));
        } finally {
            driver.manage().timeouts().implicitlyWait(originalImplicitWait);
        }
    }

    private static WebElement findFirstCell(WebElement scrollableElement) {
        return scrollableElement.findElement(By.xpath(".//XCUIElementTypeCell[1]"));
    }

    private static WebElement findLastCell(WebElement scrollableElement) {
        List<WebElement> cells = scrollableElement.findElements(By.xpath(".//XCUIElementTypeCell"));
        return cells.get(cells.size() - 1);
    }

    private static void scrollUp(AppiumDriver driver, WebElement scrollableElement) {
        String elementId = ((RemoteWebElement) scrollableElement).getId();
        Map<String, Object> args = new HashMap<>();
        args.put("element", elementId);
        args.put("direction", "up");
        driver.executeScript("mobile: scroll", args);
    }

    private static void scrollDown(AppiumDriver driver, WebElement scrollableElement) {
        String elementId = ((RemoteWebElement) scrollableElement).getId();
        Map<String, Object> args = new HashMap<>();
        args.put("element", elementId);
        args.put("direction", "down");
        driver.executeScript("mobile: scroll", args);
    }

    private static WebElement findScrollableElement(AppiumDriver driver) {
        for (String className : IOS_SCROLLABLE_CLASSES) {
            driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));
            List<WebElement> elements = driver.findElements(AppiumBy.iOSClassChain("**/" + className));
            for (WebElement el : elements) {
                if ("true".equals(el.getAttribute("enabled")) && el.isDisplayed()) {
                    return el;
                }
            }
        }

        return null; // no scrollable element found
    }

    private static WebElement findElementContainingText(AppiumDriver driver, String text) {
        String escapedText = text.replace("'", "\\'"); // Escape single quotes for NSPredicate safety
        String predicateString = String.format("(label == '%s' OR name == '%s' OR value == '%s') AND hittable == 1",
                escapedText, escapedText, escapedText);
        driver.manage().timeouts().implicitlyWait(Duration.ofSeconds(0));
        return driver.findElement(AppiumBy.iOSNsPredicateString(predicateString));
    }

    private static DeviceScreenInfo getDeviceScreenInfo(AppiumDriver driver) throws MobileException {
        IOSDriver iosDriver = (IOSDriver) driver;
        Object info = iosDriver.executeScript("mobile: deviceScreenInfo");
        if (info == null) {
            throw new MobileException("The raw data is null.");
        }

        DeviceScreenInfo deviceScreenInfo = JsonUtil.fromJson(info.toString(), DeviceScreenInfo.class);
        if (deviceScreenInfo == null) {
            throw new MobileException("Failed to parse the raw data to object.");
        }

        return deviceScreenInfo;
    }

    private static DeviceViewPortRect getDeviceViewPortRect(AppiumDriver driver) throws MobileException {
        IOSDriver iosDriver = (IOSDriver) driver;
        Object rect = iosDriver.executeScript("mobile: viewportRect");
        if (rect == null) {
            throw new MobileException("The raw data is null.");
        }

        DeviceViewPortRect deviceViewPortRect = JsonUtil.fromJson(rect.toString(), DeviceViewPortRect.class);
        if (deviceViewPortRect == null) {
            throw new MobileException("Failed to parse the raw data to object.");
        }

        return deviceViewPortRect;
    }

    public static boolean isElementInViewport(AppiumDriver driver, WebElement element) {
        try {
            if (element == null) {
                return false;
            }

            // Get screen dimensions
            Dimension screenSize = driver.manage().window().getSize();
            int screenHeight = screenSize.getHeight();
            int screenWidth = screenSize.getWidth();

            // Get element bounds
            Rectangle rect = element.getRect();
            int elemTop = rect.getY();
            int elemBottom = elemTop + rect.getHeight();
            int elemLeft = rect.getX();
            int elemRight = elemLeft + rect.getWidth();

            // Check full visibility inside screen bounds
            return elemTop >= 0 && elemBottom <= screenHeight && elemLeft >= 0 && elemRight <= screenWidth;
        } catch (Exception e) {
            return false;
        }
    }
}

class DeviceViewPortRect {
    public int left;

    public int top;

    public int width;

    public int height;

    public int getLeft() {
        return left;
    }

    public void setLeft(int left) {
        this.left = left;
    }

    public int getTop() {
        return top;
    }

    public void setTop(int top) {
        this.top = top;
    }

    public int getWidth() {
        return width;
    }

    public void setWidth(int width) {
        this.width = width;
    }

    public int getHeight() {
        return height;
    }

    public void setHeight(int height) {
        this.height = height;
    }
}

class DeviceScreenInfo {
    // https://developer.apple.com/documentation/uikit/uiscreen/1617836-scale
    public float scale;

    public StatusBarSize statusBarSize;

    public float getScale() {
        return scale;
    }

    public void setScale(int scale) {
        this.scale = scale;
    }

    public StatusBarSize getStatusBarSize() {
        return statusBarSize;
    }

    public void setStatusBarSize(StatusBarSize statusBarSize) {
        this.statusBarSize = statusBarSize;
    }
}

// https://developer.apple.com/documentation/xctest/xcuielementtypequeryprovider/1500428-statusbars
class StatusBarSize {
    public int width;

    public int height;

    public int getWidth() {
        return width;
    }

    public void setWidth(int width) {
        this.width = width;
    }

    public int getHeight() {
        return height;
    }

    public void setHeight(int height) {
        this.height = height;
    }
}
