package com.kms.katalon.core.main;

import static com.kms.katalon.core.constants.StringConstants.DF_CHARSET;

import java.io.File;
import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintStream;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Stack;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.codehaus.groovy.ast.MethodNode;
import org.codehaus.groovy.control.CompilationFailedException;
import org.codehaus.groovy.control.MultipleCompilationErrorsException;

import com.google.common.base.Optional;
import com.kms.katalon.core.annotation.SetUp;
import com.kms.katalon.core.annotation.SetupTestCase;
import com.kms.katalon.core.annotation.TearDown;
import com.kms.katalon.core.annotation.TearDownIfError;
import com.kms.katalon.core.annotation.TearDownIfFailed;
import com.kms.katalon.core.annotation.TearDownIfPassed;
import com.kms.katalon.core.annotation.TearDownTestCase;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.constants.CoreConstants;
import com.kms.katalon.core.constants.StringConstants;
import com.kms.katalon.core.context.internal.ExecutionEventManager;
import com.kms.katalon.core.context.internal.ExecutionListenerEvent;
import com.kms.katalon.core.context.internal.InternalTestCaseContext;
import com.kms.katalon.core.driver.internal.DriverCleanerCollector;
import com.kms.katalon.core.exception.StepFailedException;
import com.kms.katalon.core.helper.ChromeSnapshotHelper;
import com.kms.katalon.core.keyword.internal.KeywordExecutionContext;
import com.kms.katalon.core.logging.ErrorCollector;
import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.logging.KeywordLogger.KeywordStackElement;
import com.kms.katalon.core.logging.LogLevel;
import com.kms.katalon.core.logging.model.TestStatus;
import com.kms.katalon.core.logging.model.TestStatus.TestStatusValue;
import com.kms.katalon.core.model.FailureHandling;
import com.kms.katalon.core.testcase.TestCase;
import com.kms.katalon.core.testcase.TestCaseBinding;
import com.kms.katalon.core.testcase.TestCaseFactory;
import com.kms.katalon.core.testcase.VariableReport;
import com.kms.katalon.core.util.BrowserMobProxyManager;
import com.kms.katalon.core.util.internal.ExceptionsUtil;
import com.kms.katalon.core.util.internal.JsonUtil;
import com.kms.katalon.util.NameUtil;

import groovy.lang.Binding;
import groovy.util.ResourceException;
import groovy.util.ScriptException;

public class TestCaseExecutor {
    private final KeywordLogger logger = KeywordLogger.getInstance(this.getClass());

    private static ErrorCollector errorCollector = ErrorCollector.getCollector();

    protected TestResult testCaseResult;

    private TestCase testCase;

    private Stack<KeywordStackElement> keywordStack;

    private TestCaseMethodNodeCollector methodNodeCollector;

    private List<Throwable> parentErrors;

    protected ScriptEngine engine;

    protected Binding variableBinding;

    private TestCaseBinding testCaseBinding;

    private ExecutionEventManager eventManager;

    private boolean doCleanUp;

    private InternalTestCaseContext testCaseContext;

    private TestSuiteExecutor testSuiteExecutor;

    private RetrySettings retrySettings;

    private ChromeSnapshotHelper chromeSnapshotTool;

    private List<VariableReport> dataBindingReport = new ArrayList<VariableReport>();

    private List<String> secureValues = new ArrayList<String>();

    private Pattern secureProtectedGlobalValuesPattern;

    private Pattern secureValuesPattern;

    PrintStream originalOutStream = System.out;

    PrintStream originalErrStream = System.err;

    public void setTestSuiteExecutor(TestSuiteExecutor testSuiteExecutor) {
        this.testSuiteExecutor = testSuiteExecutor;
    }

    public TestCaseExecutor(TestCaseBinding testCaseBinding, ScriptEngine engine, ExecutionEventManager eventManager,
            InternalTestCaseContext testCaseContext, boolean doCleanUp) {
        this.testCaseBinding = testCaseBinding;
        this.engine = engine;
        this.testCase = TestCaseFactory.findTestCase(testCaseBinding.getTestCaseId());
        this.doCleanUp = doCleanUp;
        this.eventManager = eventManager;
        this.testCaseContext = testCaseContext;
        this.chromeSnapshotTool = new ChromeSnapshotHelper();

        getIterationValueByIterationName();
    }

    public TestCaseExecutor(TestCaseBinding testCaseBinding, ScriptEngine engine, ExecutionEventManager eventManager,
            InternalTestCaseContext testCaseContext) {
        this(testCaseBinding, engine, eventManager, testCaseContext, false);
    }

    private void preExecution() {
        secureValues.clear();

        Map<String, Object> protectedGlobalVariables = TestCaseMain.getProtectedGlobalVariables();
        for (Map.Entry<String, Object> entry : protectedGlobalVariables.entrySet()) {
            Object value = (Object) protectedGlobalVariables.get(entry.getKey());
            if (value != null) {
                String trimmed = String.valueOf(value).trim();
                trimmed = trimmed.replaceAll("^'|'$", "");
                addSecureValue(trimmed);
            }
        }

        secureValuesPattern = generateSecureValuesPattern(secureValues);
        logger.setSecureValuesPattern(secureValuesPattern);

        System.setOut(new PrintStream(new InterceptingOutputStream(System.out)));
        System.setErr(new PrintStream(new InterceptingOutputStream(System.err)));

        testCaseResult = TestResult.getDefault();
        keywordStack = new Stack<KeywordLogger.KeywordStackElement>();
        parentErrors = errorCollector.getCoppiedErrors();
        errorCollector.clearErrors();
        if (testCaseContext.isMainTestCase()) {
            KeywordExecutionContext.setHasHealedSomeObjects(false);
            KeywordExecutionContext.resetKeywordsUsage();
            KeywordExecutionContext.resetUsingApplitools();
        }
    }

    private void onExecutionComplete() {
        endAllUnfinishedKeywords(keywordStack);
        internallyRunMethods(methodNodeCollector.getMethodNodeWrapper(TearDownIfPassed.class));
        internallyRunMethods(methodNodeCollector.getMethodNodeWrapper(TearDown.class));
        logger.logPassed(testCase.getTestCaseId());
    }

    private void onExecutionError(Throwable t) {
        if (!keywordStack.isEmpty()) {
            logErrorForTheLastError();
            endAllUnfinishedKeywords(keywordStack);
        }

        testCaseResult.setCause(t);
        testCaseResult.getTestStatus().setStatusValue(getResultByError(t));
        String stackTraceForThrowable;
        try {
            stackTraceForThrowable = ExceptionsUtil.getStackTraceForThrowable(t);
        } catch (Exception e) {
            stackTraceForThrowable = ExceptionsUtil.getStackTraceForThrowable(t);
        }
        String message = MessageFormat.format(StringConstants.MAIN_LOG_MSG_FAILED_BECAUSE_OF, testCase.getTestCaseId(),
                stackTraceForThrowable);

        testCaseResult.setMessage(message);
        logError(t, message);

        runTearDownMethodByError(t);
    }

    private void logErrorForTheLastError() {
        if (!errorCollector.containsErrors()) {
            return;
        }
        Throwable t = errorCollector.getLastError();
        String stackTraceForThrowable;
        try {
            stackTraceForThrowable = ExceptionsUtil.getStackTraceForThrowable(t);
        } catch (Exception e) {
            stackTraceForThrowable = ExceptionsUtil.getStackTraceForThrowable(t);
        }
        String message = MessageFormat.format(StringConstants.MAIN_LOG_MSG_FAILED_BECAUSE_OF, testCase.getTestCaseId(),
                stackTraceForThrowable);
        if (!(t instanceof StepFailedException)) {
            logError(t, message);
        }
    }

    private boolean processScriptPreparationPhase() {
        // Collect AST nodes for script of test case
        try {
            methodNodeCollector = new TestCaseMethodNodeCollector(testCase);
        } catch (IOException ioException) {
            onSetupError(ioException);
            return false;
        } catch (MultipleCompilationErrorsException multiCompilationErrException) {
            onSetupError(multiCompilationErrException);
            return false;
        }

        try {
            variableBinding = collectTestCaseVariables();
        } catch (CompilationFailedException e) {
            onSetupError(e);
            return false;
        }

        return true;
    }

    private boolean processSetupPhase() {
        // Run setup method
        internallyRunMethods(methodNodeCollector.getMethodNodeWrapper(SetUp.class));
        boolean setupFailed = errorCollector.containsErrors();
        if (setupFailed) {
            internallyRunMethods(methodNodeCollector.getMethodNodeWrapper(TearDownIfError.class));
            internallyRunMethods(methodNodeCollector.getMethodNodeWrapper(TearDown.class));
            onSetupError(errorCollector.getFirstError());
        }
        return !setupFailed;
    }

    protected File getScriptFile() throws IOException {
        return new File(testCase.getGroovyScriptPath());
    }

    private void onSetupError(Throwable t) {
        String message = MessageFormat.format(StringConstants.MAIN_LOG_MSG_ERROR_BECAUSE_OF, testCase.getTestCaseId(),
                ExceptionsUtil.getMessageForThrowable(t));
        testCaseResult.setMessage(message);
        testCaseResult.getTestStatus().setStatusValue(TestStatusValue.ERROR);
        logger.logError(message, null, t);
    }

    private void postExecution() {
        boolean hasHealedSomeObjects = KeywordExecutionContext.hasHealedSomeObjects();
        if (hasHealedSomeObjects
                && (RunConfiguration.shouldApplySelfHealing() || RunConfiguration.shouldApplySelfHealingForMobile())) {
            logger.logInfo(StringUtils.EMPTY);
            logger.logInfo(StringConstants.SELF_HEALING_REPORT_AVAILABLE_OPENING);
            logger.logInfo(StringConstants.SELF_HEALING_REPORT_VISIT_INSIGHT_PART);
            logger.logInfo(StringConstants.SELF_HEALING_REFER_TO_DOCUMENT);
            logger.logInfo(StringConstants.SELF_HEALING_REPORT_AVAILABLE_ENDING);
        }

        errorCollector.clearErrors();
        errorCollector.getErrors().addAll(0, parentErrors);
        if (testCaseContext.isMainTestCase()) {
            BrowserMobProxyManager.shutdownProxy();
        }

        System.setOut(originalOutStream);
        System.setErr(originalErrStream);
    }

    @SuppressWarnings("unchecked")
    public TestResult execute(FailureHandling flowControl) {
        Map<String, Object> initialTestCaseBindedValues = new HashMap<>(
                Optional.fromNullable(testCaseBinding.getBindedValues()).or(new HashMap<String, Object>()));

        try {
            preExecution();

            if (testCaseContext.isMainTestCase()) {
                logger.startTest(testCase.getTestCaseId(),
                        getTestCaseProperties(testCaseBinding, testCase, flowControl), keywordStack);
            } else {
                logger.startCalledTest(testCase.getTestCaseId(),
                        getTestCaseProperties(testCaseBinding, testCase, flowControl), keywordStack);
            }

            if (hasTestCaseScript()) {
                if (!processScriptPreparationPhase()) {
                    return testCaseResult;
                }
                testCaseContext.setTestCaseStatus(testCaseResult.getTestStatus().getStatusValue().name());
                testCaseContext.setTestCaseVariables(variableBinding.getVariables());
            } else {
                testCaseContext.skipThisTestCase();
            }

            if (testCaseContext.isMainTestCase()) {
                eventManager.publicEvent(ExecutionListenerEvent.BEFORE_TEST_CASE, new Object[] { testCaseContext });
            }

            // Expose current test case ID test script
            RunConfiguration.getExecutionProperties()
                    .put(RunConfiguration.CURRENT_TESTCASE, testCaseContext.getTestCaseId());

            // By this point, @BeforeTestCase annotated method has already been called
            if (testCaseContext.isSkipped() == false) {
                testCaseResult = invokeTestSuiteMethod(SetupTestCase.class.getName(), StringConstants.LOG_SETUP_ACTION,
                        false, testCaseResult);
                if (ErrorCollector.getCollector().containsErrors()) {
                    Throwable error = ErrorCollector.getCollector().getFirstError();
                    testCaseResult.setMessage(ExceptionsUtil.getStackTraceForThrowable(error));
                    logger.logError(testCaseResult.getMessage(), null, error);
                    return testCaseResult;
                }

                accessMainPhase();

                invokeTestSuiteMethod(TearDownTestCase.class.getName(), StringConstants.LOG_TEAR_DOWN_ACTION, true,
                        testCaseResult);
            } else {
                TestStatus testStatus = new TestStatus();
                testStatus.setStatusValue(TestStatusValue.SKIPPED);
                testStatus.setStackTrace(StringConstants.TEST_CASE_SKIPPED);
                testCaseResult.setTestStatus(testStatus);
            }
            return testCaseResult;
        } finally {

            // Notify test execution server about test failure here if executing a Test Case
            if (!testCaseResult.getTestStatus().getStatusValue().equals(TestStatusValue.PASSED)
                    && RunConfiguration.shouldApplyTimeCapsule()) {
                String logFolderPath = logger.getLogFolderPath();
                String testArtifactFolderPath = StringUtils.isEmpty(logFolderPath) ? RunConfiguration.getProjectDir()
                        : logFolderPath;
                try {
                    this.chromeSnapshotTool.captureSnapshot(testArtifactFolderPath, testCaseContext.getTestCaseId());
                } catch (Exception error) {
                    logger.logWarning(error.getMessage(), null, error);
                }
            }

            testCaseContext.setTestCaseStatus(testCaseResult.getTestStatus().getStatusValue().name());
            testCaseContext.setMessage(testCaseResult.getMessage());

            if (testCaseContext.isMainTestCase()) {
                eventManager.publicEvent(ExecutionListenerEvent.AFTER_TEST_CASE, new Object[] { testCaseContext });
            }

            boolean shouldRetry = false;

            if (testCaseContext.isMainTestCase()) {
                if (testCaseContext.isSkipped()) {
                    logger.logSkipped(testCaseResult.getMessage());
                } else {
                    TestStatusValue testCaseStatus = testCaseResult.getTestStatus().getStatusValue();
                    switch (testCaseStatus) {
                        case FAILED:
                        case ERROR:
                            if (retrySettings != null && retrySettings.getRemainingRetryCount() > 0) {
                                shouldRetry = true;
                            }
                            if (testSuiteExecutor != null) {
                                testSuiteExecutor.updateFailedThreshold();
                            }
                        default:
                            break;
                    }
                }

                Map<String, String> attributes = new HashMap<>();
                if (retrySettings != null) {
                    attributes.put(CoreConstants.TEST_CASE_CURRENT_RETRY_COUNT,
                            Integer.toString(retrySettings.getCurrentRetryCount()));
                    attributes.put(CoreConstants.TEST_CASE_REMAINING_RETRY_COUNT,
                            Integer.toString(retrySettings.getRemainingRetryCount()));
                }
                Object keywordsUsage = KeywordExecutionContext.getKeywordsUsage();
                attributes.put(StringConstants.EXCUTION_KEYWORDS_USAGE, JsonUtil.toJson(keywordsUsage));
                
                Object executedKeywords = KeywordExecutionContext.getExecutedKeywords();
                attributes.put(StringConstants.EXECUTED_KEYWORDS, JsonUtil.toJson(executedKeywords));
                
                attributes.put(CoreConstants.PLATFORM_APPLITOOLS,
                        String.valueOf(KeywordExecutionContext.isUsingApplitools()));
                attributes.put(StringConstants.EXECUTION_BINDING_VARIABLES, JsonUtil.toJson(dataBindingReport));
                logger.endTest(testCase.getTestCaseId(), attributes);
            } else {
                logger.endCalledTest(testCase.getTestCaseId(), null);
            }

            postExecution();

            if (shouldRetry && testSuiteExecutor != null && !testSuiteExecutor.reachFailedThreshold()) {
                testCaseContext.setRetryIndex(retrySettings.getCurrentRetryCount() + 1);
                /// Reset Test Case Binded Values when retry immediately
                testCaseBinding.setBindedValues(initialTestCaseBindedValues);
                TestCaseExecutor executor = new TestCaseExecutor(testCaseBinding, engine, eventManager, testCaseContext,
                        doCleanUp);
                RetrySettingsImpl newRetry = new RetrySettingsImpl();
                newRetry.setCurrentRetryCount(retrySettings.getCurrentRetryCount() + 1);
                newRetry.setRemainingRetryCount(retrySettings.getRemainingRetryCount() - 1);
                executor.setRetrySettings(newRetry);
                executor.setTestSuiteExecutor(testSuiteExecutor);
                executor.execute(flowControl);
            }
        }
    }

    private TestResult invokeTestSuiteMethod(String methodName, String actionType, boolean ignoredIfFailed,
            TestResult testCaseResult) {
        if (testSuiteExecutor != null) {
            ErrorCollector errorCollector = ErrorCollector.getCollector();
            List<Throwable> coppiedError = errorCollector.getCoppiedErrors();
            errorCollector.clearErrors();

            testSuiteExecutor.invokeEachTestCaseMethod(methodName, actionType, ignoredIfFailed);

            if (!ignoredIfFailed && errorCollector.containsErrors()) {
                coppiedError.add(errorCollector.getFirstError());
            }

            errorCollector.clearErrors();
            errorCollector.getErrors().addAll(coppiedError);

            if (errorCollector.containsErrors() && ignoredIfFailed) {
                Throwable firstError = errorCollector.getFirstError();
                TestStatus testStatus = new TestStatus();
                TestStatusValue errorType = ErrorCollector.isErrorFailed(firstError) ? TestStatusValue.FAILED
                        : TestStatusValue.ERROR;
                testStatus.setStatusValue(errorType);
                String errorMessage = ExceptionsUtil.getMessageForThrowable(firstError);
                testStatus.setStackTrace(errorMessage);
                testCaseResult.setTestStatus(testStatus);

                return testCaseResult;
            }
        }
        return testCaseResult;
    }

    private void accessMainPhase() {
        if (!processSetupPhase()) {
            return;
        }

        processExecutionPhase();
    }

    private void processExecutionPhase() {
        try {
            // Prepare configuration before execution
            engine.changeConfigForExecutingScript();
            setupContextClassLoader();
            doExecute();
        } catch (ExceptionInInitializerError e) {
            // errors happened in static initializer like for Global Variable
            errorCollector.addError(e.getCause());
        } catch (Throwable e) {
            // logError(e, ExceptionsUtil.getMessageForThrowable(e));
            errorCollector.addError(e);
        }

        if (errorCollector.containsErrors()) {
            onExecutionError(errorCollector.getFirstError());
        } else {
            onExecutionComplete();
        }

        if (doCleanUp) {
            cleanUp();
        }
    }

    protected void doExecute() throws ResourceException, ScriptException, IOException, ClassNotFoundException {
        testCaseResult.setScriptResult(runScript(getScriptFile()));
    }

    private void cleanUp() {
        DriverCleanerCollector.getInstance().cleanDrivers();
    }

    private Object runScript(File scriptFile)
            throws ResourceException, ScriptException, IOException, ClassNotFoundException {
        return engine.runScriptAsRawText(FileUtils.readFileToString(scriptFile, DF_CHARSET),
                scriptFile.toURI().toURL().toExternalForm(), variableBinding, getTestCase().getName());
    }

    protected void runMethod(File scriptFile, String methodName)
            throws ResourceException, ScriptException, ClassNotFoundException, IOException {
        engine.changeConfigForExecutingScript();
        engine.runScriptMethodAsRawText(FileUtils.readFileToString(scriptFile, DF_CHARSET),
                scriptFile.toURI().toURL().toExternalForm(), methodName, variableBinding);
    }

    private Map<String, String> getTestCaseProperties(TestCaseBinding testCaseBinding, TestCase testCase,
            FailureHandling flowControl) {
        Map<String, String> testProperties = new HashMap<String, String>();
        if (retrySettings != null) {
            testProperties.put(CoreConstants.TEST_CASE_CURRENT_RETRY_COUNT,
                    Integer.toString(retrySettings.getCurrentRetryCount()));
            testProperties.put(CoreConstants.TEST_CASE_REMAINING_RETRY_COUNT,
                    Integer.toString(retrySettings.getRemainingRetryCount()));
        }

        testProperties.put(StringConstants.XML_LOG_NAME_PROPERTY, testCaseBinding.getTestCaseId());
        testProperties.put(StringConstants.XML_LOG_DESCRIPTION_PROPERTY, testCase.getDescription());
        testProperties.put(StringConstants.XML_LOG_ID_PROPERTY, testCase.getTestCaseId());
        testProperties.put(StringConstants.XML_LOG_SOURCE_PROPERTY, testCase.getMetaFilePath());
        testProperties.put(StringConstants.XML_LOG_TAG_PROPERTY, testCase.getTag());
        testProperties.put(StringConstants.XML_LOG_IS_OPTIONAL,
                String.valueOf(flowControl == FailureHandling.OPTIONAL));
        testProperties.put(StringConstants.XML_LOG_ITERATION_PROPERTY, testCase.getIterationVariableValue());
        testProperties.put(StringConstants.XML_LOG_ORIGIN_PROPERTY, testCase.getOrigin());
        return testProperties;
    }

    /**
     * Returns DEFAULT test case variables and their values.
     */
    private Map<String, Object> getBindedValues() {
        Map<String, Object> bindedValues = testCaseBinding.getBindedValues();
        return bindedValues != null ? bindedValues : Collections.emptyMap();
    }

    private Binding collectTestCaseVariables() {
        Binding variableBinding = new Binding(
                testCaseBinding != null ? testCaseBinding.getBindedValues() : Collections.emptyMap());
        engine.changeConfigForCollectingVariable();

        logger.logDebug(StringConstants.MAIN_LOG_INFO_START_EVALUATE_VARIABLE);
        testCase.getVariables().stream().forEach(testCaseVariable -> {
            String variableName = testCaseVariable.getName();
            if (getBindedValues().containsKey(variableName)) {
                Object variableValue = testCaseBinding.getBindedValues().get(variableName);
                variableBinding.setVariable(variableName, variableValue);

                if (testCaseVariable.isMasked()) {
                    addSecureValue(variableValue);
                } else if (secureProtectedGlobalValuesPattern != null) {
                    Matcher matcher = secureProtectedGlobalValuesPattern.matcher(Objects.toString(variableValue));
                    if (matcher.matches()) {
                        addSecureValue(variableValue);
                    }
                }

                dataBindingReport.add(VariableReport.value(maskValue(Objects.toString(variableValue)), variableName,
                        testCaseVariable.isMasked()));

                logger.logInfo(MessageFormat.format(StringConstants.MAIN_LOG_INFO_VARIABLE_NAME_X_IS_SET_TO_Y,
                        variableName, Objects.toString(variableValue)));
                return;
            }

            try {
                String defaultValue = StringUtils.defaultIfEmpty(testCaseVariable.getDefaultValue(),
                        StringConstants.NULL_AS_STRING);
                Object defaultValueObject = engine.runScriptWithoutLogging(defaultValue, null);

                variableBinding.setVariable(variableName, defaultValueObject);

                if (testCaseVariable.isMasked()) {
                    addSecureValue(defaultValueObject);
                }

                dataBindingReport.add(VariableReport.defaultValue(maskValue(Objects.toString(defaultValueObject)),
                        variableName, testCaseVariable.isMasked()));

                logger.logInfo(
                        MessageFormat.format(StringConstants.MAIN_LOG_INFO_VARIABLE_NAME_X_IS_SET_TO_Y_AS_DEFAULT,
                                variableName, Objects.toString(defaultValueObject)));
            } catch (ExceptionInInitializerError e) {
                logger.logWarning(MessageFormat.format(StringConstants.MAIN_LOG_MSG_SET_TEST_VARIABLE_ERROR_BECAUSE_OF,
                        variableName, e.getCause().getMessage()), null, e);
            } catch (Exception e) {
                logger.logWarning(MessageFormat.format(StringConstants.MAIN_LOG_MSG_SET_TEST_VARIABLE_ERROR_BECAUSE_OF,
                        variableName, e.getMessage()), null, e);
            }
        });
        getBindedValues().entrySet()
                .stream()
                .filter(entry -> !variableBinding.hasVariable(entry.getKey()))
                .forEach(entry -> {
                    String variableName = entry.getKey();
                    Object variableValue = entry.getValue();
                    variableBinding.setProperty(variableName, variableValue);
                    logger.logInfo(MessageFormat.format(StringConstants.MAIN_LOG_INFO_VARIABLE_NAME_X_IS_SET_TO_Y,
                            variableName, Objects.toString(variableValue)));
                });
        return variableBinding;
    }

    private void logError(Throwable t, String message) {
        logger.logMessage(ErrorCollector.fromError(t), message, t);
    }

    private TestStatusValue getResultByError(Throwable t) {
        return TestStatusValue.valueOf(ErrorCollector.fromError(t).name());
    }

    private void endAllUnfinishedKeywords(Stack<KeywordStackElement> keywordStack) {
        while (!keywordStack.isEmpty()) {
            KeywordStackElement keywordStackElement = keywordStack.pop();
            logger.endKeyword(keywordStackElement.getKeywordName(), null, keywordStackElement.getNestedLevel());
        }
    }

    private void internallyRunMethods(TestCaseMethodNodeWrapper methodNodeWrapper) {
        List<MethodNode> methodList = methodNodeWrapper.getMethodNodes();
        if (methodList == null || methodList.isEmpty()) {
            return;
        }

        logger.logDebug(methodNodeWrapper.getStartMessage());
        int count = 1;
        for (MethodNode method : methodList) {
            runMethod(method.getName(), methodNodeWrapper.getActionType(), count++,
                    methodNodeWrapper.isIgnoredIfFailed());
        }
    }

    private void runMethod(String methodName, String actionType, int index, boolean ignoreIfFailed) {
        Stack<KeywordStackElement> keywordStack = new Stack<KeywordStackElement>();
        Map<String, String> startKeywordAttributeMap = new HashMap<String, String>();
        startKeywordAttributeMap.put(StringConstants.XML_LOG_STEP_INDEX, String.valueOf(index));
        if (ignoreIfFailed) {
            startKeywordAttributeMap.put(StringConstants.XML_LOG_IS_IGNORED_IF_FAILED, String.valueOf(ignoreIfFailed));
        }
        boolean isKeyword = true;
        logger.startKeyword(methodName, actionType, startKeywordAttributeMap, keywordStack);
        try {
            runMethod(getScriptFile(), methodName);
            endAllUnfinishedKeywords(keywordStack);
            logger.logPassed(MessageFormat.format(StringConstants.MAIN_LOG_PASSED_METHOD_COMPLETED, methodName),
                    Collections.emptyMap(), isKeyword);
        } catch (Throwable e) {
            endAllUnfinishedKeywords(keywordStack);
            String message = MessageFormat.format(StringConstants.MAIN_LOG_WARNING_ERROR_OCCURRED_WHEN_RUN_METHOD,
                    methodName, e.getClass().getName(), ExceptionsUtil.getMessageForThrowable(e));
            if (ignoreIfFailed) {
                logger.logWarning(message, null, e, isKeyword);
                return;
            }
            logger.logError(message, null, e, isKeyword);
            errorCollector.addError(e);
        } finally {
            logger.endKeyword(methodName, actionType, Collections.emptyMap(), keywordStack);
        }
    }

    private void runTearDownMethodByError(Throwable t) {
        LogLevel errorLevel = ErrorCollector.fromError(t);
        TestCaseMethodNodeWrapper failedMethodWrapper = methodNodeCollector
                .getMethodNodeWrapper(TearDownIfFailed.class);
        if (errorLevel == LogLevel.ERROR) {
            failedMethodWrapper = methodNodeCollector.getMethodNodeWrapper(TearDownIfError.class);
        }

        internallyRunMethods(failedMethodWrapper);
        internallyRunMethods(methodNodeCollector.getMethodNodeWrapper(TearDown.class));
    }

    public void setupContextClassLoader() {
        // AccessController.doPrivileged(new DoSetContextAction(Thread.currentThread(), engine.getGroovyClassLoader()));
    }

    public TestCase getTestCase() {
        return testCase;
    }

    public RetrySettings getRetrySettings() {
        return retrySettings;
    }

    public void setRetrySettings(RetrySettings retrySettings) {
        this.retrySettings = retrySettings;
    }

    public TestResult getTestCaseResult() {
        return testCaseResult;
    }

    public void getIterationValueByIterationName() {
        testCase.getVariables().stream().forEach(testCaseVariable -> {
            String variableName = testCaseVariable.getName();
            String iterationVariableName = testCaseBinding.getIterationVariableName();
            if (iterationVariableName != null && iterationVariableName.equals(variableName)) {
                Object variableValue = testCaseBinding.getBindedValues().get(variableName);
                if (variableValue == null) {
                    try {
                        String defaultValue = StringUtils.defaultIfEmpty(testCaseVariable.getDefaultValue(),
                                StringConstants.NULL_AS_STRING);
                        variableValue = engine.runScriptWithoutLogging(defaultValue, null);
                    } catch (Exception e) {
                        logger.logWarning(
                                MessageFormat.format(StringConstants.MAIN_LOG_MSG_SET_TEST_VARIABLE_ERROR_BECAUSE_OF,
                                        variableName, e.getMessage()),
                                null, e);
                    }
                }

                if (variableValue != null) {
                    testCase.setIterationVariableValue(variableValue.toString());
                }

                if (testCaseVariable.isMasked()) {
                    addSecureValue(variableValue);
                }
            }
        });
    }

    private void addSecureValue(Object variableValue) {
        String value = Objects.toString(variableValue);
        if ("''".equals(value) || "".equals(value) || "0".equals(value)) {
            return;
        }

        secureValues.add(value);
        secureValuesPattern = generateSecureValuesPattern(secureValues);
        logger.setSecureValuesPattern(secureValuesPattern);
    }

    private Pattern generateSecureValuesPattern(List<String> secureValues) {
        if (secureValues.isEmpty()) {
            return null;
        }

        String regex = secureValues.stream()
                .sorted((a, b) -> Integer.compare(b.length(), a.length()))
                .map(Pattern::quote)
                .collect(Collectors.joining("|"));
        if (StringUtils.isEmpty(regex)) {
            return null;
        }

        return Pattern.compile(regex);
    }

    public String maskValue(String message) {
        if (message == null || secureValuesPattern == null) {
            return message;
        }

        Matcher matcher = secureValuesPattern.matcher(message);
        return matcher.replaceAll(NameUtil.BULLET_MASK);
    }

    class InterceptingOutputStream extends OutputStream {
        private final PrintStream originalStream;  // Keep the original System.out

        public InterceptingOutputStream(PrintStream originalStream) {
            this.originalStream = originalStream;
        }

        @Override
        public void write(int b) {
            char c = (char) b;
            processLog(String.valueOf(c));  // Process and redirect
        }

        @Override
        public void write(byte[] b, int off, int len) {
            String intercepted = new String(b, off, len);
            processLog(intercepted);  // Process and redirect
        }

        private void processLog(String message) {
            // Redirect to System.out without the original log being printed
            originalStream.print(maskValue(message));
        }

        public String maskValue(String message) {
            if (message == null || secureValuesPattern == null) {
                return message;
            }

            Matcher matcher = secureValuesPattern.matcher(message);
            return matcher.replaceAll(NameUtil.ASTERISK_MASK);
        }
    }

    /**
     * Checks if the test case script file exists and is not blank.
     * 
     * @return true if the script file exists and is not empty/blank, false otherwise.
     */
    private boolean hasTestCaseScript() {
        try {
            File scriptFile = new File(testCase.getGroovyScriptPath());
            return scriptFile.exists() && StringUtils.isNotBlank(FileUtils.readFileToString(scriptFile, DF_CHARSET));
        } catch (IOException e) {
            logger.logError(e.getMessage(), null, e);
            return false;
        }
    }
}
