package com.kms.katalon.core.profiling;

import java.util.Comparator;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import oshi.SystemInfo;
import oshi.software.os.OSProcess;
import oshi.software.os.OperatingSystem;

public class ResourceTrackerService {

    private final Logger logger = LoggerFactory.getLogger(ResourceTrackerService.class);
    private static final long INTERVAL_MINUTES = 15;
    private ScheduledExecutorService scheduler;
    private final SystemInfo systemInfo = new SystemInfo();
    private final OperatingSystem os = systemInfo.getOperatingSystem();
    private boolean enableProcessProfiling = false;

    public ResourceTrackerService() {
        String profilingSettings = System.getProperty("katalon.enableProfiling", "false");
        enableProcessProfiling = Boolean.valueOf(profilingSettings);
    }

    public void start() {
        if (enableProcessProfiling) {
            logger.info("Starting ResourceTrackerService...");
            scheduler = Executors.newSingleThreadScheduledExecutor();
            scheduler.scheduleAtFixedRate(this::trackResources, INTERVAL_MINUTES, INTERVAL_MINUTES, TimeUnit.MINUTES);
            logger.info("ResourceTrackerService started. Resource Snapshot will display every {} minute(s)...", INTERVAL_MINUTES);
        }
    }

    private void trackResources() {
        try {
            long currentPid = ProcessHandle.current().pid();
            OSProcess currentProcess = os.getProcess((int) currentPid);

            if (currentProcess == null) {
                logger.warn("Current process not found!");
                return;
            }

            logger.info("\n=== Resource Snapshot ===");

            // Depth 0 → current process
            logProcess(currentProcess, "", true);

            List<OSProcess> children = os.getChildProcesses((int) currentPid,
                    p -> true,
                    (o1, o2) -> 0,
                    100);
            for (int i = 0; i < children.size(); i++) {
                OSProcess child = children.get(i);
                boolean lastChild = (i == children.size() - 1);
                String prefix = lastChild ? "└── " : "├── ";
                logProcess(child, prefix, lastChild);
            }

        } catch (Exception e) {
            logger.warn("Error tracking resources: " + e.getMessage());
        }
    }

    private void logProcess(OSProcess process, String prefix, boolean last) {
        long pid = process.getProcessID();
        double cpuPercent = process.getProcessCpuLoadCumulative() * 100;
        double memoryMb = process.getResidentSetSize() / 1024.0 / 1024.0;
        String name = process.getName();

        System.out.printf("%sPID %-6d | CPU: %6.2f%% | RAM: %8.2f MB | Name: %s%n",
                prefix, pid, cpuPercent, memoryMb, name);
    }

    public void stop() {
        if (scheduler != null) {
            scheduler.shutdownNow();
            logger.info("ResourceTrackerService stopped.");
            scheduler = null;
        }
    }
}