package com.kms.katalon.core.util;

import java.io.File;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.commons.lang.StringUtils;

import com.google.common.io.BaseEncoding;
import com.kms.katalon.core.configuration.RunConfiguration;

public class FileUtil {
    private static final String ANY_EXTENSION = "*";
    
    public static String getFilePathNoExtension(File file) {
        if (file == null) return "";
        // Get file path
        String filePath = file.getAbsolutePath();
        	
        // Remove the extension
        int dotIndex = filePath.lastIndexOf('.');
        if (dotIndex > 0) { // Ensure there's an extension
            filePath = filePath.substring(0, dotIndex);
        }

        return filePath;
    }
    public static File[] getFiles(File folder, String suffix, Date origin) throws IOException {
        final List<File> filePaths = new ArrayList<>();
        enumFiles(folder, suffix, origin, new FileProcessor() {

            @Override
            public void process(File f) {
                filePaths.add(f);
            }
        });
        return filePaths.toArray(new File[] {});
    }

    public static void enumFiles(File folder, String suffix, Date origin, FileProcessor fileProcessor) throws IOException {
        if (folder.isDirectory() == false) {
            return;
        }
        List<File> files = new ArrayList<>();
        files.addAll(Arrays.asList(folder.listFiles()));
        for (int i = 0; i < files.size(); ++i) {
            File file = files.get(i);
            if (file.isDirectory()) {
                files.addAll(Arrays.asList(file.listFiles()));
                continue;
            }
            if (isFileSatisfy(file, suffix, origin)) {
                fileProcessor.process(file);
            }
        }
    }

    private static boolean isFileSatisfy(File file, String suffix, Date origin) throws IOException {
        String fileName = file.getName();
        boolean satisfyName = suffix.equals(ANY_EXTENSION) ? true : fileName.endsWith(suffix);
        if (!satisfyName) {
            return false;
        }

        return isFileCreateAfter(file, origin);
    }
    
    public static boolean isFileCreateAfter(File file, Date time) throws IOException {
        if (time == null) {
            return true;
        } else {
            return time.getTime() < Files.readAttributes(file.toPath(), BasicFileAttributes.class)
                    .creationTime()
                    .toMillis();
        }
    }

    public static int countAllFiles(File folder, String suffix, Date origin) throws IOException {
        final int[] count = { 0 };
        enumFiles(folder, suffix, origin, new FileProcessor() {

            @Override
            public void process(File f) {
                count[0]++;
            }
        });

        return count[0];
    }

    public static boolean isValidPath(String path) {
        return path != null && new File(path).exists();
    }

    public static boolean fileExists(String root, String pattern) {
        return findFileByName(root, pattern) != null;
    }

    public static File findFileByName(String root, String pattern) {
        return findFile(root, file -> {
            return file.getName().matches(pattern);
        });
    }

    public static File findFile(String root, FileFilter filter) {
        return find(root, filter, FileType.FILE);
    }

    public static List<File> findAllFiles(String root, FileFilter filter) {
        return findAll(root, filter, FileType.FILE);
    }

    public static File findFolderByName(String root, String pattern) {
        return findFolder(root, file -> {
            return file.getName().matches(pattern);
        });
    }

    public static File findFolder(String root, FileFilter filter) {
        return find(root, filter, FileType.FOLDER);
    }

    public static List<File> findAllFolders(String root, FileFilter filter) {
        return findAll(root, filter, FileType.FOLDER);
    }

    public static File find(String root, FileFilter filter) {
        return find(root, filter, FileType.ANY);
    }

    public static File find(String root, FileFilter filter, FileType type) {
        List<File> matches = findAll(root, filter, type);
        return matches.isEmpty() ? null : matches.get(0);
    }

    public static List<File> findAll(String root, FileFilter filter) {
        return findAll(root, filter, FileType.ANY);
    }

    public static List<File> findAll(String root, FileFilter filter, FileType type) {
        if (StringUtils.isBlank(root) || !new File(root).exists()) {
            return Collections.emptyList();
        }
        File rootFile = new File(root);
        String[] directories = rootFile.list(new FilenameFilter() {
            @Override
            public boolean accept(File parentFolder, String name) {
                File file = new File(parentFolder, name);
                boolean typeCheck = type == FileType.ANY || (type == FileType.FILE && !file.isDirectory())
                        || (type == FileType.FOLDER && file.isDirectory());
                return filter.test(file) && typeCheck;
            }
        });
        return Arrays.asList(directories).stream().map(subDirectory -> new File(root, subDirectory)).collect(Collectors.toList());
    }

    public static interface FileProcessor {
        public void process(File f);
    }

    public static interface FileFilter {
        public boolean test(File file);
    }

    public enum FileType {
        FILE, FOLDER, ANY
    }
    
    public static File getFile(String path) {
        File file = new File(path);
        if (file.isAbsolute()) {
            return file;
        }
        if (StringUtils.isNotBlank(RunConfiguration.getProjectDir())) {
            return new File(RunConfiguration.getProjectDir(), path);
        }
        return null;
    }
    
    public static Set<File> listFiles(String dir) throws IOException {
        if (StringUtils.isEmpty(dir)) {
            return Collections.emptySet();
        }
        if (!new File(dir).exists()) {
            return Collections.emptySet();
        }
        try (Stream<Path> stream = Files.list(Paths.get(dir))) {
            return stream.map(Path::toFile).collect(Collectors.toSet());
        }
    }

    public static String readFileToImageDataURL(File file) throws IOException {
        if (!file.exists()) {
            return null;
        }
        return "data:image/png;base64," + readFileAsBase64(file);
    }

    public static String readFileAsBase64(File file) throws IOException {
        return BaseEncoding.base64().encode(Files.readAllBytes(file.toPath()));
    }

    public static void saveBase64ToFile(String base64Data, File outputFile) throws IOException {
        if (base64Data == null || base64Data.isEmpty()) {
            throw new IllegalArgumentException("Base64 data cannot be null or empty");
        }
        if (outputFile == null) {
            throw new IllegalArgumentException("Output file cannot be null");
        }

        // Remove data URL prefix if present (e.g., "data:image/png;base64,")
        String base64String = base64Data;
        if (base64Data.contains(",")) {
            base64String = base64Data.substring(base64Data.indexOf(",") + 1);
        }

        // Decode Base64 and write to file
        byte[] decodedBytes = BaseEncoding.base64().decode(base64String);
        Files.write(outputFile.toPath(), decodedBytes);
    }
}
