package com.kms.katalon.core.webui.util;

import java.io.File;
import java.nio.file.Paths;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;
import org.openqa.selenium.Capabilities;
import org.openqa.selenium.MutableCapabilities;
import org.openqa.selenium.Platform;
import org.openqa.selenium.chrome.ChromeOptions;
import org.openqa.selenium.edge.EdgeOptions;
import org.openqa.selenium.firefox.FirefoxOptions;
import org.openqa.selenium.firefox.FirefoxProfile;

import com.kms.katalon.core.logging.KeywordLogger;
import com.kms.katalon.core.util.ConsoleCommandExecutor;
import com.kms.katalon.core.util.TestCloudPropertyUtil;
import com.kms.katalon.core.webui.constants.StringConstants;
import com.kms.katalon.core.webui.driver.WebUIDriverType;

public class WebDriverPropertyUtil {

    private static final KeywordLogger logger = KeywordLogger.getInstance(WebDriverPropertyUtil.class);

    public static final String CHROME_SWITCHES = "chrome.switches";

    public static final String CHROME_NO_SANDBOX = "--no-sandbox";

    private static final String CHROME_ARGUMENT_PROPERTY_KEY = "args";

    public static final String CHROME_BINARY_PROPERTY_KEY = "binary";

    private static final String CHROME_EXTENSIONS_PROPERTY_KEY = "extensions";

    private static final String CHROME_PREFERENCES_PROPERTY_KEY = "prefs";

    private static final String CHROME_LOCALSTATE_PROPERTY_KEY = "localState";

    private static final String CHROME_DETACH_PROPERTY_KEY = "detach";

    private static final String CHROME_DEBUGGER_ADDRESS_PROPERTY_KEY = "debuggerAddress";

    private static final String CHROME_EXCLUDE_SWITCHES_PROPERTY_KEY = "excludeSwitches";

    private static final String CHROME_MINI_DUMP_PATH_PROPERTY_KEY = "minidumpPath";

    private static final String CHROME_MOBILE_EMULATION_PROPERTY_KEY = "mobileEmulation";

    private static final String CHROME_PREF_LOGGING_PREFS_PROPERTY_KEY = "perfLoggingPrefs";

    private static final String[] CHROME_CAPABILITIES = { CHROME_ARGUMENT_PROPERTY_KEY, CHROME_BINARY_PROPERTY_KEY,
            CHROME_EXTENSIONS_PROPERTY_KEY, CHROME_PREFERENCES_PROPERTY_KEY, CHROME_LOCALSTATE_PROPERTY_KEY,
            CHROME_DETACH_PROPERTY_KEY, CHROME_DEBUGGER_ADDRESS_PROPERTY_KEY, CHROME_EXCLUDE_SWITCHES_PROPERTY_KEY,
            CHROME_MINI_DUMP_PATH_PROPERTY_KEY, CHROME_MOBILE_EMULATION_PROPERTY_KEY,
            CHROME_PREF_LOGGING_PREFS_PROPERTY_KEY };

    private static final String STARTUP_HOMEPAGE_WELCOME_URL_ADDITIONAL_PREFERENCE = "startup.homepage_welcome_url.additional";

    private static final String STARTUP_HOMEPAGE_WELCOME_URL_PREFERENCE = "startup.homepage_welcome_url";

    public static final String BROWSER_STARTUP_HOMEPAGE_PREFERENCE = "browser.startup.homepage";

    private static final String FIREFOX_BLANK_PAGE = "about:blank";

    private static final String FIREFOX_PROFILE_KEY = "profile";

    private static final String EDGE_BINARY_PROPERTY_KEY = "binary";

    public static final String KATALON_DOCKER_ENV_KEY = "KATALON_DOCKER";

    private static final String CHROME_PROPERTY_USER_DATA_DIR = "user-data-dir=";

    private static final String CHROME_PROPERTY_PROFILE_DIR = "profile-directory=";

    private static final String CHROME_KATALON_COMPACT_UTILITY_EXTENSION_RELATIVE_PATH = "Extensions" + File.separator
            + "gkihajmjffefinkmpokfepcdbhnpflee";

    private static final String EDGE_CHROMIUM_KATALON_COMPACT_UTILITY_EXTENSION_RELATIVE_PATH = "Extensions"
            + File.separator + "kkmafoknllgdaapbegpkpmbncidodkaa";

    public static MutableCapabilities toDesireCapabilities(Map<String, Object> propertyMap,
            WebUIDriverType webUIDriverType) {
        if (propertyMap == null) {
            return null;
        }
        switch (webUIDriverType) {
            case CHROME_DRIVER:
            case HEADLESS_DRIVER:
                return getDesireCapabilitiesForChrome(propertyMap);
            case FIREFOX_DRIVER:
            case FIREFOX_HEADLESS_DRIVER:
                return getDesireCapabilitiesForFirefox(propertyMap);
            case EDGE_CHROMIUM_DRIVER:
                return getDesiredCapabilitiesForEdgeChromium(propertyMap, true);
            default:
                return toDesireCapabilities(propertyMap);
        }
    }

    public static MutableCapabilities toDesireCapabilities(Map<String, Object> propertyMap) {
        return toDesireCapabilities(propertyMap, new MutableCapabilities(), true);
    }

    public static MutableCapabilities toDesireCapabilities(Map<String, Object> propertyMap,
            MutableCapabilities desireCapabilities, boolean isLog) {
        for (Entry<String, Object> property : propertyMap.entrySet()) {
            desireCapabilities.setCapability(property.getKey(), property.getValue());
            if (isLog) {
                if (!isRunFromTestCloudWithSauceLabs(property)) {
                    logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_WEB_UI_PROPERTY_SETTING,
                            property.getKey(), property.getValue()));
                }
            }
        }
        return desireCapabilities;
    }

    @SuppressWarnings("unchecked")
    public static MutableCapabilities getDesiredCapabilitiesForEdgeChromium(Map<String, Object> propertyMap,
            boolean isLog) {
        EdgeOptions options = new EdgeOptions();
        MutableCapabilities mutableCapabilities = new MutableCapabilities();

        for (Entry<String, Object> property : propertyMap.entrySet()) {
            if (isLog) {
                logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_WEB_UI_PROPERTY_SETTING, property.getKey(),
                        property.getValue()));
            }
            mutableCapabilities.setCapability(property.getKey(), property.getValue());
        }

        options = options.merge(mutableCapabilities);

        if (OSUtil.isMac()) {
            options.setPlatformName(Platform.MAC.toString());
        } else if (OSUtil.isWindows()) {
            options.setPlatformName(Platform.WINDOWS.toString());
        } else {
            options.setPlatformName(Platform.LINUX.toString());
        }

        if (options.getPlatformName() == Platform.LINUX) {
            Map<String, Object> msEdgeOptions = (Map<String, Object>) options.getCapability(EdgeOptions.CAPABILITY);
            if (msEdgeOptions == null) {
                msEdgeOptions = new HashMap<>();
            }

            if (!msEdgeOptions.containsKey(EDGE_BINARY_PROPERTY_KEY)) {
                String msEdgeBinary = ConsoleCommandExecutor.safeWhere("microsoft-edge");
                if (StringUtils.isEmpty(msEdgeBinary)) {
                    msEdgeBinary = "/usr/bin/microsoft-edge";
                }

                File msEdgeBinaryFilePath = new File(msEdgeBinary);
                if (msEdgeBinaryFilePath.exists()) {
                    options.setBinary(msEdgeBinaryFilePath);
                }
            }
        }

        return options;
    }

    public static MutableCapabilities getDesireCapabilitiesForFirefox(Map<String, Object> propertyMap) {
        MutableCapabilities mutableCapabilities = new MutableCapabilities();
        FirefoxProfile firefoxProfile = createDefaultFirefoxProfile();
        for (Entry<String, Object> property : propertyMap.entrySet()) {
            if (property.getKey().equals(FIREFOX_PROFILE_KEY) && property.getValue() instanceof Map<?, ?>) {
                processFirefoxPreferencesSetting(firefoxProfile, (Map<?, ?>) property.getValue());
            } else {
                mutableCapabilities.setCapability(property.getKey(), property.getValue());
                logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_WEB_UI_PROPERTY_SETTING, property.getKey(),
                        property.getValue()));
            }
        }

        FirefoxOptions options = new FirefoxOptions(mutableCapabilities);
        options = options.merge(mutableCapabilities);
        options.setProfile(firefoxProfile);
        return options;
    }

    private static void processFirefoxPreferencesSetting(FirefoxProfile firefoxProfile, Map<?, ?> firefoxPropertyMap) {
        for (Entry<?, ?> entry : firefoxPropertyMap.entrySet()) {
            if (!(entry.getKey() instanceof String)) {
                continue;
            }
            String entryKey = (String) entry.getKey();
            if (setFirefoxPreferenceValue(firefoxProfile, entryKey, entry.getValue())) {
                logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_FIREFOX_PROPERTY_SETTING, entryKey,
                        entry.getValue()));
            }
        }
    }

    private static boolean setFirefoxPreferenceValue(FirefoxProfile firefoxProfile, String entryKey,
            Object entryValue) {
        if (entryValue instanceof Number) {
            firefoxProfile.setPreference(entryKey, ((Number) entryValue).intValue());
            return true;
        }
        if (entryValue instanceof Boolean) {
            firefoxProfile.setPreference(entryKey, (Boolean) entryValue);
            return true;
        }
        if (entryValue instanceof String) {
            firefoxProfile.setPreference(entryKey, (String) entryValue);
            return true;
        }
        return false;
    }

    public static FirefoxProfile createDefaultFirefoxProfile() {
        FirefoxProfile firefoxProfile = new FirefoxProfile();
        firefoxProfile.setPreference(BROWSER_STARTUP_HOMEPAGE_PREFERENCE, FIREFOX_BLANK_PAGE);
        firefoxProfile.setPreference(STARTUP_HOMEPAGE_WELCOME_URL_PREFERENCE, FIREFOX_BLANK_PAGE);
        firefoxProfile.setPreference(STARTUP_HOMEPAGE_WELCOME_URL_ADDITIONAL_PREFERENCE, FIREFOX_BLANK_PAGE);
        return firefoxProfile;
    }

    public static MutableCapabilities getDesireCapabilitiesForChrome(Map<String, Object> propertyMap) {
        MutableCapabilities mutableCapabilities = new MutableCapabilities();
        Map<String, Object> chromeOptions = new HashMap<String, Object>();
        for (Entry<String, Object> driverProperty : propertyMap.entrySet()) {
            if (Arrays.asList(CHROME_CAPABILITIES).contains(driverProperty.getKey())) {
                chromeOptions.put(driverProperty.getKey(), driverProperty.getValue());
            } else {
                mutableCapabilities.setCapability(driverProperty.getKey(), driverProperty.getValue());
            }
            logger.logInfo(MessageFormat.format(StringConstants.KW_LOG_WEB_UI_PROPERTY_SETTING, driverProperty.getKey(),
                    driverProperty.getValue()));
        }

        injectAddtionalArgumentsForChrome(chromeOptions);

        mutableCapabilities.setCapability(ChromeOptions.CAPABILITY, chromeOptions);

        return generateDefaultChromeOptions().merge(mutableCapabilities);
    }

    private static void injectAddtionalArgumentsForChrome(Map<String, Object> chromeOptions) {
        if (chromeOptions == null) {
            return;
        }
        List<Object> argumentsList = new ArrayList<Object>();
        if (chromeOptions.get(CHROME_ARGUMENT_PROPERTY_KEY) instanceof List) {
            argumentsList.addAll((List<?>) chromeOptions.get(CHROME_ARGUMENT_PROPERTY_KEY));
        }
        // https://github.com/kms-technology/katalon/issues/2815
        // argumentsList.add(CHROME_SWITCHES);
        // argumentsList.add(DISABLE_EXTENSIONS);
        if (isRunningInDocker()) {
            argumentsList.add(CHROME_NO_SANDBOX);
        }
        chromeOptions.put(CHROME_ARGUMENT_PROPERTY_KEY, argumentsList);
    }

    public static boolean isRunningInDocker() {
        if (System.getenv().containsKey(KATALON_DOCKER_ENV_KEY)) {
            return Boolean.valueOf(System.getenv(KATALON_DOCKER_ENV_KEY));
        }
        return false;
    }

    private static boolean isRunFromTestCloudWithSauceLabs(Entry<String, Object> property) {
        if (TestCloudPropertyUtil.getInstance().isRunFromTestCloud()) {
            if (property.getKey().equals("accessKey") || property.getKey().equals("username")) {
                return true;
            }
        }
        return false;
    }

    public static boolean didUserProfileInstallKatalonCompactUtility(WebUIDriverType driverType,
            Capabilities capabilities) {
        String capabilityKey = "";
        switch (driverType) {
            case CHROME_DRIVER:
                capabilityKey = ChromeOptions.CAPABILITY;
                break;
            case EDGE_CHROMIUM_DRIVER:
                capabilityKey = EdgeOptions.CAPABILITY;
                break;
            default:
                return false;
        }

        @SuppressWarnings("unchecked")
        Map<String, Object> options = (Map<String, Object>) capabilities.getCapability(capabilityKey);
        if (options == null) {
            return false;
        }

        @SuppressWarnings("unchecked")
        List<String> argsEntry = (List<String>) options.get(CHROME_ARGUMENT_PROPERTY_KEY);
        if (argsEntry == null) {
            return false;
        }

        Optional<String> userDataDirOptional = argsEntry.stream()
                .filter(arg -> arg.contains(CHROME_PROPERTY_USER_DATA_DIR))
                .map(arg -> arg.replace(CHROME_PROPERTY_USER_DATA_DIR, ""))
                .findAny();
        if (!userDataDirOptional.isPresent() || userDataDirOptional.get().isEmpty()) {
            return false;
        }

        Optional<String> profileDirOptional = argsEntry.stream()
                .filter(arg -> arg.contains(CHROME_PROPERTY_PROFILE_DIR))
                .map(arg -> arg.replace(CHROME_PROPERTY_PROFILE_DIR, ""))
                .findAny();
        if (!profileDirOptional.isPresent() || profileDirOptional.get().isEmpty()) {
            return false;
        }

        String userDataDir = userDataDirOptional.get().replace(CHROME_PROPERTY_USER_DATA_DIR, "");
        String profileDir = Paths.get(userDataDir, profileDirOptional.get().replace(CHROME_PROPERTY_PROFILE_DIR, ""))
                .toString();

        switch (driverType) {
            case CHROME_DRIVER:
                File extensionChromeKCUFolder = new File(profileDir,
                        CHROME_KATALON_COMPACT_UTILITY_EXTENSION_RELATIVE_PATH);
                return extensionChromeKCUFolder.exists();
            case EDGE_CHROMIUM_DRIVER:
                File extensionEdgeKCUFolder = new File(profileDir,
                        EDGE_CHROMIUM_KATALON_COMPACT_UTILITY_EXTENSION_RELATIVE_PATH);
                return extensionEdgeKCUFolder.exists();
            default:
                return false;
        }
    }
    
    public static ChromeOptions generateDefaultChromeOptions() {
        return new ChromeOptions()
                // add this argument to keep the load extension CLI from GGChrome 137 temporarily
                .addArguments("disable-features=DisableLoadExtensionCommandLineSwitch");
    }
}
