package com.kms.katalon.core.windows.driver;

import java.io.File;
import java.io.IOException;
import java.net.Socket;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import org.eclipse.core.runtime.FileLocator;
import org.eclipse.core.runtime.Platform;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.kms.katalon.logging.LogUtil;
import com.kms.katalon.network.apache.services.ApacheHttpClient;
import com.kms.katalon.network.core.model.HttpOptions;
import com.kms.katalon.network.core.model.config.ProxyConfig;
import com.kms.katalon.network.core.model.exception.HttpException;

import net.lingala.zip4j.ZipFile;

public class FlaUIDriverManager {
    public enum StartMode {
        ONLY_START_IF_NOT_HEALTHY, FORCE_RESTART
    }

    private static String LOCALHOST = "127.0.0.1";

    private static int PREFERRED_PORT = 4723;

    private static int SESSION_CLEANUP_INTERVAL_IN_SECONDS = 300; // 5 minutes

    private static String LOG_FOLDER = "logs";

    private static final long LOG_RETENTION_DAYS = 3;

    public static class FlaUIServer {
        private Process process; // non-null if spawned by KS

        private String host;

        private int port;

        public FlaUIServer(String host, int port, Process process) {
            this.host = host;
            this.port = port;
            this.process = process;
        }

        public String getServerUrl() {
            return "http://" + host + ":" + port;
        }

        public boolean isSpanwedByKs() {
            return process != null;
        }
    }

    private String driverPath;

    private FlaUIServer localServer;

    public static final FlaUIDriverManager singleton;
    static {
        singleton = new FlaUIDriverManager();

        try {
            singleton.setDriverPath(resolveDriverPath());
        } catch (Exception e) {
            LogUtil.logError(e, "Failed to resolve FlaUI.WebDriver's path");
        }
    }

    private static String resolveDriverPath() throws IOException {
        var relativeDirPath = Path.of("resources", "extensions", "FlaUIWebDriver");
        var relativeExeFilePath = relativeDirPath.resolve("FlaUI.WebDriver.exe");
        var relativeZipFilePath = relativeDirPath.resolve("FlaUI.WebDriver.zip");

        String fullExeFilePath = resolveFilePath(relativeExeFilePath.toString());
        if (Files.exists(Paths.get(fullExeFilePath))) {
            return fullExeFilePath;
        }

        String fullZipFilePath = resolveFilePath(relativeZipFilePath.toString());
        LogUtil.logInfo("FlaUI: extract zip file: " + fullZipFilePath);

        try (ZipFile zipFile = new ZipFile(fullZipFilePath)) {
            zipFile.extractAll(Paths.get(fullExeFilePath).getParent().toAbsolutePath().toString());
        }

        return fullExeFilePath;
    }

    private static String resolveFilePath(String relativePath) {
        try {
            String currentPath = FlaUIDriverManager.class.getProtectionDomain().getCodeSource().getLocation().getPath();
            if (currentPath.endsWith("jar")) {
                String configurationFolderPath = new File(
                        FileLocator.resolve(Platform.getConfigurationLocation().getURL()).getFile()).toString();
                return Paths.get(configurationFolderPath, relativePath).toAbsolutePath().toString();
            }

            File file = new File(currentPath + relativePath);
            return file.getAbsolutePath();
        } catch (Exception e) {
            LogUtil.logError(e, "Failed to resolve relativePath: " + relativePath);
            return null;
        }
    }

    public static FlaUIDriverManager getInstance() {
        return singleton;
    }

    private FlaUIDriverManager() {
    }

    public void setDriverPath(String driverPath) {
        LogUtil.logInfo("FlaUI.WebDriver's path: " + driverPath);
        this.driverPath = driverPath;
    }

    private void setLocalServer(FlaUIServer server) {
        this.localServer = server;
    }

    /**
     * Sometimes the process is running but it still fails to serve request at the target port
     * 
     * @return true if up and healthy and false otherwise
     * @throws URISyntaxException
     * @throws HttpException
     */
    public boolean checkIfServerReady(FlaUIServer server, ProxyConfig proxyConfig) {
        if (server == null) {
            return false;
        }

        // Started by KS
        if (server.process != null && !server.process.isAlive()) {
            return false;
        }

        // Check server status no matter if server was spawned by KS or not.
        // GET http://127.0.0.1:<port>/status
        try {
            var httpClient = new ApacheHttpClient();
            Map<String, String> headers = new HashMap<String, String>();

            HttpOptions httpOptions = new HttpOptions.Builder().headers(headers)
                    .proxy(proxyConfig)
                    .connectTimeout(10000)
                    .socketTimeout(10000)
                    .build();

            URI uri = new URI(server.getServerUrl() + "/status");
            var response = httpClient.get(uri, httpOptions);

            String rawResponse = response.getBody();
            JsonObject jsonResponse = JsonParser.parseString(rawResponse).getAsJsonObject();
            boolean isReady = jsonResponse.getAsJsonObject("value").get("ready").getAsBoolean();

            if (!isReady) {
                LogUtil.logInfo("FlaUI server is not ready yet. Response = " + rawResponse);
                return false;
            }

        } catch (Exception e) {
            LogUtil.logError(e, "Failed to check FlaUI server status");
            return false;
        }

        return true;
    }

    private boolean waitUntilServerReady(FlaUIServer server, ProxyConfig proxyConfig) {
        int maxRetries = 10;
        for (int i = 0; i < maxRetries; i++) {
            if (checkIfServerReady(server, proxyConfig)) {
                return true;
            }

            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                return false;
            }
        }

        return false;
    }

    public FlaUIServer startLocalServer(StartMode mode, ProxyConfig proxyConfig) {
        if (mode == StartMode.ONLY_START_IF_NOT_HEALTHY) {
            if (checkIfServerReady(this.localServer, proxyConfig)) {
                LogUtil.logInfo("FlaUI driver is running at " + this.localServer.getServerUrl());
                return this.localServer; // Already up and healthy. No need to do anything else.
            }

            stopLocalServer(); // Server can be unhealthy. Need to stop before starting a new one in the next step
        }

        if (mode == StartMode.FORCE_RESTART) {
            stopLocalServer();
        }

        int portFrom = PREFERRED_PORT;
        int portTo = PREFERRED_PORT + 5;

        for (int port = portFrom; port < portTo; port++) {
            if (isPortBusy(port)) {
                continue;
            }
            LogUtil.logInfo("Attempt to start FlaUI driver at port " + port);
            var server = attemptToStartServerWithPort(port, proxyConfig);
            if (server != null) {
                setLocalServer(server);
                LogUtil.logInfo("FlaUI driver is started at " + server.getServerUrl());
                return server;
            }
        }

        LogUtil.logInfo("Failed to start FlaUI driver"); // TODO: throw exception
        return null;
    }

    private boolean isPortBusy(int port) {
        try (Socket socket = new Socket(LOCALHOST, port)) {
            socket.close();
            return true;
        } catch (Exception e) {
            return false;
        }
    }

    private FlaUIServer attemptToStartServerWithPort(int port, ProxyConfig proxyConfig) {
        var host = LOCALHOST;

        ProcessBuilder processBuilder = new ProcessBuilder(driverPath, "--urls=http://" + host + ":" + port,
                "--environment=Production",
                "--SessionCleanup:SchedulingIntervalSeconds=" + SESSION_CLEANUP_INTERVAL_IN_SECONDS);

        processBuilder.redirectErrorStream(true);

        // Redirect flaui's output to a log file if possible. If failed, keep going.
        try {
            DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMdd_HHmmss_SSS");
            String timestamp = LocalDateTime.now().format(formatter);
            String logFileName = "flaui_" + timestamp + ".log";
            File logFile = new File(LOG_FOLDER, logFileName);

            logFile.getParentFile().mkdirs();

            processBuilder.redirectOutput(ProcessBuilder.Redirect.to(logFile));
            LogUtil.logInfo("FlaUI's log will be redirected to log file: " + logFile.getAbsolutePath());

        } catch (Exception e) {
            LogUtil.logError(e, "Failed to redirect FlaUI's log to file");
        }

        Process process;

        try {
            process = processBuilder.start();
            Runtime.getRuntime().addShutdownHook(new Thread(() -> {
                safelyDestroyProcess(process);
            }));

            FlaUIServer server = new FlaUIServer(host, port, process);
            boolean isReady = waitUntilServerReady(server, proxyConfig);

            if (!isReady) {
                // Server started but un-usable (ex: blocked/unavailable port) --> kill it and try with another port
                safelyDestroyProcess(process);
                return null;
            }

            return server;

        } catch (Exception e) {
            LogUtil.logError(e, "Failed to start FlaUI driver");
        }

        return null; // Failure
    }

    public void stopLocalServer() {
        if (this.localServer == null) {
            return;
        }

        safelyDestroyProcess(this.localServer.process);
        LogUtil.logInfo("FlaUI driver has stopped at " + this.localServer.getServerUrl());

        this.localServer = null;

        // Clean up old log files
        try {
            deleteOldLogFiles();
        } catch (Exception e) {
            LogUtil.logError(e, "Failed to delete old log files");
        }
    }

    private void safelyDestroyProcess(Process process) {
        if (process == null) {
            return;
        }

        if (process.isAlive()) {
            try {
                process.destroyForcibly();
            } catch (Exception e) {
                LogUtil.logError(e, "Failed to force kill FlaUI driver with PID: " + String.valueOf(process.pid()));
            }
        }
    }

    private void deleteOldLogFiles() {
        File logDir = new File(LOG_FOLDER);
        if (!logDir.exists() || !logDir.isDirectory()) {
            return;
        }
        File[] files = logDir.listFiles((dir, name) -> name.startsWith("flaui") && name.endsWith(".log"));
        if (files == null) {
            return;
        }
        long cutoff = System.currentTimeMillis() - TimeUnit.DAYS.toMillis(LOG_RETENTION_DAYS);
        for (File file : files) {
            if (file.lastModified() < cutoff) {
                try {
                    Files.deleteIfExists(file.toPath());
                    LogUtil.logInfo("Deleted old log file: " + file.getAbsolutePath());
                } catch (IOException e) {
                    LogUtil.logError(e, "Failed to delete old log file: " + file.getAbsolutePath());
                }
            }
        }
    }
}
