package com.kms.katalon.core.webservice.common;

import org.apache.http.impl.auth.DigestSchemeFactory;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.security.GeneralSecurityException;
import java.security.KeyManagementException;
import java.util.Map;
import java.util.Objects;
import org.apache.http.auth.AuthScope;
import javax.net.ssl.HostnameVerifier;
import javax.net.ssl.KeyManager;
import javax.net.ssl.KeyManagerFactory;
import javax.net.ssl.SSLContext;
import javax.net.ssl.SSLSession;
import javax.net.ssl.TrustManager;
import javax.net.ssl.X509TrustManager;
import org.apache.http.auth.AuthSchemeProvider;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.ssl.KeyMaterial;
import org.apache.http.HttpEntity;
import org.apache.http.HttpHost;
import org.apache.http.HttpResponse;
import org.apache.http.ParseException;
import org.apache.http.auth.NTCredentials;
import org.apache.http.auth.UsernamePasswordCredentials;
import org.apache.http.client.CredentialsProvider;
import org.apache.http.client.config.AuthSchemes;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.conn.ConnectTimeoutException;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.impl.auth.DigestScheme;
import org.apache.http.impl.auth.KerberosSchemeFactory;
import org.apache.http.impl.auth.NTLMSchemeFactory;
import org.apache.http.impl.auth.SPNegoSchemeFactory;
import org.apache.http.impl.client.BasicAuthCache;
import org.apache.http.impl.client.BasicCredentialsProvider;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.BasicHttpContext;
import org.apache.http.protocol.HttpContext;
import org.apache.http.util.EntityUtils;
import com.kms.katalon.core.configuration.RunConfiguration;
import com.kms.katalon.core.model.SSLClientCertificateSettings;
import com.kms.katalon.core.network.ProxyInformation;
import com.kms.katalon.core.testobject.RequestObject;
import com.kms.katalon.core.testobject.authorization.DigestAuthorization;
import com.kms.katalon.core.testobject.authorization.AwsSignatureAuthorization;
import com.kms.katalon.core.testobject.authorization.DigestQualityOfProtectionDirective;
import com.kms.katalon.core.testobject.authorization.NTLMAuthorization;
import com.kms.katalon.core.testobject.authorization.RequestAuthorization;
import com.kms.katalon.core.webservice.exception.ConnectionTimeoutException;
import com.kms.katalon.core.webservice.exception.ResponseSizeLimitException;
import com.kms.katalon.core.webservice.exception.SendRequestException;
import com.kms.katalon.core.webservice.exception.SocketTimeoutException;
import com.kms.katalon.core.webservice.exception.WebServiceException;
import com.kms.katalon.core.webservice.helper.WebServiceCommonHelper;
import com.kms.katalon.core.webservice.setting.SSLCertificateOption;
import com.kms.katalon.core.webservice.util.WebServiceCommonUtil;
import com.kms.katalon.network.apache.services.HttpProxyConfigurator;
import com.kms.katalon.network.core.model.config.ProxyConfig;
import com.kms.katalon.util.CryptoUtil;
import com.kms.katalon.core.webservice.aws.Authenticator;

public class HttpUtil {

    private static final String TLS = "TLS";

    private static PoolingHttpClientConnectionManager connectionManager;

    private static final String SOCKET_FACTORY_REGISTRY = "http.socket-factory-registry";
    
    private static Registry<AuthSchemeProvider> digestAuthSchemeRegistry;
    
    private static Authenticator awsAuthenticator;

    // It is an application-level attribute and should be set only once when application started
    private static boolean kerberosSupported;

    static {
        connectionManager = new PoolingHttpClientConnectionManager();
        connectionManager.setValidateAfterInactivity(1);
        connectionManager.setMaxTotal(2000);
        connectionManager.setDefaultMaxPerRoute(500);
        
        digestAuthSchemeRegistry = RegistryBuilder.<AuthSchemeProvider>create()
    		    .register("digest" /* it's defined by Apache, don't change */, new DigestSchemeFactory())
    		    .build();
        awsAuthenticator = new Authenticator();
    }

    private HttpUtil() {}
    public static boolean isKerberosSupported() {
        return kerberosSupported;
    }

    public static void setKerberosSupported(boolean isKerberosSupported) {
        HttpUtil.kerberosSupported = isKerberosSupported;
    }

    public static HttpResponse sendRequest(HttpUriRequest request) throws SendRequestException {
        return sendRequest(request, null);
    }

    public static HttpResponse sendRequest(HttpUriRequest request, ProxyInformation proxyInformation)
            throws SendRequestException {
        return sendRequest(request, true, proxyInformation, SSLCertificateOption.BYPASS, null);
    }

    public static HttpResponse sendRequest(HttpUriRequest request, boolean followRedirects,
            ProxyInformation proxyInformation, SSLCertificateOption certOption,
            SSLClientCertificateSettings clientCertSettings) throws SendRequestException {
        return sendRequest(request, followRedirects, proxyInformation, RequestObject.DEFAULT_TIMEOUT,
                RequestObject.DEFAULT_TIMEOUT, RequestObject.DEFAULT_MAX_RESPONSE_SIZE, certOption, clientCertSettings,
                null);
    }

    public static HttpResponse sendRequest(HttpUriRequest request, boolean followRedirects,
            ProxyInformation proxyInformation, int connectionTimeout, int socketTimeout, long maxResponseSize,
            SSLCertificateOption certificateOption, SSLClientCertificateSettings clientCertSettings,
            RequestAuthorization requestAuthorization) throws SendRequestException {
        CloseableHttpClient httpClient = null;
        try {
            String url = request.getURI().toURL().toString();
            ProxyConfig proxy = proxyInformation == null ? null : proxyInformation.toNewProxyConfigModel();

            HttpContext httpContext = createDefaultHttpContext(certificateOption, clientCertSettings);
            HttpProxyConfigurator.getInstance()
                    .configureProxyAuth(httpContext, proxy, request.getURI(), kerberosSupported);
            HttpClientBuilder clientBuilder = createClientBuilder(url, followRedirects, proxy, connectionTimeout,
                    socketTimeout);
            if (Objects.nonNull(requestAuthorization)
                    && StringUtils.isNotBlank(requestAuthorization.getAuthorizationType())) {
                switch (requestAuthorization.getAuthorizationType()) {
                    case NTLMAuthorization.NTLM:
                        clientBuilder = configureNTLM(clientBuilder, requestAuthorization.getAuthorizationInfo());
                        break;
                    case AwsSignatureAuthorization.AUTHORIZATION_TYPE_NAME:
                        var awsAuth = AwsSignatureAuthorization.adapt(requestAuthorization);
                        // duy.lam HACK: The UI has no step for validation before saving, validate
                        // here to fail fast
                        awsAuth.validate();
                        request = configureAwsAuthentication(request, awsAuth);
                        break;
                    case DigestAuthorization.AUTHORIZATION_TYPE:
                        var da = new DigestAuthorization(requestAuthorization);
                        // duy.lam HACK: The UI has no step for validation before saving, validate
                        // here to fail fast
                        da.validate();
                        configureDigestAuthentication(request, clientBuilder, da, httpContext);
                        break;
                }
            }

            httpClient = clientBuilder.build();
            CloseableHttpResponse response = null;
            try {
                response = httpClient.execute(request, httpContext);
            } catch (ConnectTimeoutException exception) {
                throw new ConnectionTimeoutException(exception);
            } catch (java.net.SocketTimeoutException exception) {
                throw new SocketTimeoutException(exception);
            }

            HttpEntity responseEntity = response.getEntity();
            if (responseEntity != null) {
                long headerLength = WebServiceCommonHelper.calculateHeaderLength(response);
                long bodyLength = responseEntity.getContentLength();
                long totalResponseSize = headerLength + bodyLength;
                boolean isLimitResponseSize = WebServiceCommonUtil.isLimitedRequestResponseSize(maxResponseSize);
                if (isLimitResponseSize && totalResponseSize > maxResponseSize) {
                    request.abort();
                    throw new ResponseSizeLimitException();
                }
            }

            return response;
        } catch (IllegalArgumentException | URISyntaxException | IOException | GeneralSecurityException | WebServiceException e) {
            throw new SendRequestException(e);
        }
        finally {
            if (Objects.nonNull(httpClient)) {
                IOUtils.closeQuietly(httpClient);
            }
        }
    }

    public static String getStringContent(String url) throws SendRequestException {
        try {
            HttpGet get = new HttpGet(url);
            HttpResponse response = sendRequest(get);
            return EntityUtils.toString(response.getEntity());
        } catch (ParseException | IOException e) {
            throw new SendRequestException(e);
        }
    }

    public static byte[] getBytes(String url) throws SendRequestException {
        try {
            HttpGet get = new HttpGet(url);
            HttpResponse response = sendRequest(get);
            return EntityUtils.toByteArray(response.getEntity());
        } catch (SendRequestException | IOException e) {
            throw new SendRequestException(e);
        }
    }

    private static HttpClientBuilder createClientBuilder(String url, boolean followRedirects, ProxyConfig proxy,
            int connectionTimeout, int socketTimeout) throws KeyManagementException, GeneralSecurityException, URISyntaxException, IOException {
        HttpClientBuilder clientBuilder = HttpClients.custom();

        if (followRedirects) {
            clientBuilder.setRedirectStrategy(new DefaultRedirectStrategy());
        } else {
            clientBuilder.disableRedirectHandling();
        }

        clientBuilder.setConnectionManager(connectionManager);
        clientBuilder.setConnectionManagerShared(true);

        if (proxy != null) {
            HttpProxyConfigurator.getInstance().configureProxy(clientBuilder, proxy, SSLContext.getInstance(getHttpsProtocol()), new URI(url));
        }

        configTimeout(clientBuilder, connectionTimeout, socketTimeout);
        clientBuilder.setKeepAliveStrategy(new DefaultConnectionKeepAliveStrategy());
        return clientBuilder;
    }

    private static void configTimeout(HttpClientBuilder httpClientBuilder, int connectionTimeout, int socketTimeout) {
        RequestConfig config = RequestConfig.custom()
                .setConnectTimeout(WebServiceCommonUtil.getValidRequestTimeout(connectionTimeout))
                .setSocketTimeout(WebServiceCommonUtil.getValidRequestTimeout(socketTimeout))
                .build();
        httpClientBuilder.setDefaultRequestConfig(config);
    }

    private static HttpContext createDefaultHttpContext(
            SSLCertificateOption certificateOption,
            SSLClientCertificateSettings clientCertSettings)
            throws KeyManagementException, GeneralSecurityException, IOException {

        HttpContext httpContext = new BasicHttpContext();
        SSLContext sc = SSLContext.getInstance(getHttpsProtocol());
        sc.init(getKeyManagers(clientCertSettings), getTrustManagers(certificateOption), null);
        Registry<ConnectionSocketFactory> reg = RegistryBuilder.<ConnectionSocketFactory> create()
                .register("http", PlainConnectionSocketFactory.INSTANCE)
                .register("https", new SSLConnectionSocketFactory(sc, getHostnameVerifier(certificateOption)))
                .build();
        httpContext.setAttribute(SOCKET_FACTORY_REGISTRY, reg);
        return httpContext;
    }

    private static String getHttpsProtocol() {
        if (RunConfiguration.getProperty(RunConfiguration.HTTPS_PROTOCOL) != null) {
            return (String) RunConfiguration.getProperty(RunConfiguration.HTTPS_PROTOCOL);
        }
        return TLS;
    }

    private static TrustManager[] getTrustManagers(SSLCertificateOption certificateOption) throws IOException {
        if (certificateOption == SSLCertificateOption.BYPASS) {
            return new TrustManager[] { new X509TrustManager() {
                @Override
                public java.security.cert.X509Certificate[] getAcceptedIssuers() {
                    return null;
                }

                @Override
                public void checkClientTrusted(java.security.cert.X509Certificate[] certs, String authType) {
                }

                @Override
                public void checkServerTrusted(java.security.cert.X509Certificate[] certs, String authType) {
                }
            } };
        }
        return new TrustManager[0];
    }

    private static KeyManager[] getKeyManagers(SSLClientCertificateSettings clientCertSettings)
            throws GeneralSecurityException, IOException {
        if (clientCertSettings == null) {
            return new KeyManager[0];
        }

        String keyStoreFilePath = clientCertSettings.getKeyStoreFile();
        if (!StringUtils.isBlank(keyStoreFilePath)) {
            File keyStoreFile = new File(keyStoreFilePath);
            String keyStorePassword = !StringUtils.isBlank(clientCertSettings.getKeyStorePassword())
                    ? clientCertSettings.getKeyStorePassword() : StringUtils.EMPTY;
            if (keyStoreFile.exists()) {
                KeyManagerFactory keyManagerFactory = KeyManagerFactory
                        .getInstance(KeyManagerFactory.getDefaultAlgorithm());
                KeyMaterial km = new KeyMaterial(keyStoreFile, keyStorePassword.toCharArray());
                keyManagerFactory.init(km.getKeyStore(), keyStorePassword.toCharArray());
                return keyManagerFactory.getKeyManagers();
            }
        }
        return new KeyManager[0];
    }

    private static HostnameVerifier getHostnameVerifier(SSLCertificateOption certificateOption) {
        return new HostnameVerifier() {
            @Override
            public boolean verify(String urlHostName, SSLSession session) {
                return certificateOption == SSLCertificateOption.BYPASS;
            }
        };
    }

    private static HttpClientBuilder configureNTLM(HttpClientBuilder clientBuilder,
            Map<String, String> ntlmAuthorization) throws IOException, GeneralSecurityException {
        CredentialsProvider credentialsProvider = new BasicCredentialsProvider();

        CryptoUtil.CrytoInfo cryptoInfo = CryptoUtil.getDefault(ntlmAuthorization.get(NTLMAuthorization.PASSWORD));

        String password = StringUtils.EMPTY;

        password = CryptoUtil.decode(cryptoInfo);

        NTCredentials ntlmCredential = new NTCredentials(ntlmAuthorization.get(NTLMAuthorization.USERNAME), password,
                ntlmAuthorization.get(NTLMAuthorization.WORKSTATION), ntlmAuthorization.get(NTLMAuthorization.DOMAIN));
        credentialsProvider.setCredentials(
                new AuthScope(AuthScope.ANY_HOST, AuthScope.ANY_PORT, AuthScope.ANY_REALM, AuthSchemes.NTLM),
                ntlmCredential);

        Registry<AuthSchemeProvider> authSchemeRegistry = RegistryBuilder.<AuthSchemeProvider> create()
                .register(AuthSchemes.NTLM, new NTLMSchemeFactory())
                .register(AuthSchemes.SPNEGO, new SPNegoSchemeFactory())
                .register(AuthSchemes.KERBEROS, new KerberosSchemeFactory())
                .build();

        clientBuilder.setDefaultAuthSchemeRegistry(authSchemeRegistry)
                .setDefaultCredentialsProvider(credentialsProvider);

        return clientBuilder;
    }
    
    private static HttpUriRequest configureAwsAuthentication(HttpUriRequest request, AwsSignatureAuthorization authorization) throws GeneralSecurityException, IOException, URISyntaxException, WebServiceException {
        var outputRequest = awsAuthenticator.sign(request, authorization);
        return outputRequest;

    }

	private static void configureDigestAuthentication(HttpUriRequest request, HttpClientBuilder clientBuilder,
			DigestAuthorization authorization, HttpContext httpContext) throws GeneralSecurityException, IOException {
		var url = request.getURI().toURL();
		var target = new HttpHost(url.getHost(), url.getPort(), url.getProtocol());
		var cresProvider = new BasicCredentialsProvider();
		cresProvider.setCredentials(new AuthScope(target),
				new UsernamePasswordCredentials(authorization.getUsername(), authorization.getPassword()));

		if (authorization.useChallengeResponseMechanism()) {
			clientBuilder.setDefaultAuthSchemeRegistry(digestAuthSchemeRegistry);
			clientBuilder.setDefaultCredentialsProvider(cresProvider);
		} else {
			var digestAuth = new DigestScheme();
			digestAuth.overrideParamter("realm", authorization.getRealm());
			digestAuth.overrideParamter("nonce", authorization.getNonce());
			digestAuth.overrideParamter("algorithm", authorization.getAlgorithm().getName());

			if (authorization.getQop() != DigestQualityOfProtectionDirective.UNSPECIFIED) {
				digestAuth.overrideParamter("qop", authorization.getQop().getName());
			}

			if (Objects.nonNull(authorization.getNonceCount())) {
				digestAuth.overrideParamter("nc", authorization.getNonceCount().toString());
			}

			if (StringUtils.isNotBlank(authorization.getClientNounce())) {
				digestAuth.overrideParamter("cnonce", authorization.getClientNounce());
			}

			if (StringUtils.isNotBlank(authorization.getOpaque())) {
				digestAuth.overrideParamter("opaque", authorization.getOpaque());
			}

			var authCache = new BasicAuthCache();
			authCache.put(target, digestAuth);
			var overridedContext = HttpClientContext.adapt(httpContext);
			overridedContext.setAuthCache(authCache);
			overridedContext.setCredentialsProvider(cresProvider);
		}
	}
}
