package com.kms.katalon.core.auth;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.Objects;

import org.apache.commons.io.FileUtils;
import org.eclipse.core.runtime.Platform;

import com.kms.katalon.logging.LogUtil;

// https://www.ibm.com/docs/en/semeru-runtime-ce-z/11?topic=differences-jaas-options-kerberos-login
public class JaasSetupService {

    private static String PROP_KEY_USE_SUBJECT_CREDS_ONLY = "javax.security.auth.useSubjectCredsOnly";

    private static String PROP_KEY_LOGIN_CONFIG_PATH = "java.security.auth.login.config";

    private static String PROP_KEY_KATALON_KERBEROS_ENABLED = "katalon.kerberosEnabled";

    private static String ENV_KEY_KATALON_KERBEROS_ENABLED = "KATALON_KERBEROS_ENABLED";

    private boolean kerberosSupported = false;

    public static final JaasSetupService singleton;
    static {
        singleton = new JaasSetupService();
    }

    public static JaasSetupService getInstance() {
        return singleton;
    }

    private JaasSetupService() {
        this.kerberosSupported = false;
    }

    public boolean isKerberosSupported() {
        return kerberosSupported;
    }

    public void enableKerberosIfSupported() {
        String os = System.getProperty("os.name").toLowerCase();
        
        if (!os.contains("win")) {
            return;
        }

        if (!Boolean.valueOf(System.getProperty(PROP_KEY_KATALON_KERBEROS_ENABLED)).booleanValue()
                && !Boolean.valueOf(System.getenv(ENV_KEY_KATALON_KERBEROS_ENABLED)).booleanValue()) {
            return;
        }

        try {
            enableKerberos();
            this.kerberosSupported = true;
            LogUtil.logInfo("Already enabled Kerberos support");
        } catch (Exception e) {
            LogUtil.logError(e, "Failed to enable Kerberos support.");
        }
    }

    private void enableKerberos() throws IOException {
        String loginConfPath = Platform.getInstallLocation().getURL().getPath() + "login.conf";
        createLoginConfFileIfNonExistent(loginConfPath);

        if (Objects.isNull(System.getProperty(PROP_KEY_LOGIN_CONFIG_PATH))) {
            System.setProperty(PROP_KEY_LOGIN_CONFIG_PATH, loginConfPath);
        }

        if (Objects.isNull(System.getProperty(PROP_KEY_USE_SUBJECT_CREDS_ONLY))) {
            System.setProperty(PROP_KEY_USE_SUBJECT_CREDS_ONLY, "false");
        }
    }

    private void createLoginConfFileIfNonExistent(String path) throws IOException {
        File file = new File(path);
        
        // We may allow user to edit the config file. Therefore, if the file existed (and possibly edited by user), we keep it unchanged
        if (file.exists()) {
            return;
        }

        LogUtil.logInfo(String.format("Jass config file is non-existent at %s. Will create one.", path));
        String content = """
                com.sun.security.jgss.initiate  {
                    com.sun.security.auth.module.Krb5LoginModule required
                    client=TRUE useTicketCache=true
                    doNotPrompt=true storeKey=false
                    isInitiator=true
                    useKeyTab=false
                    renewTGT=false debug=true;
                };
                """;

        FileUtils.writeStringToFile(file, content, StandardCharsets.US_ASCII);
    }
}
